/******************************************************************************
 ** $Id: Auswahl.java 2533 2021-01-03 13:53:57Z wmh $
 ** Diese Datei ist Bestandteil der Java-Quelltexte des Wrfelspiels JaFuffy.
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************
 ** Die aktuellste Version von JaFuffy findet sich im Internet unter
 ** <http://jafuffy.3kelvin.de>.
 **
 ** Kommentare, Fehler oder Erweiterungswnsche bitte per E-Mail senden an
 ** <jafuffy@3kelvin.de>.
 ******************************************************************************/
package jafuffy.bedienung.auswahl;

import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.text.NumberFormat;
import java.util.Locale;

import javax.swing.BorderFactory;
import javax.swing.DefaultCellEditor;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JTable;

import jafuffy.Eigenschaften;
import jafuffy.bedienung.Geraeusch;
import jafuffy.bedienung.Tabelle;
import jafuffy.logik.Kategorie;
import jafuffy.logik.Turnier;
import jafuffy.logik.analyse.Analyse;
import jafuffy.logik.auswertung.Auswertung;

/**
 * Auswahl der Zelle, in die das Wrfelresultat gesetzt wird. Setzt auf Wunsch in die Tabelle (nach einmaligen bzw.
 * zweimaligem Klick).
 */
@SuppressWarnings("serial")
public abstract class Auswahl extends DefaultCellEditor implements ActionListener {

    /** Innenabstand der Zelle. */
    private static final int ABSTAND = 4;
    /** Zur Formatierung der Wichtung. */
    private static final NumberFormat NF;

    /** Akustisches Signal fr besonderes Wrfelergebnis. */
    protected static final Geraeusch HURRA = new Geraeusch("Hurra.wav");

    static {
        NF = NumberFormat.getInstance(Locale.GERMAN);
        NF.setMinimumFractionDigits(2);
        NF.setMaximumFractionDigits(2);
    }

    /** Setzen. */
    private final JButton setzen;

    /** Wie oft angeklickt? */
    protected int clicks;
    /** Aktuell angewhlter Eintrag des aktiven Spielers. */
    protected Kategorie aktuell;
    /** Tabelle zum Neuzeichnen. */
    protected Tabelle tabelle;
    /** Turnier. */
    protected final Turnier turnier;

    /**
     * Konstruktor.
     *
     * @param turnier
     *            Das Turnier, fr welches die Auswahl getroffen werden soll.
     */
    public Auswahl(Turnier turnier) {
        super(new JCheckBox());
        this.turnier = turnier;
        setClickCountToStart(1);
        setzen = new JButton("setzen");
        setzen.setBorder(BorderFactory.createEmptyBorder(0, ABSTAND, 0, ABSTAND));
        setzen.setFocusPainted(false);
        setzen.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent event) {
                clicks++;
                if (clicks == 2 || Eigenschaften.GLOBAL.aktiv("Einfachklick")) {
                    fireEditingStopped();
                    reagiereAufSetzen();
                }
            }
        });
        editorComponent = setzen;
    }

    /**
     * Aktiven Eintrag auswhlen.
     *
     * @param kategorie
     *            Tabellenzeile, die als aktiver Eintrag ausgewhlt wird.
     */
    public void aktiviere(Kategorie kategorie) {
        aktuell = kategorie;
        clicks = 0;
    }

    @Override
    public void actionPerformed(ActionEvent event) {
        aktiviere(turnier.auswertung().analyse().bester());
        doClick();
    }

    /** Button klicken (beachtet die Einstellung zum Setzen per Einfachklick). */
    public void doClick() {
        setzen.doClick();
        if (!Eigenschaften.GLOBAL.aktiv("Einfachklick")) {
            setzen.doClick();
        }
    }

    /** @return Zelleneditor (JButton) */
    @Override
    public Component getTableCellEditorComponent(JTable table, Object value, boolean isSelected, int row, int col) {
        aktiviere(Kategorie.kategorie(row));
        tabelle = (Tabelle) table;
        aktualisiere();
        return editorComponent;
    }

    /** Durchzufhrende Aktionen nach dem Wrfeln wie etwa Tipp aktualisieren. */
    public abstract void reagiereAufWurf();

    /** Tipp aktualisieren. */
    protected void aktualisiere() {
        String text;
        if (aktuell != null) {
            fireEditingCanceled();
            Auswertung auswertung = turnier.auswertung();
            Analyse analyse = auswertung.analyse();
            text = "<p>" + "Jetzt klicken liefert " + analyse.wert(aktuell) + " Punkte." + "</p>";
            if (auswertung.sindTippsAngefordert()) {
                text += "<p>Bewertung: " + NF.format(analyse.wichtung(aktuell)) + "</p>";
            }
            text += "<p><small>" + "berspringen der Besttigung im Men: Einstellungen&rarr;Setzen durch Einfachklick"
                    + "</small></p>";
            editorComponent.setToolTipText("<html>" + text + "<html>");
        }
    }

    /** Ausgefhrt falls vom Spieler per Klick auf den Setzen-Button der Wurfergebnis tatschlich gesetzt wird. */
    protected abstract void reagiereAufSetzen();

}
