/******************************************************************************
 ** $Id: Hilfe.java 2429 2020-06-07 21:03:54Z wmh $
 ** Diese Datei ist Bestandteil der Java-Quelltexte des Wrfelspiels JaFuffy.
 ** Lauffhig ab Java 7.
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************
 ** Die aktuellste Version von JaFuffy findet sich im Internet unter
 ** <http://jafuffy.3kelvin.de>.
 **
 ** Kommentare, Fehler oder Erweiterungswnsche bitte per E-Mail senden an
 ** <jafuffy@3kelvin.de>.
 ******************************************************************************/
package jafuffy.bedienung;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.GridLayout;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.net.URL;
import java.util.Stack;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JEditorPane;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;
import javax.swing.event.HyperlinkEvent;
import javax.swing.event.HyperlinkListener;

import jafuffy.Ressource;

/** Hilfe (HTML-Minibrowser). */
@SuppressWarnings("serial")
class Hilfe extends JDialog implements PropertyChangeListener, HyperlinkListener {

    /** Abstand vom Dialograhmen zum Inhalt. */
    private static final int DIALOGRAHMENABSTAND = 8;
    /** Abstand vom Feldrahmen zum Inhalt. */
    private static final int FELDRAHMENABSTAND = 4;
    /** Abstand Feldrahmen zu Buttons. */
    private static final int BUTTONABSTAND = 2 * FELDRAHMENABSTAND;
    /** Anfngliche Breite des Hilfefensters. */
    private static final int BREITE = 640;
    /** Anfngliche Hhe des Hilfefensters. */
    private static final int HOEHE = 448;
    /** URL des Hilfeindex. */
    private static final URL STARTSEITE = Ressource.ANLEITUNG.url("index.html");

    /** Darstellungsobjekt fr HTML-Hilfetext. */
    private final JEditorPane browser = new JEditorPane();
    /** Seitenverlauf. */
    private final Stack<URL> history = new Stack<>();
    /** Inhalt-Button (Hilfeindex), zurck zum Anfang. */
    private final JButton inhalt = new JButton("Zum Hilfebeginn");
    /** Startseite, initialisiert mit aktueller Seite. */
    private URL seite = STARTSEITE;
    /** Zurck-Button, einen Schritt zurck. */
    private final JButton zurueck = new JButton("Schritt zurck");

    /**
     * Konstruktor.
     *
     * @throws Exception
     *             Problem bei Erstellung der Hilfe.
     */
    Hilfe() throws Exception {

        UIManager.addPropertyChangeListener(this);

        browser.setPage(seite);
        browser.setEditable(false);
        browser.addHyperlinkListener(this);

        inhalt.addActionListener(event -> {
            try {
                history.push(seite);
                seite = STARTSEITE;
                browser.setPage(seite);
                inhalt.setEnabled(false);
            } catch (Exception exception) {
                exception.printStackTrace();
            }
        });

        zurueck.addActionListener(event -> {
            try {
                seite = history.pop();
                browser.setPage(seite);
            } catch (Exception exception) {
                exception.printStackTrace();
                Signal.WARNUNG.play();
                JOptionPane.showMessageDialog(Hilfe.this,
                        "Hilfe-Datei nicht gefunden!\n" + "Java-Exception:\n" + exception, "JaFuffy (Warnung)",
                        JOptionPane.WARNING_MESSAGE);
            }
            zurueck.setEnabled(!history.empty());
            inhalt.setEnabled(seite != STARTSEITE);
        });

        setTitle("JaFuffy " + "(" + "Hilfe "
                + (STARTSEITE.getProtocol().equals("http") ? "ber Internet" : "aus Installation") + ")");
        setIconImages(Oberflaeche.LOGOS);
        setContentPane(hilfefeld());
        setModalExclusionType(ModalExclusionType.APPLICATION_EXCLUDE);
        pack();

    }

    @Override
    public void hyperlinkUpdate(HyperlinkEvent event) {
        if (event.getEventType() == HyperlinkEvent.EventType.ACTIVATED) {
            try {
                history.push(seite);
                seite = event.getURL();
                browser.setPage(seite);
                zurueck.setEnabled(true);
                inhalt.setEnabled(seite != STARTSEITE);
            } catch (Exception exception) {
                exception.printStackTrace();
                Signal.WARNUNG.play();
                JOptionPane.showMessageDialog(Hilfe.this, "Kann die folgende Seite nicht ffnen:\n" + seite.toString()
                        + "\n" + "Java-Exception:\n" + exception, "JaFuffy (Warnung)", JOptionPane.WARNING_MESSAGE);
            }
        }
    }

    /**
     * Realisiert Vernderungen des Look&Feel.
     *
     * @param event
     *            nderungsereignis.
     */
    @Override
    public void propertyChange(PropertyChangeEvent event) {
        if (event.getPropertyName().equals("lookAndFeel")) {
            SwingUtilities.updateComponentTreeUI(this);
        }
    }

    /** @return Hilfefeld */
    private JPanel hilfefeld() {
        JPanel hilfefeld = new JPanel(new BorderLayout(FELDRAHMENABSTAND, FELDRAHMENABSTAND));
        hilfefeld.setBorder(BorderFactory.createEmptyBorder(DIALOGRAHMENABSTAND, DIALOGRAHMENABSTAND,
                DIALOGRAHMENABSTAND, DIALOGRAHMENABSTAND));
        JScrollPane scrollPane = new JScrollPane(browser);
        scrollPane.setPreferredSize(new Dimension(BREITE, HOEHE));
        hilfefeld.add(scrollPane, BorderLayout.CENTER);
        hilfefeld.add(leiste(), BorderLayout.SOUTH);
        return hilfefeld;
    }

    /**
     * Erzeugt die Kontrollleiste, wobei der Standard-Button gesetzt wird.
     *
     * @return Leiste mit Kontrollen.
     */
    private JPanel leiste() {
        inhalt.setEnabled(false);
        zurueck.setEnabled(false);
        JPanel leiste = new JPanel(new GridLayout(1, 2, BUTTONABSTAND, 0));
        leiste.add(inhalt);
        leiste.add(zurueck);
        JButton beenden = new JButton("Hilfe beenden");
        beenden.requestFocus();
        beenden.addActionListener(e -> setVisible(false));
        getRootPane().setDefaultButton(beenden);
        leiste.add(beenden);
        return leiste;
    }

}
