/******************************************************************************
 ** $Id: Botstaerke.java 2520 2020-12-31 13:22:45Z wmh $
 ** Diese Datei ist Bestandteil der Java-Quelltexte des Wrfelspiels JaFuffy.
 ** Lauffhig ab Java 7.
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************
 ** Die aktuellste Version von JaFuffy findet sich im Internet unter
 ** <http://jafuffy.3kelvin.de>.
 **
 ** Kommentare, Fehler oder Erweiterungswnsche bitte per E-Mail senden an
 ** <jafuffy@3kelvin.de>.
 ******************************************************************************/
package jafuffy.bedienung;

import java.awt.BorderLayout;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Hashtable;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JSlider;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;
import javax.swing.border.EmptyBorder;
import javax.swing.border.TitledBorder;

import jafuffy.Eigenschaften;
import jafuffy.logik.Bot;
import jafuffy.logik.Bot.Staerke;

/** Dialog zur Einstellungen der Spielstrken der einzelnen Bots. */
@SuppressWarnings("serial")
public class Botstaerke extends JDialog implements PropertyChangeListener {

    /** Abstand vom Dialograhmen zum Inhalt. */
    private static final int DIALOGRAHMENABSTAND = 8;
    /** Abstand vom Feldrahmen zum Inhalt. */
    private static final int FELDRAHMENABSTAND = DIALOGRAHMENABSTAND / 2;
    /** Abstand zwischen Buttons. */
    private static final int BUTTONABSTAND = DIALOGRAHMENABSTAND;

    /** Die Schieberegler zur Einstellung der Spielstrke der einzelnen Bots. */
    private final JSlider[] staerkeneinstellungen = new JSlider[Bot.BOTNAMEN.length];

    /**
     * Baut Dialog zusammen.
     *
     * @param eigenschaften
     *            Die globalen, persistenten Einstellungen von JaFuffy.
     */
    Botstaerke(Eigenschaften eigenschaften) {
        UIManager.addPropertyChangeListener(this); // Look&Feel-Wechsel
        setContentPane(baueDarstellungsfeld(eigenschaften));
        setModal(true);
        setTitle("JaFuffy (Spielstrke der Bots)");
        setIconImages(Oberflaeche.LOGOS);
        pack();
    }

    @Override
    public void propertyChange(PropertyChangeEvent event) {
        if (event.getPropertyName().equals("lookAndFeel")) {
            SwingUtilities.updateComponentTreeUI(this);
            pack();
        }
    }

    /**
     * Baut das Feld zur Einstellung der einzelnen Spielstrken mit der Aktionsleiste zusammen.
     *
     * @param eigenschaften
     *            Die globalen, persistenten Einstellungen von JaFuffy.
     * @return Das Darstellungsfeld mit Einstellungen und Aktionsleiste.
     */
    private JPanel baueDarstellungsfeld(Eigenschaften eigenschaften) {
        JPanel feld = new JPanel(new BorderLayout(0, 2 * FELDRAHMENABSTAND));
        feld.setBorder(
                new EmptyBorder(DIALOGRAHMENABSTAND, DIALOGRAHMENABSTAND, DIALOGRAHMENABSTAND, DIALOGRAHMENABSTAND));
        feld.add(baueEinstellungsfeld(eigenschaften), BorderLayout.CENTER);
        feld.add(baueLeistenfeld(eigenschaften), BorderLayout.PAGE_END);
        return feld;
    }

    /**
     * Baut das Feld zur Einstellung der einzelnen Spielstrken zusammen.
     *
     * @param eigenschaften
     *            Die globalen, persistenten Einstellungen von JaFuffy.
     * @return Das Einstellungsfeld.
     */
    private JPanel baueEinstellungsfeld(Eigenschaften eigenschaften) {
        Hashtable<Integer, JLabel> bezeichnungen = new Hashtable<>();
        Bot.Staerke[] staerken = Bot.Staerke.values();
        for (Bot.Staerke staerke : staerken) {
            bezeichnungen.put(staerke.ordinal(), new JLabel(staerke.toString()));
        }
        JPanel einstellungen = new JPanel(new GridLayout(Bot.BOTNAMEN.length, 1, 0, 2 * FELDRAHMENABSTAND));
        for (int index = 0; index < Bot.BOTNAMEN.length; index++) {
            staerkeneinstellungen[index] = new JSlider(0, staerken.length - 1);
            staerkeneinstellungen[index].setBorder(
                    BorderFactory.createCompoundBorder(new TitledBorder("Spielstrke von " + Bot.BOTNAMEN[index]),
                            new EmptyBorder(FELDRAHMENABSTAND / 2, FELDRAHMENABSTAND / 2, 0, FELDRAHMENABSTAND / 2)));
            staerkeneinstellungen[index].setMajorTickSpacing(1);
            staerkeneinstellungen[index].setPaintLabels(true);
            staerkeneinstellungen[index].setLabelTable(bezeichnungen);
            einstellungen.add(staerkeneinstellungen[index]);
            String botstaerke = eigenschaften.getProperty("Botstaerke" + index);
            staerkeneinstellungen[index].setValue(Bot.Staerke.valueOf(botstaerke).ordinal());
        }
        return einstellungen;
    }

    /**
     * Baut die Aktionsleiste zusammen und verknpft sie mit den Aktionen zur Besttigung oder zum Abbruch.
     *
     * @param eigenschaften
     *            Die globalen, persistenten Einstellungen von JaFuffy.
     * @return Das Aktionsleistenfeld.
     */
    private JPanel baueLeistenfeld(final Eigenschaften eigenschaften) {
        JButton ok = new JButton("OK");
        ok.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent event) {
                for (int index = 0; index < Bot.BOTNAMEN.length; index++) {
                    eigenschaften.setProperty("Botstaerke" + index,
                            Staerke.values()[staerkeneinstellungen[index].getValue()].name());
                }
                setVisible(false);
            }

        });
        getRootPane().setDefaultButton(ok);
        JButton abbruch = new JButton("Abbruch");
        abbruch.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent event) {
                for (int index = 0; index < Bot.BOTNAMEN.length; index++) {
                    String botstaerke = eigenschaften.getProperty("Botstaerke" + index);
                    staerkeneinstellungen[index].setValue(Staerke.valueOf(botstaerke).ordinal());
                }
                setVisible(false);
            }

        });
        JPanel leiste = new JPanel(new GridLayout(1, 2, BUTTONABSTAND, 0));
        leiste.add(ok);
        leiste.add(abbruch);
        return leiste;
    }

}
