/******************************************************************************
 ** $Id: Vermittlung.java 2620 2021-02-21 21:11:53Z wmh $
 ** Diese Datei ist Bestandteil der Java-Quelltexte des Wrfelspiels JaFuffy.
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************
 ** Die aktuellste Version von JaFuffy findet sich im Internet unter
 ** <http://jafuffy.3kelvin.de>.
 **
 ** Kommentare, Fehler oder Erweiterungswnsche bitte per E-Mail senden an
 ** <jafuffy@3kelvin.de>.
 ******************************************************************************/
package jafuffy.netzwerk;

import java.io.EOFException;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.lang.reflect.InvocationTargetException;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.SocketException;
import java.util.ArrayList;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;

import javax.swing.SwingUtilities;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import jafuffy.logik.Kategorie;
import jafuffy.logik.Plan;
import jafuffy.logik.Spieler;
import jafuffy.logik.Turnier;
import jafuffy.logik.ereignis.Ablauf;
import jafuffy.logik.ereignis.Umschlag;
import jafuffy.netzwerk.ereignis.Ereignis;
import jafuffy.netzwerk.ereignis.Lebenszeichen;
import jafuffy.netzwerk.ereignis.Wuerfelabwahl;
import jafuffy.netzwerk.ereignis.Wuerfelanwahl;
import jafuffy.netzwerk.ereignis.Wurfdarlegung;
import jafuffy.netzwerk.ereignis.Wurfergebnis;
import jafuffy.netzwerk.ereignis.Wurfsetzung;

/**
 * Stellt Verbindungen zu allen Austragungsorten her, wartet auf eingehende Verbindungsanfragen von allen
 * Austragungsorten und stellt daraufhin Server fr die anfragenden Orte bereit.
 */
public class Vermittlung extends Thread implements ChangeListener {

    /** Schnittstelle zur Aufnahme des Turnierbetriebs nach erfolgter Organisation bers Netzwerk. */
    public interface Eroeffnung {
        /**
         * Nimmt den Turnierbetrieb auf zu einem Plan.
         *
         * @param plan
         *            Der Plan, dem der Turnierbetrieb folgt.
         */
        void starte(Plan plan);
    }

    /**
     * Schnittstelle zum Nachfhrung des Turnierablaufs ber das Netzwerk fr einzelne Austragungsorte, also zur
     * Reaktion auf die Wurfdarlegung eines Ergebnisses aus einem Wurf fr den ganzen Wrfelsatz sowie zum Setzen des
     * vorliegenden Wurfes.
     */
    public interface Nachfuehrung {
        /** Reagiert auf einen Wrfelvorgang des aktiven Spielers. */
        void reagiere();

        /**
         * Setzt den gewnschten Eintrag, so dass der nchste Spieler an die Reihe kommt.
         *
         * @param kategorie
         *            Der zu setzende Eintrag, fr den der vorliegende Wurfergebnis ausgewertet wird.
         */
        void setze(Kategorie kategorie);
    }

    /** Schnittstelle zum Abbruch aufgrund einer Strung. */
    public interface Stoerung {
        /**
         * Bricht das Turniergeschehen aufgrund einer Ausnahme ab im Betrieb ber ein Netzwerk.
         *
         * @param ausnahme
         *            Die Ausnahme, aufgrund derer das Turnier von der Aufsicht abgebrochen wird.
         */
        void abbreche(Exception ausnahme);
    }

    /** Schnittstelle zur bertragung des Ergebnisses fr einen Wrfel. */
    public interface Wurf {
        /**
         * bertrage die Augenzahl auf den Wrfel mit Ausfhrung der Nachfolgeaktionen.
         *
         * @param augen
         *            Die vorgegebene Augenzahl
         */
        void uebertrage(int augen);

        /**
         * Whlt fr den wrfel, ob er in den Becher gelegt wird.
         *
         * @param selektiert
         *            Bestimmt, ob der Wrfel gelegt wird.
         */
        void waehle(boolean selektiert);
    }

    /**
     * Bearbeiter der Ereignisse, welche ber ein Netzwerk hereinkommen bei Erffnung eines Turniers ber eben dieses.
     */
    private class Bearbeiter {
        /** Schnittstelle zur Aufnahme des Turnierbetriebs nach erfolgter Organisation bers Netzwerk. */
        private final Eroeffnung eroeffnung;
        /** Schnittstellen fr den ganzen Wrfelsatz zur bertragung des Ergebnisses fr einen Wrfel. */
        private Wurf[] wuerfe;
        /** Schnittstelle zur Reaktion auf die Darlegung eines Ergebnisses aus einem Wurf fr den ganzen Wrfelsatz. */
        private Nachfuehrung nachfuehrung;

        /**
         * Konstruiert einen Bearbeiter, welcher auf eine Schnittstelle zur Aufnahme des Betriebs fut.
         *
         * @param eroeffnung
         */
        Bearbeiter(Eroeffnung eroeffnung) {
            this.eroeffnung = eroeffnung;
        }

        /**
         * Verarbeitet ein eingegangenes Ereignis.
         *
         * @param ereignis
         *            Das Ereignis, welches ber ein Netzwerk eingegangen ist.
         */
        void betrachte(Ereignis ereignis) {
            Class<? extends Ereignis> klasse = ereignis.getClass();
            if (klasse == Plan.class) {
                eroeffnung.starte((Plan) ereignis);
            } else if (klasse == Wuerfelabwahl.class) {
                Wuerfelabwahl abwahl = (Wuerfelabwahl) ereignis;
                if (wuerfe != null) {
                    wuerfe[abwahl.index].waehle(false);
                }
            } else if (klasse == Wuerfelanwahl.class) {
                Wuerfelanwahl anwahl = (Wuerfelanwahl) ereignis;
                if (wuerfe != null) {
                    wuerfe[anwahl.index].waehle(true);
                }
            } else if (klasse == Wurfergebnis.class) {
                Wurfergebnis ergebnis = (Wurfergebnis) ereignis;
                if (wuerfe != null) {
                    wuerfe[ergebnis.index].uebertrage(ergebnis.augen);
                }
            } else if (klasse == Wurfdarlegung.class) {
                nachfuehrung.reagiere();
            } else if (klasse == Wurfsetzung.class) {
                Wurfsetzung setzung = (Wurfsetzung) ereignis;
                nachfuehrung.setze(setzung.kategorie);
            } else if (klasse == Lebenszeichen.class) {
                // Bislang keine weiteren Aktionen ntig
            }
        }

        /**
         * Verknpft diese Vermittlung mit den Elementen, welche ein Turnier umsetzen.
         *
         * @param wuerfe
         *            Der gesamte Wrfelsatz, welcher zum Turnier gehrt.
         * @param nachfuehrung
         *            Fhrt die Aktionen nach, welche sich beim Eroeffnung ber ein Netzwerk auf diesen Austragungsort
         *            auswirken.
         */
        void verknuepfe(Wurf[] wuerfe, Nachfuehrung nachfuehrung) {
            this.wuerfe = wuerfe;
            this.nachfuehrung = nachfuehrung;
        }

    }

    /** Bedient fr fr eine gegebene Austragung eines Turniers ber ein Netzwerk die Anfragen auf einem Kanal. */
    private class Bediener extends Thread {
        /** Der Kanal fr diesen Bediener von einem anderen Austragungsort. */
        private final Kanal kanal;

        /**
         * Konstruiert den Bediener, welche ber einen Socket eingehende Anfragen bedient
         *
         * @param socket
         *            Der Socket, ber den die Anfragen hereinkommen.
         * @throws Exception
         *             Verbindungsanfrage kann nicht angenommen werden.
         */
        Bediener(Socket socket) throws Exception {
            System.out.println("! " + socket.getRemoteSocketAddress() + " > " + socket.getLocalSocketAddress());
            kanal = new Kanal(new ObjectOutputStream(socket.getOutputStream()),
                    new ObjectInputStream(socket.getInputStream()));
            kanal.annehme();
            quellkanaele.add(kanal);
            verbindungszaehler.countDown();
        }

        @Override
        public void run() {
            do {
                try {
                    prozessiere();
                } catch (EOFException | SocketException ausnahme) {
                    // Abbruch durch Benutzer oder nach Ablauf der maximalen Wartezeit, Ausstieg aus Schleife
                    break;
                } catch (Exception ausnahme) {
                    schliesse();
                    stoerung.abbreche(ausnahme);
                    break;
                }
            } while (true);
        }

        /**
         * Prozessiert die eingehenden Ereignisse, welche von Austragungsorten stammen. Die Abarbeitung erfolgt im
         * AWT-Event Thread, so als ob eine Benutzerinteraktion erfolgt wre.
         *
         * @throws Exception
         *             Probleme bei bermittlungen ber das Netzwerk.
         */
        private void prozessiere() throws Exception {
            Ereignis ereignis = kanal.empfange();
            Runnable auftrag = () -> {
                bearbeiter.betrachte(ereignis);
                try {
                    kanal.bestaetige();
                } catch (IOException ausnahme) {
                    stoerung.abbreche(ausnahme);
                }
            };
            if (aussetzungszaehler == null) {
                SwingUtilities.invokeAndWait(auftrag);
            } else {
                ausfuehrer.execute(() -> {
                    try {
                        aussetzungszaehler.await();
                        SwingUtilities.invokeAndWait(auftrag);
                    } catch (InterruptedException | InvocationTargetException ausnahme) {
                        stoerung.abbreche(ausnahme);
                    }
                });
            }

        }
    }

    /** Betreut einen Kanal zwischen diesem und einem anderen Austragungsort. */
    private static class Kanal {

        /** Wird zur Kennzeichnung einer Verbindungsanfrage bers Netzwerk verschickt. */
        private static final String ANFRAGE = "JaFuffy?";
        /** Fehlerausgabe in Ausnahmebehandlung in Reaktion of unerwartete Antworten */
        private static final String ANFRAGEFEHLER = "Fehler in Verbindungsanfrage";
        /** Wird zur Kennzeichnung der Annahme einer Verbindungsanfrage bers Netzwerk verschickt. */
        private static final String ANNAHME = "JaFuffy!";
        /** Fehlerausgabe in Ausnahmebehandlung in Reaktion of unerwartete Antworten */
        private static final String ANNAHMEFEHLER = "Fehler in Verbindungsannahme";
        /** Wird als Besttigung der Ausfhrung eines Auftrags verschickt. */
        private static final String BESTAETIGUNG = "OK";
        /** Fehlerausgabe falls keine Besttigung erhalten wurde. */
        private static final String BESTAETIGUNGSFEHLER = "Keine Besttigung (OK) erhalten.";
        /** Der Strom, auf dem Nachrichten verschickt werden. */
        private final ObjectOutputStream versandstrom;
        /** Der Strom, auf dem Nachrichten empfangen werden. */
        private final ObjectInputStream empfangsstrom;

        /**
         * Konstruiert einen Kanal entsprechend einer Verbindung zwischen diesem und einem anderen Austragungsort.
         *
         * @param versandstrom
         *            Der Strom, auf dem Nachrichten verschickt werden.
         * @param empfangsstrom
         *            Der Strom, auf dem Nachrichten empfangen werden.
         */
        Kanal(ObjectOutputStream versandstrom, ObjectInputStream empfangsstrom) {
            this.versandstrom = versandstrom;
            this.empfangsstrom = empfangsstrom;
        }

        /**
         * Frage um eine Verbindung an und warte auf die Annahme.
         *
         * @throws Exception
         *             Verbindung konnte nicht hergestellt werden.
         */
        void anfrage() throws Exception {
            sende(ANFRAGE);
            if (!empfangsstrom.readUTF().equals(ANNAHME)) {
                throw new Exception(ANNAHMEFEHLER);
            }
        }

        /**
         * Nimmt eine Verbindungsanfrage an.
         *
         * @throws Exception
         *             Fehlerhafte Anfrage erhalten oder Antwort nicht mglich.
         */
        void annehme() throws Exception {
            if (!empfangsstrom.readUTF().equals(ANFRAGE)) {
                throw new Exception(ANFRAGEFEHLER);
            }
            sende(ANNAHME);
        }

        /**
         * Besttigt die Bearbeitung eines Ereignisses.
         *
         * @throws IOException
         */
        void bestaetige() throws IOException {
            sende(BESTAETIGUNG);
        }

        /**
         * Empfngt blockierend ein Ereignis.
         *
         * @return Das empfangene Ereignis fr die weitere Bearbeitung
         * @throws IOException
         *             Probleme beim Empfang.
         * @throws ClassNotFoundException
         *             Probleme bei Konvertierung in Ereignis.
         */
        Ereignis empfange() throws ClassNotFoundException, IOException {
            return (Ereignis) empfangsstrom.readObject();
        }

        /**
         * Schliet Versandstrom und Empfangsstrom.
         *
         * @throws IOException
         *             Problem beim Schlieen.
         */
        void schliesse() throws IOException {
            versandstrom.close();
            empfangsstrom.close();
        }

        /**
         * Sendet ein Objekt.
         *
         * @param Das
         *            zu versendende Objekt.
         * @throws IOException
         *             Objekt konnte nicht versendet werden.
         */
        void sende(Serializable objekt) throws IOException {
            versandstrom.writeObject(objekt);
            versandstrom.flush();
        }

        /**
         * Sendet einen Text.
         *
         * @param text
         *            Der zu versendende Text.
         * @throws IOException
         *             Objekt konnte nicht versendet werden.
         */
        void sende(String text) throws IOException {
            versandstrom.writeUTF(text);
            versandstrom.flush();
        }

        /**
         * Wartet auf die Quittierung zu einem versendeten Ereignis.
         *
         * @throws Exception
         *             Quittierung gar nicht oder fehlerhaft erhalten.
         */
        void warte() throws Exception {
            if (!empfangsstrom.readUTF().equals(BESTAETIGUNG)) {
                throw new Exception(BESTAETIGUNGSFEHLER);
            }
        }

    }

    /** berwacht die Verbindung zu anderen Austragungsorten. */
    private class Ueberwachung extends TimerTask {

        /** Gibt an, ob die berwachung derzeit aktiv ist. */
        private volatile boolean aktiv;

        @Override
        public void run() {
            try {
                verteile(new Lebenszeichen());
            } catch (Exception ausnahme) {
                if (aktiv) {
                    schliesse();
                    stoerung.abbreche(ausnahme);
                }
            }
        }

        /**
         * Bestimmt, ob die berwachung aktiv ist.
         *
         * @param aktiv
         *            Status der berwachung.
         */
        void aktiviere(boolean aktiv) { // TODO
            this.aktiv = aktiv;
        }

    }

    /** Zeit in Millisekunden, mit der auf das Zustandekommen aller wechselseitigen Verbindungen gewartet wird. */
    private static final int VERBINDUNGSTIMEOUT = 5000;

    /** Enthlt alle gefunden Austragungsorte, an welche Anfragen geschickt werden. */
    private final ArrayList<Kanal> zielkanaele = new ArrayList<>(Spieler.SPIELER);
    /** Enthlt alle gefunden Austragungsorte, von denen Anfragen erhalten werden knnen. */
    private final ArrayList<Kanal> quellkanaele = new ArrayList<>(Spieler.SPIELER);
    /** Zum Verteilen von Nachrichten. */
    private final ExecutorService exekutor = Executors.newSingleThreadExecutor();
    /** Der Unicast-Socket zur Kommunikation zwecks Verbindungsherstellung zwischen einzelnen JaFuffy-Instanzen. */
    private final ServerSocket willkommen;
    {
        ServerSocket socket = null;
        try {
            socket = new ServerSocket(0);
        } catch (IOException ausnahme) {
            ausnahme.printStackTrace();
        }
        willkommen = socket;
    }
    /** Zur berwachung der Verbindungen zu anderen Austragungsorten. */
    private final Ueberwachung ueberwachung = new Ueberwachung();
    /** Zum Verschicken von Lebenszeichen. */
    private final Timer lebenszeichen = new Timer();
    /** Der Beobachter, welcher eingehende Anfragen bearbeitet. */
    private final Bearbeiter bearbeiter;
    /** Schnittstelle zum Abbruch aufgrund einer Strung. */
    private final Stoerung stoerung;

    /** Zum Warten auf alle eingehenden und ausgehenden Verbindungen. */
    private CountDownLatch verbindungszaehler;
    /** Bentigt zus Aussetzung der Ereignisabarbeitung. */
    private CountDownLatch aussetzungszaehler;
    /** Zum Warten auf einen Zeitpunkt, welcher durch ein Ereignis definiert wird. */
    private final ExecutorService ausfuehrer = Executors.newSingleThreadExecutor();

    /**
     * Erstellt eine Vermittlung, welcher alle Verbindungen zu anderen Austragungsorten erstellt und betreut.
     *
     * @param eroeffnung
     *            Schnittstelle zur Aufnahme des Turnierbetriebs nach erfolgter Organisation bers Netzwerk.
     */
    public Vermittlung(Eroeffnung eroeffnung, Stoerung stoerung) {
        bearbeiter = new Bearbeiter(eroeffnung);
        this.stoerung = stoerung;
    }

    @Override
    public void run() {
        do {
            try {
                behandle(willkommen.accept());
            } catch (SocketException ausnahme) {
                // Stornierung
            } catch (Exception ausnahme) {
                schliesse();
                stoerung.abbreche(ausnahme);
            }
        } while (verbindungszaehler.getCount() > 0 && !willkommen.isClosed());
        try {
            willkommen.close();
        } catch (IOException ausnahme) {
            ausnahme.printStackTrace();
        }
    }

    @Override
    public void stateChanged(ChangeEvent aenderungsereignis) {
        if (Umschlag.adressiert(aenderungsereignis, Ablauf.class)) {
            Umschlag<Ablauf, Turnier> umschlag = Umschlag.ereignisbehaelter(aenderungsereignis);
            Ablauf ereignis = umschlag.ereignis();
            switch (ereignis) {
            case START:
                ueberwache();
                ueberwachung.aktiviere(true);
                break;
            case SPIEL:
                ueberwachung.aktiviere(true);
                aussetzungszaehler.countDown();
                break;
            case WECHSEL:
                aussetzungszaehler = new CountDownLatch(1);
                break;
            case RESULTAT:
                ueberwachung.aktiviere(false);
                break;
            case ABBRUCH:
            case ENDE:
                if (aussetzungszaehler != null) {
                    aussetzungszaehler.countDown();
                }
                SwingUtilities.invokeLater(() -> schliesse());
            default:
            }
        }
    }

    /**
     * Verknpft diese Vermittlung mit den Elementen, welche ein Turnier umsetzen.
     *
     * @param wuerfelsatz
     *            Der gesamte Wrfelsatz, welcher zum Turnier gehrt.
     * @param nachfuehrung
     *            Fhrt die Aktionen nach, welche sich beim Erffnung ber ein Netzwerk auf diesen Austragungsort
     *            auswirken.
     */
    public void verknuepfe(Wurf[] wuerfe, Nachfuehrung nachfuehrung) {
        bearbeiter.verknuepfe(wuerfe, nachfuehrung);
    }

    /**
     * Verteilt in einem separaten Thread ein Ereignis aus dem Turnierablauf ber das Netzwerk an alle Austragungsorte
     * des Turniers.
     *
     * @param ereignis
     *            Das zu verteilende Ereignis.
     * @throws Exception
     *             Ein Problem ist bei der Verteilung aufgetreten.
     */
    public void verteile(Ereignis ereignis) throws Exception {
        Future<Void> fertig = exekutor.submit(() -> {
            verteile((Serializable) ereignis); // Cast zur Vermeidung einer Rekursion
            return null;
        });
        fertig.get();
    }

    /**
     * Behandelt eine neu eingegangene Verbindungsanfrage.
     *
     * @param socket
     *            Der Socket, der aus der Verbindungsanfrage hervorgeht.
     * @throws Exception
     *             Verbindungsanfrage kann nicht behandelt werden.
     */
    private void behandle(Socket socket) throws Exception {
        Bediener bediener = new Bediener(socket);
        bediener.start();
    }

    /** Installiert die berwachung der Austragungsorte mithilfe von Lebenszeichen. */
    private void ueberwache() {
        lebenszeichen.schedule(ueberwachung, VERBINDUNGSTIMEOUT, VERBINDUNGSTIMEOUT);
    }

    /** Schliet diese Vermittlung, so dass das Warten auf eingehende Verbindungsanfragen gestoppt wird. */
    void schliesse() {
        lebenszeichen.cancel();
        try {
            willkommen.close();
            for (Kanal kanal : quellkanaele) {
                kanal.schliesse();
            }
            quellkanaele.clear();
            for (Kanal kanal : zielkanaele) {
                kanal.schliesse();
            }
            zielkanaele.clear();
        } catch (SocketException ausnahme) {
            // Abbruch oder Ende
        } catch (IOException ausnahme) {
            ausnahme.printStackTrace();
        }
    }

    /** Liefert den Socket zurck, welcher Verbindungsanfragen entgegennimmt. */
    ServerSocket socket() {
        return willkommen;
    }

    /**
     * Bereitet sich auf eingehende Verbindungen vor und verbindet sich mit allen Kontakten, auer mit dem hiesigen
     * Vorortkontakt.
     *
     * @param kontakte
     *            Enthlt alle bekannten Kontakte.
     * @param vorortkontakt
     *            Der hiesige Vorortkontakt.
     */
    void verbinde(Kontakt[] kontakte, Kontakt vorortkontakt) {
        verbindungszaehler = new CountDownLatch(2 * (kontakte.length - 1));
        start();
        for (Kontakt kontakt : kontakte) {
            if (!vorortkontakt.equals(kontakt)) {
                try {
                    Socket socket = new Socket(kontakt.adresse(), kontakt.port());
                    ObjectOutputStream versandstrom = new ObjectOutputStream(socket.getOutputStream());
                    ObjectInputStream empfangsstrom = new ObjectInputStream(socket.getInputStream());
                    Kanal kanal = new Kanal(versandstrom, empfangsstrom);
                    System.out.println("? " + socket.getRemoteSocketAddress() + " > " + socket.getLocalSocketAddress()
                            + " auf " + kanal);
                    kanal.anfrage();
                    zielkanaele.add(kanal);
                    System.out.println("? OK auf " + kanal);
                    verbindungszaehler.countDown();
                } catch (Exception ausnahme) {
                    ausnahme.printStackTrace();
                }
            }
        }
    }

    /**
     * Verteilt ein Objekt als Anfrage an alle anderen Austragungsorte.
     *
     * @param objekt
     *            Das zu versendende Objekt.
     * @throws Exception
     *             Objekt konnte nicht verschickt werden.
     */
    synchronized void verteile(Serializable objekt) throws Exception {
        // Zunchst das Objekt versenden, damit alle anderen Austragungsorte diese parallel verarbeiten knnen
        for (Kanal kanal : zielkanaele) {
            kanal.sende(objekt);
        }
        // Und jetzt die Reaktionen einsammeln
        for (Kanal kanal : zielkanaele) {
            kanal.warte();
        }
    }

    /**
     * Wartet darauf, dass alle eingehenden und ausgehenden Verbindungen zustande gekommen sind.
     *
     * @throws InterruptedException
     *             Geworfen, falls eine maximale Zeit berschritten worden ist.
     * @return Gibt an, ob tatschlich alleVerbindungen innerhalb einer gewissen Zeit zustande gekommen sind.
     */
    boolean warte() throws InterruptedException {
        return verbindungszaehler.await(VERBINDUNGSTIMEOUT, TimeUnit.MILLISECONDS);
    }

}