/******************************************************************************
 ** $Id: Anmelder.java 2610 2021-02-01 01:16:44Z wmh $
 ** Diese Datei ist Bestandteil der Java-Quelltexte des Wrfelspiels JaFuffy.
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************
 ** Die aktuellste Version von JaFuffy findet sich im Internet unter
 ** <http://jafuffy.3kelvin.de>.
 **
 ** Kommentare, Fehler oder Erweiterungswnsche bitte per E-Mail senden an
 ** <jafuffy@3kelvin.de>.
 ******************************************************************************/
package jafuffy.netzwerk;

import java.io.BufferedOutputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectOutputStream;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.InetAddress;
import java.net.SocketException;
import java.time.LocalDateTime;
import java.util.LinkedHashSet;
import java.util.Timer;
import java.util.TimerTask;

/** Fhrt die Anmeldung zur Teilnahme an einem Turnier ber ein lokales Netzwerk durch. */
class Anmelder extends Timer {

    /** Wiederholung des Anmeldevorgangs. */
    private class Wiederholung extends TimerTask {
        /** Der Hashwert ber alle Teilnehmer an allen Austragungssttten, von dieser Austragungssttte berechnet. */
        private byte[] teilnehmerhashwert;
        /** Die Identifikation des Vorsitzes, so wie dem lokalen Austragungsort bekannt. */
        private Kontakt vorsitzkontakt;
        /** Der Zeitpunkt der Meldung zum Vorsitz, so wie dem lokalen Austragungsort bekannt. */
        private LocalDateTime vorsitzmeldungszeit;

        @Override
        public void run() {
            try {
                Anmeldung anmeldung;
                synchronized (Anmelder.this) {
                    anmeldung = new Anmeldung(absender, vorortteilnehmer, teilnehmerhashwert, vorsitzkontakt,
                            vorsitzmeldungszeit);
                }
                sende(anmeldung);
            } catch (SocketException ausnahme) {
                cancel();
            } catch (IOException ausnahme) {
                ausnahme.printStackTrace();
            }
        }

        /** Setzt den vor Ort ermittelten Hashwert ber alle Teilnehmer. */
        void setze(byte[] teilnehmerhashwert) {
            this.teilnehmerhashwert = teilnehmerhashwert;
        }

        /** Setzt die vor Ort bekannten Informationen zum Turniervorsitz. */
        void setze(Kontakt vorsitzkontakt, LocalDateTime vorsitzmeldungszeit) {
            this.vorsitzkontakt = vorsitzkontakt;
            this.vorsitzmeldungszeit = vorsitzmeldungszeit;
        }
    }

    /** Anmeldevorgang wird alle 1000ms wiederholt. */
    private static final int ANMELDEPERIODE = 1000;

    /** Absenderadresse dieser Austragungssttte fr das Turnier. */
    private final Kontakt absender;
    /** Teilnehmer dieser Austragungssttte fr das Turnier. */
    private final LinkedHashSet<String> vorortteilnehmer;
    /** Socket Socket zum Senden von Objekten zur Organisation eines Turniers ber ein lokales Netzwerk. */
    private final DatagramSocket socket;
    /** Zur regelmigen Wiederholung der Anmeldung. */
    private final Wiederholung wiederholung = new Wiederholung();

    /**
     * Erzeugt die Anmeldungsnachricht zur Durchfhrung der Anmeldevorgnge.
     *
     * @param absender
     *            Die Absenderadresse dieser Austragungssttte.
     * @param vorortteilnehmer
     *            Die Teilnehmer dieser Austragungssttte.
     * @throws SocketException
     *             Sende-Socket kann nicht erstellt werden.
     */
    Anmelder(Kontakt absender, LinkedHashSet<String> vorortteilnehmer) throws SocketException {
        super(Anmelder.class.getSimpleName());
        this.absender = absender;
        this.vorortteilnehmer = vorortteilnehmer;
        this.socket = new DatagramSocket();
    }

    /** Serialisiert und verschickt ein Objekt ber den Organisationssocket. */
    synchronized private void sende(Object objekt) throws IOException, SocketException {
        ByteArrayOutputStream bytestrom = new ByteArrayOutputStream();
        ObjectOutputStream objektstrom = new ObjectOutputStream(new BufferedOutputStream(bytestrom));
        objektstrom.writeObject(objekt);
        objektstrom.close();
        byte[] bytes = bytestrom.toByteArray();
        bytestrom.close();
        DatagramPacket paket = new DatagramPacket(bytes, bytes.length, InetAddress.getByName(Leitung.ADRESSE),
                Leitung.PORT);
        socket.send(paket);
    }

    /** Setzt den vor Ort ermittelten Hashwert ber alle Teilnehmer. */
    synchronized void setze(byte[] teilnehmerhashwert) {
        wiederholung.setze(teilnehmerhashwert);
    }

    /** Setzt die vor Ort bekannten Informationen zum Turniervorsitz. */
    synchronized void setze(Kontakt vorsitzkontakt, LocalDateTime vorsitzmeldungszeit) {
        wiederholung.setze(vorsitzkontakt, vorsitzmeldungszeit);
    }

    /** Startet die periodische Durchfhrung der Anmeldevorgnge. */
    void starte() {
        scheduleAtFixedRate(wiederholung, 0, ANMELDEPERIODE);
    }

    /** Stoppt alle laufenden oder zuknftigen Sendeversuche. */
    void stoppe() {
        socket.close();
    }

    /** Versendet ein Objekt ber das Netzwerk. */
    void verschicke(Object objekt) {
        try {
            sende(objekt);
        } catch (IOException exception) {
            exception.printStackTrace();
        }
    }

}
