/******************************************************************************
 ** $Id: Aufsicht.java 2620 2021-02-21 21:11:53Z wmh $
 ** Diese Datei ist Bestandteil der Java-Quelltexte des Wrfelspiels JaFuffy.
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************
 ** Die aktuellste Version von JaFuffy findet sich im Internet unter
 ** <http://jafuffy.3kelvin.de>.
 **
 ** Kommentare, Fehler oder Erweiterungswnsche bitte per E-Mail senden an
 ** <jafuffy@3kelvin.de>.
 ******************************************************************************/
package jafuffy.logik;

import java.awt.event.ActionEvent;
import java.net.SocketException;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import jafuffy.Eigenschaften;
import jafuffy.bedienung.Becher;
import jafuffy.bedienung.Fenster;
import jafuffy.bedienung.Setzen;
import jafuffy.bedienung.Verfolgung;
import jafuffy.bedienung.Vorschlagen;
import jafuffy.logik.ereignis.Ablauf;
import jafuffy.logik.ereignis.Umschlag;
import jafuffy.logik.ereignis.Wuerfelstand;
import jafuffy.netzwerk.Vermittlung;
import jafuffy.netzwerk.Vermittlung.Stoerung;
import jafuffy.netzwerk.ereignis.Wuerfelabwahl;
import jafuffy.netzwerk.ereignis.Wuerfelanwahl;
import jafuffy.netzwerk.ereignis.Wurfdarlegung;
import jafuffy.netzwerk.ereignis.Wurfergebnis;
import jafuffy.netzwerk.ereignis.Wurfsetzung;

/**
 * Steuert die Ausfhrung der einzelnen Schritte der Bot-Aktionen, so dass diese von menschlichen Mitspielern
 * nachvollzogen werden knnen.
 */
public class Aufsicht implements Stoerung, ChangeListener {

    /** Schnittstelle zur Ausfhrung eines Aktionsschrittes des Bots. */
    private interface Schritt {
        /**
         * Macht den Aktionsschritt.
         *
         * @param event
         *            Das Ereignis, welches den Schritt auslst.
         */
        void mache(ActionEvent event);
    }

    /** Die Aktion zum Setzen des Wurfes. */
    private class Setzaktion implements Schritt {
        @Override
        public void mache(ActionEvent event) {
            alles = false;
            setzen.doClick();
            alleAktionen.setEnabled(true);
        }

        @Override
        public String toString() {
            return " mchte jetzt setzen.";
        }
    }

    /**
     * Whlt die Wrfel aus, welche der Bot nochmals werfen soll. Muss als einzige Aktion die nchste Aktion
     * registrieren, weil nach dem berlegen kein eigenes Spielereignis wie etwa "gewrfelt" stattfindet.
     */
    private class Ueberlegungsaktion implements Schritt {
        @Override
        public void mache(ActionEvent event) {
            vorschlagen.doClick();
            if (wuerfelbereit()) {
                registriere(wurfaktion);
            } else {
                registriere(setzaktion);
            }
            erledige(alles || autonom());
        }

        @Override
        public String toString() {
            return knapper() ? " berlegt jetzt." : " mchte jetzt berlegen.";
        }
    }

    /** Wurfergebnis aller vom Bot ausgewhlten Wrfel. */
    private class Wurfaktion implements Schritt {
        @Override
        public void mache(ActionEvent event) {
            alleAktionen.setEnabled(false);
            becher.doClick();
        }

        @Override
        public String toString() {
            return knapper() ? " mchte jetzt wrfeln und berlegen." : " mchte jetzt wrfeln.";
        }
    }

    /** Lange Anzeigedauer um Aktion zu verdeutlichen. */
    private static final int AKTIONSDAUER_DEUTLICH = 400;
    /** Kurze Anzeigedauer um Aktion anzudeuten. */
    private static final int AKTIONSDAUER_SCHNELL = 200;
    /** Erlaubt die Simulation eines Turniers mittels Bots mit minimalen Anzeigedauern. */
    private static boolean optimiert;

    /** Bezeichnet die fr den Bot gerade anstehende Aktion. */
    public static final String ANSTEHENDE_AKTION = "Anstehende Aktion";
    /** Bezeichnet alle fr den Bot anstehenden Aktionen. */
    public static final String ALLE_AKTIONEN = "Alle Aktionen";

    /** Stellt Aufsicht auf Simulationsmodus. */
    public static void simuliere() {
        optimiert = true;
    }

    /** Zum Ausfhren aller ausstehenden Aktionen. */
    @SuppressWarnings("serial")
    private final Action alleAktionen = new AbstractAction() {
        @Override
        public void actionPerformed(ActionEvent event) {
            alles = true;
            schritt.mache(event);
        }
    };
    /** Zum Ausfhren der nchsten anstehenden Aktion. */
    @SuppressWarnings("serial")
    private final Action naechsteAktion = new AbstractAction() {
        @Override
        public void actionPerformed(ActionEvent event) {
            schritt.mache(event);
        }
    };
    /** Referenz auf globale Eigenschaften. */
    private final Eigenschaften eigenschaften;
    /** Das Fenster, in dem das Turniergeschehen stattfindet. */
    private final Fenster fenster;
    /** Referenz auf Wrfelbecher vom Wrfelfeld. */
    private final Becher becher;
    /** Referenz zum Meneintrag zum Erstellen eines Vorschlages, wird hier knstlich angeklickt. */
    private final Vorschlagen vorschlagen;
    /** Referenz zum Meneintrag zum Setzen eines Wurfs, wird hier knstlich angeklickt. */
    private final Setzen setzen;
    /** Feld zur Verfolgung der Aktionen eines Spielers, einschlielich eines Bots. */
    private final Verfolgung verfolgung;
    /** Der Schritt zum berlegen fr das weitere Vorgehen beim vorliegenden Wurfergebnis. */
    private final Ueberlegungsaktion ueberlegungsaktion = new Ueberlegungsaktion();
    /** Der Schritt fr den Wurfergebnis aller ausgewhlten Wrfel. */
    private final Wurfaktion wurfaktion = new Wurfaktion();
    /** Der Schritt fr das Setzen des vorliegenden Wurfs. */
    private final Setzaktion setzaktion = new Setzaktion();
    /** Gibt an, ob ein Stellvertreter in einem Turnier ber ein Netzwerk aktiv ist. */
    private boolean nichtverteilend;
    /** Anzahl der bereitgestellten Wrfel, fr den nchsten Spielzug. */
    private int bereitgestellt = 0;

    /**
     * Enthlt die Beobachter fr jeden einzelnen Wrfel, welchen den Aktionen der Wrfel folgen, wobei die Wrfel durch
     * ihren Index identifiziert werden.
     */
    private final ChangeListener[] wuerfelsatzbeobachter = new ChangeListener[Turnier.WUERFELSATZGROESSE];
    {
        for (int w = 0; w < wuerfelsatzbeobachter.length; w++) {
            final int index = w;
            wuerfelsatzbeobachter[index] = anderungsereignis -> {
                Umschlag<Wuerfelstand, Wuerfel> umschlag = Umschlag.ereignisbehaelter(anderungsereignis);
                Wuerfel wuerfel = umschlag.quelle();
                Wuerfelstand ereignis = umschlag.ereignis();
                try {
                    folgeWuerfel(index, wuerfel, ereignis);
                } catch (Exception ausnahme) {
                    abbreche(ausnahme);
                }
            };
        }
    }

    /** Vermittlung zwischen allen Austragungsorten. Null, falls normaler Aufnahme, d.h. nicht ber Netzwerk. */
    private Vermittlung vermittlung;
    /** Das Turnier, welches gerade durchgefhrt wird und hier beaufsichtigt wird. */
    private volatile Turnier turnier;
    /** Der Schritt, welcher als nchstes ansteht. */
    private Schritt schritt;
    /** Gibt an, ob alle Schritt zu einem Vorgang zusammengefasst werden sollen. */
    private boolean alles;

    /**
     * Konstruiert die Aufsicht und verknpft dies mit dem Wrfelbecher und Feld zur Verfolgung aller Aktionen des Bots.
     *
     * @param eigenschaften
     *            Die globale Eigenschaften von JaFuffy.
     * @param fenster
     *            Das Fenster, in dem das Turniergeschehen stattfindet.
     * @param vorschlagen
     *            Meneintrag zum Erstellen einer Wrfelauswahl.
     * @param becher
     *            Der Wrfelbecher aus dem Wrfelfeld.
     * @param verfolgung
     *            Feld zur Verfolgung der Aktionen eines Spielers.
     * @param setzen
     *            Meneintrag zum Setzen des vorliegenden Wurfs.
     */
    Aufsicht(Eigenschaften eigenschaften, Fenster fenster, Vorschlagen vorschlagen, Becher becher,
            Verfolgung verfolgung, Setzen setzen) {
        this.eigenschaften = eigenschaften;
        this.fenster = fenster;
        this.vorschlagen = vorschlagen;
        this.becher = becher;
        this.setzen = setzen;
        this.verfolgung = verfolgung;
        verfolgung.verknuepfe(alleAktionen, ALLE_AKTIONEN,
                "<html>" + "Bot soll alle seine anstehenden Aktionen abschlieen und dann die Wrfel abgeben." + "<br>"
                        + "<small>Mausklick oder Taste \"Ende\"</small>" + "</html>");
        verfolgung.verknuepfe(naechsteAktion, ANSTEHENDE_AKTION,
                "<html>" + "Bot soll nur mit seiner nchsten anstehenden Aktion weitermachen." + "<br>"
                        + "<small>Mausklick oder Taste \"Bild ab\"</small>" + "</html>");
    }

    /**
     * Bricht das Turniergeschehen aufgrund einer Ausnahme ab im Betrieb ber ein Netzwerk.
     *
     * @param ausnahme
     *            Die Ausnahme, aufgrund derer das Turnier von der Aufsicht abgebrochen wird.
     */
    @Override
    public synchronized void abbreche(Exception ausnahme) {
        ausnahme.printStackTrace();
        if (turnier != null) {
            final Turnier ueberbleibsel = turnier;
            SwingUtilities.invokeLater(() -> {
                SwingUtilities.invokeLater(() -> JOptionPane.showMessageDialog(fenster,
                        "Strung im Turnierbetrieb ber Netzwerk.\n" + "Turnier muss abgebrochen werden.\n"
                                + "\nJava-Exception:\n" + ausnahme.getLocalizedMessage(),
                        "JaFuffy (Fehler)", JOptionPane.ERROR_MESSAGE));
                ueberbleibsel.beende();
            });
        }
        turnier = null;
    }

    @Override
    public void stateChanged(ChangeEvent aenderungsereignis) {
        if (Umschlag.adressiert(aenderungsereignis, Ablauf.class)) {
            Umschlag<Ablauf, Turnier> umschlag = Umschlag.ereignisbehaelter(aenderungsereignis);
            Ablauf ereignis = umschlag.ereignis();
            switch (ereignis) {
            case START:
                turnier = umschlag.quelle();
                alleAktionen.setEnabled(true);
                break;
            case ABBRUCH:
            case ENDE:
                turnier = null;
                break;
            default:
            }
            if (turnier != null) {
                if (turnier.aktiver().real()) {
                    folgeSpieler(ereignis);
                } else {
                    folgeBot(ereignis);
                }
            }
        }
    }

    /**
     * Verknpft die Aufsicht mit der Vermittlung.
     *
     * @param vermittlung
     *            Die Vermittlung.
     */
    public void verknuepfe(Vermittlung vermittlung) {
        this.vermittlung = vermittlung;
    }

    /** Liefert die Dauer der Darstellung einer Aktion zurck, in Millisekunden. */
    private int aktionsdauer() {
        int dauer;
        if (optimiert) {
            dauer = 0;
        } else if (eigenschaften.aktiv("Kurzdarstellung")) {
            dauer = AKTIONSDAUER_SCHNELL;
        } else {
            dauer = AKTIONSDAUER_DEUTLICH;
        }
        if (alles) {
            dauer *= 2;
        }
        return dauer;
    }

    /** Gibt, ob Bots vollstndig autonom handeln. */
    private boolean autonom() {
        return eigenschaften.aktiv("Geisterhand");
    }

    /**
     * Sorgt fr die zeitverzgerte Erledigung der nchsten Aktion.
     *
     * @param zulaessig
     *            Gibt an, ob die Erledigung derzeit zulssig ist.
     */
    private void erledige(boolean zulaessig) {
        if (zulaessig) {
            SwingUtilities.invokeLater(() -> verfolgung.erledigeNaechsteAktion(aktionsdauer()));
        }
    }

    /**
     * Folgt den Ereignissen, die sich aus dem Spiel eines Bots ergeben.
     *
     * @param ereignis
     *            Ein Ereignis aus dem Turnierablauf.
     */
    private void folgeBot(Ablauf ereignis) {
        switch (ereignis) {
        case START:
        case SPIEL:
            registriere(ueberlegungsaktion);
            erledige(alles || autonom());
            break;
        case GEWORFEN:
            registriere(ueberlegungsaktion);
            erledige(alles || knapper());
            break;
        case FERTIG:
            registriere(wurfaktion);
            erledige(autonom());
            break;
        default:
            break;
        }
    }

    /**
     * Folgt den Ablufen des Turniergeschehens fr den relevanten (menschlichen) Spieler, allerdings insbesondere nur
     * falls es sich nicht um den Stellvertreter handelt.
     *
     * @param ereignis
     *            Ein Ereignis aus den Ablufen.
     */
    private void folgeSpieler(Ablauf ereignis) {
        if (vermittlung == null) {
            return;
        }
        try {
            switch (ereignis) {
            case START:
                nichtverteilend = turnier.aktiver().stellvertretend();
                break;
            case GEWORFEN:
                nichtverteilend = turnier.aktiver().stellvertretend();
                if (nichtverteilend) {
                    return;
                }
                vermittlung.verteile(new Wurfdarlegung());
                break;
            case FERTIG:
            case RESULTAT:
                bereitgestellt = wuerfelsatzbeobachter.length;
                nichtverteilend = turnier.aktivermerker().stellvertretend();
                if (nichtverteilend) {
                    return;
                }
                vermittlung.verteile(new Wurfsetzung(turnier.kategoriemerker()));
                nichtverteilend = turnier.aktiver().stellvertretend();
                break;
            case SPIEL:
                bereitgestellt = wuerfelsatzbeobachter.length;
                nichtverteilend = turnier.aktiver().stellvertretend();
                break;
            default:
                nichtverteilend = true;
                break;
            }
        } catch (SocketException ausnahme) {
            // Abbruch oder Ende
        } catch (Exception ausnahme) {
            abbreche(ausnahme);
        }
    }

    /**
     * Folgt den verschiedenen Wrfelzustnden, allerdings insbesondere nur falls es sich nicht um den Stellvertreter
     * handelt.
     *
     * @param index
     *            Identifikation des Wrfels
     * @param wuerfel
     *            Der Wrfel, welcher seine Identifikation nicht kennt.
     * @param ereignis
     *            Ein Ereignis aus den Ablufen.
     * @throws Exception
     *             Problem bei Verteilung von abgeleiteten Ereignissen,welche ber das Netzwerk an andere
     *             Austragungsorte bermittelt werden.
     */
    private void folgeWuerfel(int index, Wuerfel wuerfel, Wuerfelstand ereignis) throws Exception {
        if (vermittlung == null || nichtverteilend()) {
            return;
        }
        switch (ereignis) {
        case ABWAHL:
            vermittlung.verteile(new Wuerfelabwahl(index));
            break;
        case ANWAHL:
            vermittlung.verteile(new Wuerfelanwahl(index));
            break;
        case WURF:
            vermittlung.verteile(new Wurfergebnis(index, wuerfel.augen()));
            break;
        }
    }

    /** Gibt an, ob Vorgnge knapper dargestellt werden sollen. */
    private boolean knapper() {
        return !eigenschaften.aktiv("Ueberlegungsbestaetigung");
    }

    /** Ermittelt, ob Ereignisse an andere Teilnehmer im Netzwerk verteilt werden soll. */
    private boolean nichtverteilend() {
        boolean nichtverteilend = this.nichtverteilend;
        if (bereitgestellt > 0) {
            bereitgestellt--;
            if (bereitgestellt == 0) {
                this.nichtverteilend = turnier.aktiver().stellvertretend();
            }
        }
        return nichtverteilend;
    }

    /**
     * Registriert den nchsten Schritt in der Abfolge von Aktionen eines Bots.
     *
     * @param schritt
     *            Der nchste anstehende Schritt.
     */
    private void registriere(Schritt schritt) {
        this.schritt = schritt;
        naechsteAktion.putValue(Action.LONG_DESCRIPTION, turnier.aktiver().anzeigename() + schritt.toString());
    }

    /** Gibt an, ob aktiver Spieler bereit zum Wrfeln ist. */
    private boolean wuerfelbereit() {
        for (Wuerfel wuerfel : turnier.wuerfelsatz()) {
            if (wuerfel.darstellungsmodell().isSelected()) {
                return true;
            }
        }
        return false;
    }

    /**
     * Liefert Wrfelbeobachter zurck, welcher Ablufe aus dem Turnierverlauf fr einen spezifischen Wrfel verfolgt.
     *
     * @param index
     *            Identifiziert den Wrfel.
     * @return Wrfelbeboachter
     */
    ChangeListener wuerfelbeobachter(int index) {
        return wuerfelsatzbeobachter[index];
    }
}
