/******************************************************************************
 ** $Id: Strasse.java 2533 2021-01-03 13:53:57Z wmh $
 ** Diese Datei ist Bestandteil der Java-Quelltexte des Wrfelspiels JaFuffy.
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************
 ** Die aktuellste Version von JaFuffy findet sich im Internet unter
 ** <http://jafuffy.3kelvin.de>.
 **
 ** Kommentare, Fehler oder Erweiterungswnsche bitte per E-Mail senden an
 ** <jafuffy@3kelvin.de>.
 ******************************************************************************/
package jafuffy.logik.analyse;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;

/** Hilfsmittel zur Analyse der Strae. */
class Strasse {
    /** Zhlt alle Eintrge aus dem oberen Teil der Tabelle auf. */
    protected enum Index {
    EINSER, ZWEIER, DREIER, VIERER, FUENFER, SECHSER;
        /**
         * Gibt an, ob der Wurfergebnis als kleine Strae vorliegt.
         * 
         * @param augenhaeufigkeiten
         *            Die Hufigkeiten jeder Augenzahl von Eins bis Sechs, in einem Feld gespeichert.
         */
        public static boolean istKleineStrasse(int[] augenhaeufigkeiten) {
            return augenhaeufigkeiten[DREIER.ordinal()] > 0 && augenhaeufigkeiten[VIERER.ordinal()] > 0
                    && (augenhaeufigkeiten[EINSER.ordinal()] > 0 && augenhaeufigkeiten[ZWEIER.ordinal()] > 0
                            || augenhaeufigkeiten[ZWEIER.ordinal()] > 0 && augenhaeufigkeiten[FUENFER.ordinal()] > 0
                            || augenhaeufigkeiten[FUENFER.ordinal()] > 0 && augenhaeufigkeiten[SECHSER.ordinal()] > 0);
        }

        /**
         * Gibt an, ob der Wurfergebnis als groe Strae vorliegt.
         * 
         * @param augenhaeufigkeiten
         *            Die Hufigkeiten jeder Augenzahl von Eins bis Sechs, in einem Feld gespeichert.
         */
        public static boolean istGrosseStrasse(int[] augenhaeufigkeiten) {
            return (augenhaeufigkeiten[ZWEIER.ordinal()] == 1 && augenhaeufigkeiten[DREIER.ordinal()] == 1
                    && augenhaeufigkeiten[VIERER.ordinal()] == 1 && augenhaeufigkeiten[FUENFER.ordinal()] == 1
                    && (augenhaeufigkeiten[EINSER.ordinal()] == 1 || augenhaeufigkeiten[SECHSER.ordinal()] == 1));
        }

        /** Liefert die Augenzahl zum Eintrag zurck. */
        public int augen() {
            return ordinal() + 1;
        }
    }

    /** Aufzhlung der Straenteilpaare {1, 2}, {2, 5} oder {5, 6} sowie {3, 4}. */
    enum Paar {
        /** Linkes Straenpaar {1, 2}. */
        LINKS(Index.EINSER, Index.ZWEIER),
        /** Inneres Straenpaar {2, 5}. */
        INNEN(Index.ZWEIER, Index.FUENFER),
        /** Rechtes Straenpaar {5, 6}. */
        RECHTS(Index.FUENFER, Index.SECHSER),
        /** Mittleres Straenpaar {3, 4}. */
        MITTE(Index.DREIER, Index.VIERER);
        /** Beschreibt das Teilpaar durch seine Augenzahl (weniger eins). */
        private final Index[] feld = new Index[2];

        /**
         * Konstruktor.
         *
         * @param paar0
         *            erste Augenzahl weniger Eins
         * @param paar1
         *            zweite Augenzahl weniger Eins
         */
        Paar(Index paar0, Index paar1) {
            feld[0] = paar0;
            feld[1] = paar1;
        }

        /** Liefert das Feld, welches das Teilpaar beschreibt. */
        final Index[] feld() {
            return feld;
        }

        /**
         * Ermittle, wie viele Wrfel des Straenteilpaars im Wurfergebnis vorkommen.
         *
         * @param augenhaeufigkeiten
         *            Feld mit den Anzahlen der geworfenen Augen
         * @return Anzahl der Wrfel vom Straenteilpaar, die im Wurfergebnis vorkommen.
         */
        int vorkommen(int[] augenhaeufigkeiten) {
            int gefunden = 0;
            for (Index element : feld) {
                if (augenhaeufigkeiten[element.ordinal()] > 0) {
                    gefunden++;
                }
            }
            return gefunden;
        }
    }

    /** Stochastische Hilfsklasse, welche bedingte Wahrscheinlichkeiten fr die kleine Strae enthlt. */
    static class KleinerIndex implements Serializable {
        /** Zur Serialisierung. */
        private static final long serialVersionUID = -6498075827504305923L;

        /**
         * Tabelle der bergangswahrscheinlichkeiten fr die kleine Strae, indiziert durch das Tupel (aussen, innen,
         * mitte, rest).
         */
        static final Map<KleinerIndex, Float> TABELLE = new HashMap<>();

        /** Belegung der Tabelle mit vorberechneten Werten mit Float-Werten. */
        static {
            tabelliere(0, 0, 1, 1, 0.2129630f);
            tabelliere(0, 0, 1, 2, 0.4891690f);
            tabelliere(0, 0, 2, 1, 0.3611111f);
            tabelliere(0, 0, 2, 2, 0.6584362f);
            tabelliere(0, 1, 0, 1, 0.1481481f);
            tabelliere(0, 1, 0, 2, 0.3912751f);
            tabelliere(0, 1, 1, 1, 0.2500000f);
            tabelliere(0, 1, 1, 2, 0.5150463f);
            tabelliere(0, 1, 2, 1, 0.5555556f);
            tabelliere(0, 1, 2, 2, 0.8024691f);
            tabelliere(0, 2, 0, 1, 0.1388889f);
            tabelliere(0, 2, 0, 2, 0.3526235f);
            tabelliere(0, 2, 1, 1, 0.3055556f);
            tabelliere(0, 2, 1, 2, 0.5177469f);
            tabelliere(1, 0, 0, 1, 0.1018519f);
            tabelliere(1, 0, 0, 2, 0.3410851f);
            tabelliere(1, 0, 1, 1, 0.1666667f);
            tabelliere(1, 0, 1, 2, 0.4293981f);
            tabelliere(1, 0, 2, 1, 0.3611111f);
            tabelliere(1, 0, 2, 2, 0.6296296f);
            tabelliere(1, 1, 0, 1, 0.1388889f);
            tabelliere(1, 1, 0, 2, 0.3526235f);
            tabelliere(1, 1, 1, 1, 0.3055556f);
            tabelliere(1, 1, 1, 2, 0.5177469f);
        }

        /**
         * Trgt Schlssel und Wert in Tabelle ein.
         *
         * @param aussen
         *            Schlsselanteil.
         * @param innen
         *            Schlsselanteil.
         * @param mitte
         *            Schlsselanteil.
         * @param rest
         *            Schlsselanteil.
         * @param p
         *            Wert.
         */
        private static void tabelliere(int aussen, int innen, int mitte, int rest, float p) {
            TABELLE.put(new KleinerIndex(aussen, innen, mitte, rest), p);
        }

        /** Anzahl der auen liegenden gnstigen Augen (1 oder 6). */
        private final int aussen;
        /** Anzahl der innen liegenden gnstigen Augen (2 oder 5). */
        private final int innen;
        /** Anzahl der mittig liegenden gnstigen Augen (3 oder 4). */
        private final int mitte;

        /** Anzahl der restlichen noch mglichen Wrfe. */
        private final int rest;

        /**
         * Konstruiert einen Index, welcher als Schlssel fr die Tabelle benutzt wird.
         *
         * @param aussen
         *            Anzahl der auen liegenden gnstigen Augen (1 oder 6).
         * @param innen
         *            Anzahl der innen liegenden gnstigen Augen (2 oder 5).
         * @param mitte
         *            Anzahl der mittig liegenden gnstigen Augen (3 oder 4).
         * @param rest
         *            Anzahl der restlichen noch mglichen Wrfe.
         */
        KleinerIndex(int aussen, int innen, int mitte, int rest) {
            this.aussen = aussen;
            this.innen = innen;
            this.mitte = mitte;
            this.rest = rest;
        }

        @Override
        public boolean equals(Object obj) {
            KleinerIndex index = (KleinerIndex) obj;
            return aussen == index.aussen && innen == index.innen && mitte == index.mitte && rest == index.rest;
        }

        @Override
        public int hashCode() {
            // [a][b][c][d] ([2][3][3][3]): dcbi+dcj+dk+l=d(cbi+cj+k)+l=d(c(bi+j)+k)+l
            return 3 * (3 * (3 * aussen + innen) + mitte) + rest;
        }
    }
}
