/******************************************************************************
 ** $Id: Analyse.java 2533 2021-01-03 13:53:57Z wmh $
 ** Diese Datei ist Bestandteil der Java-Quelltexte des Wrfelspiels JaFuffy.
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************
 ** Die aktuellste Version von JaFuffy findet sich im Internet unter
 ** <http://jafuffy.3kelvin.de>.
 **
 ** Kommentare, Fehler oder Erweiterungswnsche bitte per E-Mail senden an
 ** <jafuffy@3kelvin.de>.
 ******************************************************************************/
package jafuffy.logik.analyse;

import java.io.Serializable;
import java.util.Arrays;
import java.util.HashMap;
import java.util.IdentityHashMap;
import java.util.Map;
import java.util.Random;

import jafuffy.logik.Kategorie;
import jafuffy.logik.Spieler;
import jafuffy.logik.Turnier;
import jafuffy.logik.Wuerfel;

/** Hlt Eintrge, erstellt Analysen und berechnet Prognose ber die erzielbare Punktzahl bei vorgelegtem Wurfergebnis. */
public abstract class Analyse implements Serializable {

    /** Eintrag in Tabelle. */
    protected abstract class Eintrag implements Serializable {
        /** Zur Serialisierung. */
        private static final long serialVersionUID = -1491939796847928172L;

        /** Wichtung fr freie Felder, Wert liegt im Intervall [0,1]. */
        private double wichtung;

        /** Karte der Entscheidungen, welcher Wrfel erneut gewrfelt werden soll. */
        protected final IdentityHashMap<Wuerfel, Boolean> neuwurfkarte = new IdentityHashMap<>(
                Turnier.WUERFELSATZGROESSE);
        {
            nominiereNeuwurf(true);
        }

        /** Durch den Wurfergebnis tatschlich erzielte Punktzahl. */
        protected int wert;
        /**
         * Voraussichtlicher Zuwachs fr den laufenden Eintrag im Vergleich zum Startmittel, falls beim derzeitigen
         * Wrfelergebnis alle verbleibenden Wurfmglichkeiten ausgeschpft werden.
         */
        protected double zuwachsmittel;
        /** Erwartetes mittleres Ergebnis fr diesen Eintrag bei Start der Runde. */
        protected double startmittel;

        /**
         * Versieht einen Schtzwert eines Mittelwertes mit einer Unsicherheit.
         *
         * @param mittel
         *            Schtzwert fr einen Mittelwert.
         * @param delta
         *            Die Unsicherheit des Spielers oder Bots.
         * @return Neuer unsicherer Schtzwert.
         */
        private double unsicherheit(double mittel, double delta) {
            return mittel * (1 + delta * (2 * ZUFALL.nextDouble() - 1));
        }

        /**
         * Bewertet den vorliegenden Wurfergebnis durch Ermittlung der tatschlich erwrfelten Punktzahl, welche dann auch
         * tatschlich gesetzt wrde.
         *
         * @param spieler
         *            Spieler, fr welchen der vorliegende Wurfergebnis bewertet wird.
         */
        protected abstract void bewerte(Spieler spieler);

        /**
         * Bestimmt ber den Neuwurfstatus des gesamten Wrfelsatzes.
         *
         * @param istNeuwurf
         *            Entscheidet, ob ein Neuwurf anliegt.
         */
        protected void nominiereNeuwurf(boolean istNeuwurf) {
            for (Wuerfel wuerfel : wuerfelsatz) {
                neuwurfkarte.put(wuerfel, istNeuwurf);
            }
        }

        /**
         * Berechnet das mittlere Ergebnis fr diesen laufenden Eintrag bei Ausschpfung aller verbleibenden
         * Wurfmglichkeiten.
         *
         * @param spieler
         *            Der Spieler, der gerade an der Reihe ist.
         * @return Prognose fr das zu erwartende Resultat fr den laufenden Eintrag.
         */
        protected abstract double ergebnismittel(Spieler spieler);

        /**
         * Ermittelt Prognose sprich tippt erwarteten Zuwachs, wobei unter Ignorierung der Setzbarkeit alle
         * Wurfmglichkeiten ausgeschpft werden, und erstellt Wrfelauswahl fr Vorschlag.
         *
         * @param spieler
         *            Spieler, der an der Reihe ist.
         */
        protected void tippe(Spieler spieler) {
            waehleWuerfel(spieler);
            double delta = spieler.unsicherheit();
            zuwachsmittel = unsicherheit(ergebnismittel(spieler), delta) - unsicherheit(startmittel(), delta);
        }

        /**
         * Vorschlag zur Wrfelauswahl ermitteln. Wird vor der Rckgabe des Prognosewertes mittels {@link #prognose}
         * ausgefhrt.
         *
         * @param spieler
         *            Der Spieler, der an der Reihe ist.
         */
        protected void waehleWuerfel(Spieler spieler) {
            for (Wuerfel wuerfel : wuerfelsatz) {
                neuwurfkarte.put(wuerfel, spieler.rest() == 0);
            }
        }

        /** Liefert das zu erwartende Zuwachsmittel. */
        double zuwachsmittel() {
            return zuwachsmittel;
        }

        /**
         * @param wuerfel
         *            Referenz auf Wrfel.
         * @return Soll Wrfel erneut gewrfelt werden?
         */
        boolean erneut(Wuerfel wuerfel) {
            return neuwurfkarte.get(wuerfel);
        }

        /**
         * Wichtung eintragen.
         *
         * @param wichtung
         *            Setzt die Wichtung eines einzelnen Eintrags.
         */
        void gewichte(double wichtung) {
            this.wichtung = wichtung;
        }

        /**
         * Berechnet das mittlere Ergebnis fr diesen laufenden Eintrag, falls noch alle drei Wurfmglichkeiten zur
         * Verfgung stehen.
         */
        double startmittel() {
            return startmittel;
        }

        /** @return Erzielte Punktzahl fr Wurfergebnis. */
        int wert() {
            return wert;
        }

        /** @return Wichtung */
        double wichtung() {
            return wichtung;
        }
    }

    /** Eintrag im oberen Tabellenteil. Mglicher Bonus wird bercksichtigt. */
    protected class Oben extends Eintrag {
        /** Zur Serialisierung. */
        private static final long serialVersionUID = -5829769579272736329L;

        /** Die Augenzahl (1, 2, 3, 4, 5 oder 6) des Eintrags im oberen Teil der Tabelle. */
        private final int augen;
        /** Index fr Eintrag (0, 1, 2, 3, 4 oder 5) im oberen Teil der Tabelle, abgeleitet aus der Augenzahl. */
        private final int index;
        /** Bonus, zu dem dieser Eintrag beitrgt. */
        private final Bonus bonus;

        /**
         * Konstruktor.
         *
         * @param strassenindex
         *            Der Index des oberen Eintrags (0, 1, 2, 3, 4 oder 5).
         * @param bonus
         *            Der Eintrag trgt zu diesem Bonus mit bei.
         */
        Oben(int strassenindex, Bonus bonus) {
            index = strassenindex;
            this.bonus = bonus;
            augen = strassenindex + 1;
            // Zur berprfung der theoretischen berlegungen, wird spter im Turnierverlauf berschrieben
            startmittel = augen * Kombinatorik.fixgleichenmittel(Turnier.WUERFELSATZGROESSE, Turnier.WURFVERSUCHANZAHL);
        }

        /**
         * Berechnet das mittlere Ergebnis fr diesen laufenden Eintrag unter Bercksichtung der angegebenen
         * Randbedingungen.
         *
         * @param haeufigkeit
         *            Gibt an, wie oft die gnstige Augenzahl vorliegt.
         * @param r
         *            Die Anzahl der mglichen Restwrfe.
         * @param gesamt
         *            Die derzeitige Gesamtsumme des aktiven Spielers.
         */
        private double ergebnismittel(int haeufigkeit, int r, int gesamt) {
            double minimum = augen * haeufigkeit;
            double punkte = minimum;
            int n = Turnier.WUERFELSATZGROESSE - haeufigkeit;
            for (int k = 0; k <= n; k++) {
                double p = Kombinatorik.fixgleichenwahrscheinlichkeit(n, k, r);
                double beitrag;
                beitrag = k * augen;
                beitrag += bonus.mehrlingskumulationsprognose(gesamt + minimum + beitrag, augen);
                punkte += p * beitrag;
            }
            return punkte;
        }

        @Override
        protected void bewerte(Spieler spieler) {
            wert = augenhaeufigkeiten[index] * augen;
        }

        @Override
        protected double ergebnismittel(Spieler spieler) {
            return ergebnismittel(augenhaeufigkeiten[index], spieler.rest(), spieler.gesamt());
        }

        @Override
        protected void tippe(Spieler spieler) {
            startmittel = ergebnismittel(0, Turnier.WURFVERSUCHANZAHL, spieler.gesamt());
            super.tippe(spieler);
        }

        @Override
        protected void waehleWuerfel(Spieler spieler) {
            super.waehleWuerfel(spieler);
            for (Wuerfel wuerfel : wuerfelsatz) {
                neuwurfkarte.put(wuerfel, wuerfel.augen() != augen);
            }
        }
    }

    /** Eintrag fr Bonus. */
    protected class Bonus extends Eintrag {
        /** Zur Serialisierung. */
        private static final long serialVersionUID = -3335885336754745784L;

        /** Anzahl der oberen freien Eintrge. */
        private int f;
        /** Kumulierte Summe der Augenzahlen zu den freien oberen Eintrgen. */
        private int kumulation;

        /** Konstruktor. */
        Bonus() {
            f = Kategorie.OBEN.length;
            kumulation = f * (f + 1) / 2;
        }

        /**
         * Erstellt Prognose fr den Bonuseintrag fr den Fall, dass noch alle Wrfelversuche zur Verfgung stehen.
         *
         * @param gesamtsumme
         *            Die derzeitige Gesamtsumme des Spielers.
         * @param frei
         *            Die freien oberen Eintrge.
         * @param kumulationssumme
         *            Die Wertigkeit (Augenzahl) der freien oberen Eintrge aufsummiert.
         */
        private double mehrlingskumulationsprognose(double gesamtsumme, int frei, int kumulationssumme) {
            if (gesamtsumme < BONUSGRENZE) {
                if (kumulationssumme > 0) {
                    int m = (int) Math.ceil((BONUSGRENZE - gesamtsumme) / kumulationssumme * frei);
                    return Kombinatorik.mehrlingskumulationswahrscheinlichkeit(frei, m) * BONUS;
                } else {
                    return 0;
                }
            } else {
                return BONUS;
            }
        }

        @Override
        protected void bewerte(Spieler spieler) {
            if (spieler.gesamt() >= BONUSGRENZE) {
                wert = BONUS;
            } else {
                wert = 0;
            }
        }

        @Override
        protected double ergebnismittel(Spieler spieler) {
            return startmittel;
        }

        @Override
        protected void tippe(Spieler spieler) {
            f = 0;
            kumulation = 0;
            for (Kategorie kategorie : Kategorie.OBEN) {
                if (spieler.setzbar(kategorie)) {
                    f++;
                    Oben oben = (Oben) eintrag(kategorie);
                    kumulation += oben.augen;
                }
            }
            startmittel = mehrlingskumulationsprognose(spieler.gesamt(), f, kumulation);
        }

        @Override
        protected void waehleWuerfel(Spieler spieler) {
            super.waehleWuerfel(spieler); // Wird nicht weiter verwendet, sorgt aber fr definierte Werte.
        }

        /**
         * Gibt die Prognose fr eine Mehrlingskumulation ab, falls fr einen oberen freien Eintrag mit seiner Augenzahl
         * alle verbleibenden Wrfe ausgenutzt werden.
         *
         * @param gesamtsumme
         *            Die derzeitige Gesamtsumme des Spielers.
         * @param augen
         *            Die Wertigkeit (Augenzahl) der freien oberen Eintrge aufsummiert wird um Augenzahl reduziert.
         */
        double mehrlingskumulationsprognose(double gesamtsumme, int augenzahl) {
            return mehrlingskumulationsprognose(gesamtsumme, f - 1, kumulation - augenzahl);
        }
    }

    /** Eintrag im unteren Tabellenteil. */
    protected abstract class Unten extends Eintrag {
        /** Zur Serialisierung bentigt. */
        private static final long serialVersionUID = 8894740065051562154L;

        /** @return Maximale Anzahl von Wrfeln mit gleichen Augenzahlen, von eins bis fnf. */
        protected int paschlaenge() {
            int max = 1;
            for (Map.Entry<Paschaufzaehlung, Boolean> pasch : paschkarte.entrySet()) {
                int groesse = pasch.getKey().groesse();
                boolean istPaschVorhanden = pasch.getValue();
                if (istPaschVorhanden && groesse > max) {
                    max = groesse;
                }
            }
            return max;
        }
    }

    /** Zhlt alle Pasche von Zwei bis Fnf auf, wobei alle Wrfel zu einem festen Augenwert herangezogen werden. */
    protected enum Paschaufzaehlung {
        /** Zweierpasch. */
        ZWEIERPASCH,
        /** Dreierpasch und kein Zweierpasch. */
        DREIERPASCH,
        /** Viererpasch und weder Dreipasch noch Zweierpasch. */
        VIERERPASCH,
        /** Fnferpasch und weder Viererpasch noch Dreierpasch noch Zweierpasch. */
        FUENFERPASCH;

        /**
         * @return Anzahl der Wrfel im Pasch, wobei der Maximalwert fr einen festen Augenwert zurckgegeben wird.
         */
        public int groesse() {
            return ordinal() + 2;
        }
    }

    /** Paschgrundlagen. */
    protected abstract class Pasch extends Unten {
        /** Zur Serialisierung. */
        private static final long serialVersionUID = 8666151619656864766L;

        /** Art des Pasches (3=klein: drei Gleiche, 4=gro: vier Gleiche). */
        private final int paschgroesse;
        /** Mit dieser Augenzahl wird der Pasch versucht. */
        private int paschaugenzahl;

        /**
         * Konstruktor.
         *
         * @param paschgroesse
         *            Art des Pasches (drei: klein, vier: gro)
         */
        protected Pasch(int paschgroesse) {
            this.paschgroesse = paschgroesse;
            startmittel = augensumme(Turnier.WURFVERSUCHANZAHL);
        }

        /**
         * @param r
         *            Anzahl der Restwrfe.
         * @return Mittlere erwartete Augensumme.
         */
        private double augensumme(int r) {
            if (r > 0) {
                double augensumme;
                augensumme = Kombinatorik.gleichenwahrscheinlichkeit(1) * augensumme(r - 1);
                for (int k = 2; k < paschgroesse; k++) {
                    for (int a = 1; a <= 6; a++) {
                        augensumme += Kombinatorik.mehrlingswahrscheinlichkeit(a, k) * augensumme(a, r - 1, k);
                    }
                }
                augensumme += (3.5 * paschgroesse
                        + (Turnier.WUERFELSATZGROESSE - paschgroesse) * Kombinatorik.augenzahl(r))
                        * Kombinatorik.mindestgleichenwahrscheinlichkeit(paschgroesse);
                return augensumme;
            } else {
                return 0;
            }
        }

        /**
         * @param augen
         *            Mit dieser Augenzahl wird Pasch versucht.
         * @param r
         *            Anzahl der verbleibenden Restwrfe (0...3).
         * @param max
         *            Maximale Anzahl der schon vorliegenden Wrfel gleicher Augenzahl.
         * @return Mittlere erreichbare Punktzahl.
         */
        private double prognose(int augen, int r, int max) {
            double punkte = 0;
            if (max < paschgroesse) {
                if (max > 1) {
                    punkte = augensumme(augen, r, max);
                } else {
                    punkte = augensumme(r);
                }
            } else {
                for (Wuerfel wuerfel : wuerfelsatz) {
                    if (neuwurfkarte.get(wuerfel)) {
                        punkte += Kombinatorik.augenzahl(r);
                    } else {
                        punkte += wuerfel.augen();
                    }
                }
            }
            return punkte;
        }

        /**
         * @param a
         *            Fest vorgegebene Augenzahl fr einen Mehrling, wobei bei gleich langen Sequenzen diejenige mit der
         *            greren Augenzahl zu whlen ist.
         * @param r
         *            Anzahl der Restwrfe.
         * @param k
         *            Maximale Anzahl von Wrfeln zur vorgegebenen Augenzahl
         */
        protected abstract double augensumme(int a, int r, int k);

        @Override
        protected void bewerte(Spieler spieler) {
            wert = pasch() ? augensumme : 0;
        }

        /** @return Liegt ein erwrfelter Pasch vor? */
        protected abstract boolean pasch();

        @Override
        protected double ergebnismittel(Spieler spieler) {
            return prognose(paschaugenzahl, spieler.rest(), paschlaenge());
        }

        @Override
        protected void waehleWuerfel(Spieler spieler) {
            super.waehleWuerfel(spieler);
            paschaugenzahl = Wuerfel.MAXIMALAUGENAUGENZAHL;
            while (augenhaeufigkeiten[paschaugenzahl - 1] < paschlaenge()) {
                paschaugenzahl--;
            }
            int haeufigkeit = 0;
            for (Wuerfel wuerfel : wuerfelsatz) {
                if (wuerfel.augen() == paschaugenzahl) {
                    haeufigkeit++;
                    neuwurfkarte.put(wuerfel,
                            haeufigkeit > paschgroesse && Kombinatorik.augenzahl(spieler.rest()) > wuerfel.augen());
                } else {
                    neuwurfkarte.put(wuerfel, !pasch() || Kombinatorik.augenzahl(spieler.rest()) > wuerfel.augen());
                }
            }
        }
    }

    /** Chance. */
    class Chance extends Unten {
        /** Zur Serialisierung. */
        private static final long serialVersionUID = -3411668587180538841L;
        /** Erwartete Punkte (alle Augen aufsummiert). */
        private double punkte;

        /**
         * Konstruktor.
         */
        Chance() {
            startmittel = Turnier.WUERFELSATZGROESSE * Kombinatorik.augenzahl(Turnier.WURFVERSUCHANZAHL);
        }

        @Override
        protected void bewerte(Spieler spieler) {
            wert = augensumme;
        }

        @Override
        protected double ergebnismittel(Spieler spieler) {
            punkte = 0;
            for (Wuerfel wuerfel : wuerfelsatz) {
                punkte += Kombinatorik.augenzahl(wuerfel.augen(), spieler.rest());
            }
            return punkte;
        }

        @Override
        protected void waehleWuerfel(Spieler spieler) {
            super.waehleWuerfel(spieler);
            for (Wuerfel wuerfel : wuerfelsatz) {
                neuwurfkarte.put(wuerfel, Kombinatorik.augenzahl(wuerfel.augen(), spieler.rest()) > wuerfel.augen());
            }
        }
    }

    /** Alle Standardeintrge als Tabelle zusammengefasst. */
    protected abstract class Eintragungen extends HashMap<Kategorie, Eintrag> {
        /** Erlaubt Serialisierung. */
        private static final long serialVersionUID = 8130365096301351531L;

        /** Belegt die Standardeintrge, welche fr alle Varianten der Spielregeln gilt. */
        protected Eintragungen() {
            super(Kategorie.SETZBAR.length, 1);
            Bonus bonus = new Bonus();
            for (int index = 0; index < Kategorie.OBEN.length; index++) {
                put(Kategorie.oben(index + 1), new Oben(index, bonus));
            }
            put(Kategorie.BONUS, bonus);
            put(Kategorie.CHANCE, new Chance());
        }
    }

    /** Zur Serialisierung. */
    private static final long serialVersionUID = -6325984568972413728L;

    /** Grenzpunktzahl fr Bonus. */
    public static final int BONUSGRENZE = 63;
    /** Bonus. */
    public static final int BONUS = 35;

    /** Neutrale Wichtung, falls alle Wichtungen gleich sind. */
    private static final double NEUTRALWICHTUNG = 1;

    /** Zufallsgenerator fr Spielstrken und zur Auswahl von Tipp-Strategien bei Mehrdeutigkeiten. */
    protected static final Random ZUFALL = new Random();

    /** Voraussichtlicher mittlerer Punktstand pro Feld, wobei alle drei Wrfe genutzt werden. */
    private final Eintragungen eintragungen;
    /** Bester Eintrag. */
    private Kategorie optimalzeile;

    /** Anzahl der Wrfel pro Augenzahl. */
    protected final int[] augenhaeufigkeiten = new int[Wuerfel.MAXIMALAUGENAUGENZAHL];
    /** Die Karte ordnet einem Pasch (2, 3, 4, 5 gleiche Wrfel) zu, ob genau ein solcher im Wurfergebnis vorhanden ist. */
    protected final Map<Paschaufzaehlung, Boolean> paschkarte = new HashMap<>(Paschaufzaehlung.values().length, 1);
    /** Im Turnier benutzte Wrfel. */
    protected final Wuerfel[] wuerfelsatz;

    /** Summe aller Augen. */
    protected int augensumme;

    /**
     * Konstruktor.
     *
     * @param wuerfelsatz
     *            Die im Turnier benutzten Wrfel.
     */
    protected Analyse(Wuerfel[] wuerfelsatz) {
        this.wuerfelsatz = wuerfelsatz;
        eintragungen = eintragungen();
    }

    /** @return bester Eintrag (-1 heit keiner vorhanden) */
    public Kategorie bester() {
        return optimalzeile;
    }

    /**
     * Bewertet den vorliegenden Wurfergebnis des Spielers fr die gewnschte Kategorie. Der ermittelte Wert wird im Eintrag
     * abgelegt.
     *
     * @param kategorie
     *            Die gewnschte zu bewertende Kategorie
     * @param spieler
     *            Der Spieler, welcher den Wurfergebnis vorgelegt hat
     */
    public void bewerte(Kategorie kategorie, Spieler spieler) {
        eintrag(kategorie).bewerte(spieler);
    }

    /**
     * @param kategorie
     *            Eintrag in der Tabelle.
     * @param wuerfel
     *            Referenz auf Wrfel.
     * @return Wrfel nochmals wrfeln fr Vorschlag?
     */
    public boolean erneut(Kategorie kategorie, Wuerfel wuerfel) {
        return eintrag(kategorie).erneut(wuerfel);
    }

    /**
     * Alle mglichen erwrfelten Punkte fr alle freien Eintrge errechnen; auch unter Bercksichtigung, ob der Spieler
     * schon berhaupt gewrfelt hat.
     *
     * @param spieler
     *            Fr diesen Spieler werden die Berechnungen durchgefhrt.
     */
    public void errechne(Spieler spieler) {
        // Zhle Wrfel pro Augenzahl, Summe bestimmen
        Arrays.fill(augenhaeufigkeiten, 0);
        augensumme = 0;
        if (spieler.gewuerfelt()) {
            for (Wuerfel wuerfel : wuerfelsatz) {
                augensumme += wuerfel.augen();
                augenhaeufigkeiten[wuerfel.augen() - 1]++;
            }
        }
        // Genau 2, 3, 4, 5 gleiche Wrfel?
        for (Paschaufzaehlung pasch : Paschaufzaehlung.values()) {
            boolean istPaschVorhanden = false;
            for (int augenhaeufigkeit : augenhaeufigkeiten) {
                istPaschVorhanden = augenhaeufigkeit == pasch.groesse() || istPaschVorhanden;
            }
            paschkarte.put(pasch, istPaschVorhanden);
        }
        // Erwrfelte Punktzahl fr alle setzbaren Eintrge
        for (Kategorie kategorie : Kategorie.SETZBAR) {
            bewerte(kategorie, spieler);
        }
        // Bercksichtigung des Bonus aufgrund der bisherigen oberen Eintrge ohne den vorliegenden Wurfergebnis
        bewerte(Kategorie.BONUS, spieler);
    }

    /**
     * @param kategorie
     *            Der zu untersuchende Eintrag.
     * @return Wrfelauswahl entspricht Vorschlag fr Eintrag.
     */
    public boolean istVorschlagEntsprechendWuerfelauswahl(Kategorie kategorie) {
        int[] anzahlenJeAugenAusgewaehlte = new int[Wuerfel.MAXIMALAUGENAUGENZAHL];
        int[] anzahlenJeAugenVorgeschlagene = new int[Wuerfel.MAXIMALAUGENAUGENZAHL];
        for (Wuerfel wuerfel : wuerfelsatz) {
            int a = wuerfel.augen() - 1;
            if (wuerfel.darstellungsmodell().isSelected()) {
                anzahlenJeAugenAusgewaehlte[a]++;
            }
            if (eintrag(kategorie).erneut(wuerfel)) {
                anzahlenJeAugenVorgeschlagene[a]++;
            }
        }
        for (int a = 0; a < Wuerfel.MAXIMALAUGENAUGENZAHL; a++) {
            if (anzahlenJeAugenAusgewaehlte[a] != anzahlenJeAugenVorgeschlagene[a]) {
                return false;
            }
        }
        return true;
    }

    /**
     * Leitet Endstandprognosen aus dem derzeitigen Wurfergebnis ab. Tippt fr jeden freien Eintrag auf die erwartete Endsumme
     * des aktiven Spielers, wobei die Endsumme zu diesem Eintrag gehrt, wobei fr diesen freien Eintrag alle mglichen
     * Wrfe genutzt werden und schlielich gesetzt wird.
     *
     * @param spieler
     *            Spieler, der an Reihe ist.
     */
    public void prognostiziere(Spieler spieler) {
        eintrag(Kategorie.BONUS).tippe(spieler);
        for (Kategorie kategorie : Kategorie.SETZBAR) {
            if (spieler.setzbar(kategorie)) {
                eintrag(kategorie).tippe(spieler);
            }
        }
        gewichte(spieler, findeMinimum(spieler), findeMaximum(spieler));
    }

    /**
     * @param kategorie
     *            Eintragsnummer.
     * @return Erwrfelter Punktstand.
     */
    public final int wert(Kategorie kategorie) {
        return eintrag(kategorie).wert();
    }

    /**
     * @param kategorie
     *            Eintragsnummer.
     * @return Wichtung fr erstellte Vorschlge.
     */
    public double wichtung(Kategorie kategorie) {
        return eintrag(kategorie).wichtung();
    }

    /**
     * @param kategorie
     *            Die Zeile in der Tabelle, zu der ein Eintrag zurckgegeben werden soll.
     * @return Eintrag zur Tabellenzeile
     */
    private Eintrag eintrag(Kategorie kategorie) {
        return eintragungen.get(kategorie);
    }

    /**
     * Ermittelt das Maximum ber alle Endstnde fr alle setzbaren Eintrge, und merkt sich einen Eintrag mit dem
     * voraussichtlich besten Endstand.
     *
     * @param spieler
     *            Der Spieler, der gerade an der Reihe ist.
     * @return Das Maximum aller Endstnde.
     */
    private double findeMaximum(Spieler spieler) {
        double max = Double.NEGATIVE_INFINITY;
        optimalzeile = null;
        for (Kategorie kategorie : Kategorie.SETZBAR) {
            if (spieler.setzbar(kategorie)) {
                double zuwachsmittel = eintrag(kategorie).zuwachsmittel();
                if (zuwachsmittel > max) {
                    optimalzeile = kategorie;
                    max = zuwachsmittel;
                }
            }
        }
        return max;
    }

    /**
     * Ermittelt das Minimum ber alle Endstnde fr alle setzbaren Eintrge.
     *
     * @param spieler
     *            Der Spieler, der gerade an der Reihe ist.
     * @return Das Minimum aller Endstnde.
     */
    private double findeMinimum(Spieler spieler) {
        double min = Double.POSITIVE_INFINITY;
        for (Kategorie kategorie : Kategorie.SETZBAR) {
            if (spieler.setzbar(kategorie)) {
                double zuwachsmittel = eintrag(kategorie).zuwachsmittel();
                if (zuwachsmittel < min) {
                    min = zuwachsmittel;
                }
            }
        }
        return min;
    }

    /**
     * Gewichtet die setzbaren Eintrge nach erwarteter Hhe des Endstandes. Das Gewicht je Eintrag entspricht der
     * linearen Abbildung des Intervals zwischen dem minimalen und maximalen Endstand abgebildet auf das Interval [0,
     * 1].
     *
     * @param spieler
     *            Der Spieler, der gerade an der Reihe ist.
     * @param min
     *            Der minimale erwartete Endstand fr alle setzbaren Eintrge.
     * @param max
     *            Der maximale erwartete Endstand fr alle setzbaren Eintrge.
     */
    private void gewichte(Spieler spieler, double min, double max) {
        for (Kategorie kategorie : Kategorie.SETZBAR) {
            if (spieler.setzbar(kategorie)) {
                if (max - min > 0) {
                    double zuwachsmittel = eintrag(kategorie).zuwachsmittel();
                    eintrag(kategorie).gewichte((zuwachsmittel - min) / (max - min));
                } else {
                    eintrag(kategorie).gewichte(NEUTRALWICHTUNG);
                }
            } else {
                eintrag(kategorie).gewichte(0);
            }
        }
    }

    /** @return Alle Eintragungen in der Tabelle. */
    protected abstract Eintragungen eintragungen();

}
