/******************************************************************************
 ** $Id: Wuerfel.java 2556 2021-01-06 21:03:56Z wmh $
 ** Diese Datei ist Bestandteil der Java-Quelltexte des Wrfelspiels JaFuffy.
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************
 ** Die aktuellste Version von JaFuffy findet sich im Internet unter
 ** <http://jafuffy.3kelvin.de>.
 **
 ** Kommentare, Fehler oder Erweiterungswnsche bitte per E-Mail senden an
 ** <jafuffy@3kelvin.de>.
 ******************************************************************************/
package jafuffy.logik;

import java.io.Serializable;
import java.util.Random;

import javax.swing.ButtonModel;
import javax.swing.JToggleButton.ToggleButtonModel;

import jafuffy.logik.ereignis.Aenderungen;
import jafuffy.logik.ereignis.Wuerfelstand;
import jafuffy.netzwerk.Vermittlung.Wurf;

/** Zustand und Darstellungsmodell des Wrfels. */
public class Wuerfel extends Aenderungen<Wuerfelstand> implements Wurf, Comparable<Wuerfel>, Serializable {

    /** Zur Serialisierung. */
    private static final long serialVersionUID = 6889772262239748227L;

    /** Die Anzahl aller mglichen Augen. */
    public static final int MAXIMALAUGENAUGENZAHL = 6;

    /** Zufallsgenerator. */
    private static final Random ZUFALL = new Random();

    /**
     * Dar Modell zur Darstellung in der Benutzeroberflche. Als Attribut gewhlt um die Serialisierung klein zu halten,
     * ansonsten wrden auch Listener gespeichert werden.
     */
    private transient ButtonModel darstellungsmodell;

    /** Augenzahl (von eins bis sechs). */
    private int augen;
    /** Wrfel zum erneuten Wrfeln vorschlagen? */
    private boolean vorgeschlagen;

    /**
     * Konstruktor.
     *
     * @param augen
     *            Ursprngliche Augenzahl
     */
    Wuerfel(int augen) {
        this.augen = augen;
    }

    /**
     * @return Augenzahl von eins bis sechs
     */
    public int augen() {
        return augen;
    }

    @Override
    public int compareTo(Wuerfel wuerfel) {
        return Integer.compare(augen(), wuerfel.augen());
    }

    /** @return Darstellungsmodell fr die Darstellung des Wrfels in der Benutzeroberflche. */
    @SuppressWarnings("serial")
    public ButtonModel darstellungsmodell() {
        if (darstellungsmodell == null) {
            darstellungsmodell = new ToggleButtonModel() {
                /**
                 * Wrfelauswahl mit Geruscheffekten.
                 *
                 * @param selektiert
                 *            Gibt an, ob der Wrfel ausgewhlt und damit in den Becher gelegt werden soll.
                 */
                @Override
                public void setSelected(boolean selektiert) {
                    boolean feuern = isSelected() ^ selektiert;
                    super.setSelected(selektiert);
                    if (feuern) {
                        Wuerfel.this.fireStateChanged(selektiert ? Wuerfelstand.ANWAHL : Wuerfelstand.ABWAHL);
                    }
                }
            };
        }
        return darstellungsmodell;
    }

    /**
     * Wrfel zum erneuten Werfen vorschlagen.
     *
     * @param vorschlagen
     *            Gibt an, ob der Wrfel zum erneuten Werfen vorgeschlagen ist.
     */
    public void setzeVorschlag(boolean vorschlagen) {
        vorgeschlagen = vorschlagen;
    }

    @Override
    public void uebertrage(int augen) {
        this.augen = augen;
        vorgeschlagen = false;
        fireStateChanged(Wuerfelstand.WURF); // muss vor Wrfelabwahl erfolgen
        darstellungsmodell.setSelected(false);
    }

    /** @return Zum erneuten Wrfeln vorgeschlagen? */
    public boolean vorgeschlagen() {
        return vorgeschlagen;
    }

    /**
     * @param haeufigkeiten
     *            Feld mit allen Augenhufigkeiten.
     * @return Anzahl der Wrfel mit dieser Augenzahl.
     */
    public int vorkommen(int[] haeufigkeiten) {
        return haeufigkeiten[augen - 1];
    }

    @Override
    public void waehle(boolean selektiert) {
        darstellungsmodell.setSelected(selektiert);
    }

    /** Wirf den Wrfel, und nimm den anstehenden Vorschlag zurck. */
    void wirf() {
        augen = ZUFALL.nextInt(MAXIMALAUGENAUGENZAHL) + 1;
        uebertrage(augen);
    }

}
