/******************************************************************************
 ** $Id: Name.java 2522 2020-12-31 16:05:11Z wmh $
 ** Diese Datei ist Bestandteil der Java-Quelltexte des Wrfelspiels JaFuffy.
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************
 ** Die aktuellste Version von JaFuffy findet sich im Internet unter
 ** <http://jafuffy.3kelvin.de>.
 **
 ** Kommentare, Fehler oder Erweiterungswnsche bitte per E-Mail senden an
 ** <jafuffy@3kelvin.de>.
 ******************************************************************************/
package jafuffy.logik;

import java.io.Serializable;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;

import jafuffy.Eigenschaften;

/** Der Name, auf der alle Ausprgungen eines Spielers aufbauen, die an einem Turnier teilnehmen. */
public class Name implements Serializable {

    /** Zur Serialisierung. */
    private static final long serialVersionUID = 7811283064202045649L;

    /** Zhlt alle Typen des Spielers auf. */
    public enum Typus {
        /** Nimmt niemals an einem Turnier teil. */
        GEIST(' ', null), /** */
        /**
         * Steht fr einen normalen Spieler, welcher dem Menschen entspricht, der JaFuffy bedient. "@" wird als "direkt
         * gerichtet an" gelesen.
         */
        SPIELER('@', Spieler.class),
        /**
         * Steht fr einen Bot-Spieler, welcher autonom spielen kann. Das doppelte Anfhrungszeichen wird als "autonom"
         * gelesen.
         */
        BOT('"', Bot.class), /** */
        /**
         * Steht fr einen Vertreter des Spielers, welcher dem Menschen entspricht, welcher ein anderes JaFuffy im
         * Netzwerk bedient. "&" wird als "referenziert" gelesen.
         */
        VERTRETER('&', Vertreter.class);

        /** Namensprfix, welcher zum Typus des Spielers passt. */
        public final char praefix;
        /** Zur Konstruktion der Ausprgung des Spielers, welche zum Typus der Aufzhlung gehrt. */
        private final Class<? extends Spieler> klasse;

        /**
         * Zur Konstruktion der Aufzhlungselemente.
         *
         * @param praefix
         *            Namensprfix, welcher zum Typus des Spielers passt.
         * @param klasse
         *            Zur Konstruktion der Ausprgung des Spielers, welche zum Typus der Aufzhlung gehrt.
         */
        Typus(char praefix, Class<? extends Spieler> klasse) {
            this.praefix = praefix;
            this.klasse = klasse;
        }

    }

    /**
     * Prft die Gltigkeit des Anfangs des Anzeigennamens.
     *
     * @param anzeigenamenanfang
     *            Anfang des Anzeigenamens.
     * @return Ob Gltigkeit gegeben ist oder nicht.
     */
    public static boolean anfangverboten(String anzeigenamenanfang) {
        return !anzeigenamenanfang.isEmpty() && (Character.isWhitespace(anzeigenamenanfang.charAt(0))
                || anzeigenamenanfang.charAt(0) == Typus.BOT.praefix
                || anzeigenamenanfang.charAt(0) == Typus.VERTRETER.praefix);
    }

    /**
     * Extrahiert den Anzeigenamen aus dem Verwaltungsnamen unter Bercksichtigung einer mglichen Kennzeichnung fr
     * einen Bot oder einen Vertreter.
     *
     * @param verwaltungsname
     *            Der Name, unter dem der Spieler zwecks interner Verwaltung gefhrt wird.
     * @return Anzeigename des Spielers
     */
    public static String anzeigename(String verwaltungsname) {
        return verwaltungsname.isEmpty() ? verwaltungsname : verwaltungsname.substring(1);
    }

    /**
     * Gibt an, ob hier unter dem Vewaltungsnamen oder Speichernamen ein Bot vorliegt.
     *
     * @param verwaltungsname
     *            Der Name, unter dem der Spieler aus Verwaltungsgrnden gefhrt wird oder als Name in den Eigenschaften
     *            gespeichert wird.
     * @return Gibt an, ob ein Bot vorliegt.
     */
    public static boolean kuenstlich(String verwaltungsname) {
        return !verwaltungsname.isEmpty() && verwaltungsname.charAt(0) == Typus.BOT.praefix;
    }

    /**
     * Berechnet den Verwaltungsnamen, so wie er intern zur Verwaltung benutzt wird.
     *
     * @param anzeigename
     *            Der anzuzeigende Name.
     * @param typus
     *            Der Typus des Spielers.
     */
    public static String verwaltungsname(String anzeigename, Typus typus) {
        return typus.praefix + anzeigename;
    }

    /** Index des von diesem Namen abgeleiteten Spielers, eindeutig ber alle Spieler hinweg fr ein Turnier. */
    private final int index;
    /** Typus dieses Spielers. */
    private final Typus typus;

    /** Der Spielername, wie er bei der Absprache gewhlt wurde und auf dem Zettel angezeigt wird. */
    protected final String anzeigename;
    /** Der Verwaltungsname, so wie er intern zur Verwaltung benutzt wird. */
    protected final String verwaltungsname;

    /**
     * Konstruiert den Namen fr einen Spielers.
     *
     * @param anzeigename
     *            Der anzuzeigende Name.
     * @param typus
     *            Der Typus des Spielers.
     * @param index
     *            Bestimmt die Reihenfolge in der Spielabfolge und auf dem Punktezettel.
     */
    public Name(String anzeigename, Typus typus, int index) {
        this.anzeigename = anzeigename;
        this.verwaltungsname = verwaltungsname(anzeigename, typus);
        this.typus = typus;
        this.index = index;
    }

    /**
     * Baut einen Spieler auf Basis eines gegebenen Namens.
     *
     * @param name
     *            Basisname.
     */
    protected Name(Name name) {
        this(name.anzeigename, name.typus, name.index);
    }

    /**
     * Baut einen Spieler auf Basis eines gegebenen Namens.
     *
     * @param name
     *            Basisname.
     * @param typus
     *            Der Typus des Spielers.
     */
    protected Name(Name name, Typus typus) {
        this(name.anzeigename, typus, name.index);
    }

    /** @return Liefert den Namen, so wie er lesbar angezeigt werden soll. */
    public String anzeigename() {
        return anzeigename;
    }

    /** Liefert den Index des Spielers zurck. */
    public int index() {
        return index;
    }

    /** @return Liefert den Verwaltungsnamen, so wie er intern zur Verwaltung benutzt wird. */
    public String verwaltungsname() {
        return verwaltungsname;
    }

    /**
     * Konstruiert die Ausprgung des Spielers, welche zum Typus dieses Namens gehrt.
     *
     * @param eigenschaften
     *            Einige Eigenschaften werden zustzlich zur Konstruktion bentigt, wie etwa die Spielstrke von Bots.
     * @return Die Ausprgung des Spielers.
     * @throws NoSuchMethodException
     * @throws InvocationTargetException
     * @throws IllegalArgumentException
     * @throws IllegalAccessException
     * @throws InstantiationException
     */
    Spieler spieler(Eigenschaften eigenschaften) throws NoSuchMethodException, InstantiationException,
            IllegalAccessException, IllegalArgumentException, InvocationTargetException {
        Constructor<? extends Spieler> konstruktor = typus.klasse
                .getConstructor(new Class[] { Name.class, Eigenschaften.class });
        return konstruktor.newInstance(this, eigenschaften);
    }

}
