/******************************************************************************
 ** $Id: Bestenliste.java 2621 2021-02-27 01:24:16Z wmh $
 ** Diese Datei ist Bestandteil der Java-Quelltexte des Wrfelspiels JaFuffy.
 ** Lauffhig ab Java 7.
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************
 ** Die aktuellste Version von JaFuffy findet sich im Internet unter
 ** <http://jafuffy.3kelvin.de>.
 **
 ** Kommentare, Fehler oder Erweiterungswnsche bitte per E-Mail senden an
 ** <jafuffy@3kelvin.de>.
 ******************************************************************************/
package jafuffy.bedienung;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.FontMetrics;
import java.awt.Insets;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.text.NumberFormat;
import java.util.HashMap;
import java.util.Locale;

import javax.swing.BorderFactory;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTabbedPane;
import javax.swing.JTable;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;
import javax.swing.border.Border;
import javax.swing.border.EtchedBorder;
import javax.swing.border.MatteBorder;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.TableCellRenderer;

import jafuffy.logik.Bestenlistenmodell;
import jafuffy.logik.Statistik;
import jafuffy.logik.Variante;
import jafuffy.logik.ereignis.Aktualisierung;
import jafuffy.logik.ereignis.Umschlag;

/**
 * Anzeige der Bestenlisten (Name, Datum, Punktzahl, Kommentar), aufgeschlsselt nach Spielvarianten. Gleiche Punktzahl
 * bedeutet gleicher Rang. Gezeigt werden die ersten 10 Rnge.
 */
@SuppressWarnings("serial")
public class Bestenliste extends JDialog implements PropertyChangeListener, ChangeListener {

    /** Spaltenindizierung. */
    private enum Spalten {
        RANG, NAME, PUNKTE, DATUM, KOMMENTAR
    }

    /** Abstand von Elemente zum Dialograhmen. */
    private static final int DIALOGRAHMENABSTAND = 8;
    /** Icon fr den Rahmen um die Tabellen der Besten. */
    private static final ImageIcon MEDAILLE = Grafik.icon("Goldwuerfel.png");
    /** Breite fr Tabelle. */
    private static final int SCROLLFELDBREITE = 15 * MEDAILLE.getIconWidth();
    /** Hhe fr Tabelle. */
    private static final int SCROLLFELDHOEHE = 7 * MEDAILLE.getIconWidth();

    /** Abstand der Zellen zu horizontalen Spaltengrenzen. */
    private static final int ZWISCHENRAUM = DIALOGRAHMENABSTAND;
    /** Statistisches Datenfeld. */
    private final JLabel[] anzahl = new JLabel[Variante.ANZAHL];
    /** Modelle fr die einzelnen Karteikarten, je nach Regelvariante. */
    private final HashMap<Variante, Bestenlistenmodell> bestenlistenmodelle;
    /** Statistisches Datenfeld. */
    private final JLabel[] durchschnitt = new JLabel[Variante.ANZAHL];
    /** Karteikarten fr Bestenlisten (nach Spielregeln aufgeschlsselt). */
    private final JTabbedPane kartei = new JTabbedPane();
    /** Statistisches Datenfeld. */
    private final JLabel[] standardabweichung = new JLabel[Variante.ANZAHL];

    /** Die Statistikdaten. */
    private final Statistik statistik;

    /**
     * Konstruktor.
     *
     * @param statistik
     *            Daten fr die Bestenlisten
     * @param variantenindex
     *            Spielvariante bei Programmstart
     */
    public Bestenliste(Statistik statistik, int variantenindex) {
        this.statistik = statistik;
        UIManager.addPropertyChangeListener(this);
        bestenlistenmodelle = statistik.bestenlistenmodelle();
        erstelleKarteikarten(variantenindex);
        zeigeStatistikkenndaten();
        setContentPane(hauptfeld(ok()));
        setTitle("JaFuffy (Liste der zehn besten Punktzahlen)");
        setIconImages(Oberflaeche.LOGOS);
        pack();
    }

    @Override
    public void propertyChange(PropertyChangeEvent ereignis) {
        if (ereignis.getPropertyName().equals("lookAndFeel")) {
            SwingUtilities.updateComponentTreeUI(this);
        }
    }

    @Override
    public void stateChanged(ChangeEvent ereignis) {
        if (Umschlag.adressiert(ereignis, Aktualisierung.class)) {
            Umschlag<Aktualisierung, Statistik> umschlag = Umschlag.ereignisbehaelter(ereignis);
            switch (umschlag.ereignis()) {
            case TURNIER_START:
            case TURNIER_WEITER:
                kartei.setSelectedIndex(statistik.variante().ordinal());
                break;
            case SPIEL_ENDE:
                bestenlistenmodelle.get(statistik.variante()).aktualisieren();
                kartei.setSelectedIndex(statistik.variante().ordinal());
                zeigeStatistikkenndaten();
                break;
            default:
                break;
            }
        }
    }

    /** Erstellt fr jede Regelvariante eine Karteikarte mit Statistikkenndaten und einer Bestenliste. */
    private void erstelleKarteikarten(int variantenindex) {
        for (Variante variante : Variante.values()) {
            int i = variante.ordinal();
            JPanel statistikfelder = new JPanel(new FlowLayout());
            anzahl[i] = new JLabel();
            anzahl[i].setToolTipText("Anzahl aller bisherigen Spiele (" + variante.toString() + ")");
            statistikfelder.add(anzahl[i]);
            durchschnitt[i] = new JLabel();
            durchschnitt[i].setToolTipText("Punktedurchschnitt aller bisherigen Spiele (" + variante.toString() + ")");
            statistikfelder.add(durchschnitt[i]);
            standardabweichung[i] = new JLabel();
            standardabweichung[i]
                    .setToolTipText("Standardabweichung fr bisherige Spiele (" + variante.toString() + ")");
            statistikfelder.add(standardabweichung[i]);
            JPanel daten = new JPanel(new BorderLayout());
            daten.add(statistikfelder, BorderLayout.NORTH);
            daten.add(scrollfeld(tabelle(bestenlistenmodelle.get(variante))), BorderLayout.CENTER);
            kartei.addTab(variante.toString(), daten);
        }
        kartei.setSelectedIndex(variantenindex);
    }

    /**
     * Erstellt das Hauptfeld des Dialogs.
     *
     * @param ok
     *            OK-Button
     * @return Das Hauptfeld
     */
    private JPanel hauptfeld(JButton ok) {
        JPanel hauptfeld = new JPanel(new BorderLayout(0, ZWISCHENRAUM));
        hauptfeld.setBorder(BorderFactory.createEmptyBorder(DIALOGRAHMENABSTAND, DIALOGRAHMENABSTAND,
                DIALOGRAHMENABSTAND, DIALOGRAHMENABSTAND));
        hauptfeld.add(kartei, BorderLayout.CENTER);
        hauptfeld.add(ok, BorderLayout.SOUTH);
        return hauptfeld;
    }

    /**
     * Erzeugt den OK-Button und macht ihn zum Standard des Dialogs.
     *
     * @return OK-Button
     */
    private JButton ok() {
        JButton ok = new JButton("OK");
        ok.addActionListener(e -> setVisible(false));
        getRootPane().setDefaultButton(ok);
        return ok;
    }

    /**
     * Erstellt das Scrollfeld, welches die Bestenlistentabelle umfasst.
     *
     * @param tabelle
     *            Die Bestenlistetabelle.
     * @return Das Scrollfeld.
     */
    private JScrollPane scrollfeld(JTable tabelle) {
        JScrollPane scrollfeld = new JScrollPane(tabelle);
        Border rahmen = BorderFactory.createCompoundBorder(
                BorderFactory.createCompoundBorder(BorderFactory.createRaisedBevelBorder(),
                        BorderFactory.createLoweredBevelBorder()),
                BorderFactory.createCompoundBorder(new MatteBorder(MEDAILLE), new EtchedBorder(EtchedBorder.LOWERED)));
        scrollfeld.setBorder(rahmen);
        Insets insets = rahmen.getBorderInsets(scrollfeld);
        scrollfeld.setPreferredSize(new Dimension(SCROLLFELDBREITE + (insets.left + insets.right) / 2,
                SCROLLFELDHOEHE + (insets.top + insets.bottom)));
        return scrollfeld;
    }

    /**
     * Berechnet berschlagsweise Spaltenbreite.
     *
     * @param zeichner
     *            Der Zeichner fr den Spaltentitel.
     * @param titel
     *            Spaltentitel.
     * @return Spaltenbreite.
     */
    private int spaltenbreite(Component zeichner, String titel) {
        FontMetrics metrik = zeichner.getFontMetrics(zeichner.getFont());
        int breite = metrik.stringWidth(titel);
        return breite;
    }

    /**
     * Erzeugt die eigentliche Tabelle zur Auflistung der besten Punktzahlen.
     *
     * @param modell
     *            Das Modell, welches zur Darstellung kommt.
     * @return Die Tabelle.
     */
    private JTable tabelle(Bestenlistenmodell modell) {
        JTable tabelle = new JTable(modell) {
            @Override
            public Component prepareRenderer(TableCellRenderer renderer, int zeile, int spalte) {
                DefaultTableCellRenderer zelle = (DefaultTableCellRenderer) super.prepareRenderer(renderer, zeile,
                        spalte);
                zelle.setBorder(BorderFactory.createEmptyBorder(0, ZWISCHENRAUM, 0, ZWISCHENRAUM));
                return zelle;
            }
        };
        Component zeichner = (Component) tabelle.getTableHeader().getDefaultRenderer();
        tabelle.getColumnModel().getColumn(Spalten.RANG.ordinal()).setPreferredWidth(spaltenbreite(zeichner, "Rang"));
        tabelle.getColumnModel().getColumn(Spalten.NAME.ordinal())
                .setPreferredWidth(spaltenbreite(zeichner, "!Mitspieler!"));
        tabelle.getColumnModel().getColumn(Spalten.PUNKTE.ordinal())
                .setPreferredWidth(spaltenbreite(zeichner, "Punkte"));
        tabelle.getColumnModel().getColumn(Spalten.DATUM.ordinal())
                .setPreferredWidth(spaltenbreite(zeichner, " 00.00.0000"));
        tabelle.getColumnModel().getColumn(Spalten.KOMMENTAR.ordinal())
                .setMinWidth(spaltenbreite(zeichner, "Ein Spielerkommentar."));
        tabelle.setRowSelectionAllowed(false);
        return tabelle;
    }

    /** Belege Ausgabefelder mit statistische Daten. */
    private void zeigeStatistikkenndaten() {
        for (Variante variante : Variante.values()) {
            int i = variante.ordinal();
            anzahl[i].setText("Anzahl: " + statistik.n(variante));
            if (statistik.n(variante) > 1) {
                NumberFormat format = NumberFormat.getInstance(Locale.GERMAN);
                format.setMinimumFractionDigits(1);
                format.setMaximumFractionDigits(1);
                durchschnitt[i].setText("Durchschnitt: " + format.format(statistik.durchschnitt(variante)));
                standardabweichung[i]
                        .setText("Standardabweichung: " + format.format(statistik.standardabweichung(variante)));
            } else {
                durchschnitt[i].setText("Durchschnitt: -");
                standardabweichung[i].setText("Standardabweichung: -");
            }
        }
    }

}
