/******************************************************************************
 ** $Id: Absprache.java 2620 2021-02-21 21:11:53Z wmh $
 ** Diese Datei ist Bestandteil der Java-Quelltexte des Wrfelspiels JaFuffy.
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************
 ** Die aktuellste Version von JaFuffy findet sich im Internet unter
 ** <http://jafuffy.3kelvin.de>.
 **
 ** Kommentare, Fehler oder Erweiterungswnsche bitte per E-Mail senden an
 ** <jafuffy@3kelvin.de>.
 ******************************************************************************/
package jafuffy.bedienung;

import java.awt.BorderLayout;
import java.awt.CardLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.GridLayout;
import java.awt.Insets;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.net.SocketException;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;

import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JSlider;
import javax.swing.JTextField;
import javax.swing.ProgressMonitor;
import javax.swing.SwingConstants;
import javax.swing.SwingUtilities;
import javax.swing.SwingWorker;
import javax.swing.Timer;
import javax.swing.UIManager;
import javax.swing.border.EmptyBorder;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.text.AbstractDocument;
import javax.swing.text.AttributeSet;
import javax.swing.text.BadLocationException;
import javax.swing.text.DocumentFilter;

import jafuffy.Eigenschaften;
import jafuffy.bedienung.netzwerk.Organisation;
import jafuffy.logik.Beginner;
import jafuffy.logik.Bot;
import jafuffy.logik.Manager;
import jafuffy.logik.Name;
import jafuffy.logik.Name.Typus;
import jafuffy.logik.Plan;
import jafuffy.logik.Spieler;
import jafuffy.logik.Variante;
import jafuffy.netzwerk.Leitung;
import jafuffy.netzwerk.Leitung.Phase;

/**
 * Dialog zum Spielbeginn, Absprache der Spielbedingungen. Eingabe der Spielernamen, der Spielvariante und der Anzahl
 * der Spiele (optional). Auf Wunsch Organisation eines Turniers im Netzwerkbetrieb. Start des Turniers.
 */
@SuppressWarnings("serial")
public class Absprache extends JDialog implements PropertyChangeListener {

    /** Ermglicht Aktivierung und Deaktivierung eines Bots. */
    private static class Botselektion extends JCheckBox implements ActionListener {

        /** Schriftfarbe, falls Bot inaktiv ist. */
        private static final Color BOT_INAKTIV_FARBE = Color.GRAY;
        /** Schriftfarbe, falls Bot aktiv ist. */
        private static final Color BOT_AKTIV_FARBE = Color.ORANGE.darker();

        /** Bentigt, weil Bot nicht fr alle Varianten zur Verfgung steht. */
        private final JComboBox<Variante> variantenauswahl;

        /**
         * Konstruiert das Auswahlfeld, welches den Bot aktivieren oder deaktivieren kann.
         *
         * @param nominierung
         *            Zugehriges Nominierungsfeld.
         * @param variantenauswahl
         *            Variantenauswahl, welche die Verfgbarkeit des Bots beeinflusst.
         */
        Botselektion(Nominierung nominierung, JComboBox<Variante> variantenauswahl) {
            super("Bot");
            this.variantenauswahl = variantenauswahl;
            addActionListener(nominierung);
            setIcon(RUHEBOT);
            setRolloverIcon(AKTIVBOT);
            setSelectedIcon(AKTIVBOT);
            setToolTipText("<html>" + "<p>Wenn angewhlt, bernimmt <i>JaFuffy</i> die Rolle des Spielers.</p>"
                    + "<p><small>" + "Spielstrke im Men: Einstellungen&rarr;Botverhalten&rarr;Spielstrke der Bots..."
                    + "</small></p>" + "<p><small>Bot steht bei erweiterten Regeln nicht zur Verfgung.</small></p>"
                    + "</html>");
            addActionListener(this);
            boolean botfaehig = variantenauswahl.getSelectedItem() != Variante.ERWEITERT;
            setEnabled(botfaehig);
            if (botfaehig) {
                if (isSelected()) {
                    setForeground(BOT_AKTIV_FARBE);
                } else {
                    setForeground(BOT_INAKTIV_FARBE);
                }
            } else {
                setSelected(false);
                setForeground(Color.LIGHT_GRAY);
            }
        }

        @Override
        public void actionPerformed(ActionEvent event) {
            boolean botfaehig = variantenauswahl.getSelectedItem() != Variante.ERWEITERT;
            setEnabled(botfaehig);
            if (botfaehig) {
                if (isSelected()) {
                    setForeground(BOT_AKTIV_FARBE);
                } else {
                    setForeground(BOT_INAKTIV_FARBE);
                }
            } else {
                setSelected(false);
                setForeground(Color.LIGHT_GRAY);
            }
        }

    }

    /** Festlegung auf knstliche Intelligenz oder menschlichen Spieler mit Nameneingabe. */
    private static class Nominierung extends JPanel implements ActionListener {

        /** Filter fr die Eingabe der Spielernamen. */
        private static class Filter extends DocumentFilter {
            @Override
            public void insertString(FilterBypass fb, int offset, String string, AttributeSet attr)
                    throws BadLocationException {
                if (gefiltert(fb, offset, 0, string)) {
                    return;
                }
                super.insertString(fb, offset, string, attr);
            }

            @Override
            public void remove(FilterBypass fb, int offset, int length) throws BadLocationException {
                String string = fb.getDocument().getText(length, fb.getDocument().getLength() - length);
                if (gefiltert(fb, offset, fb.getDocument().getLength(), string)) {
                    return;
                }
                super.remove(fb, offset, length);
            }

            @Override
            public void replace(FilterBypass fb, int offset, int length, String string, AttributeSet attr)
                    throws BadLocationException {
                if (gefiltert(fb, offset, length, string)) {
                    return;
                }
                super.replace(fb, offset, length, string, attr);
            }

            /** Filtert auf Eingabelnge, meldet Filterung per Tonsignal und gibt zurck, ob gefiltert wurde. */
            private boolean gefiltert(FilterBypass fb, int offset, int laenge, String string) {
                boolean gefiltert = fb.getDocument().getLength() - laenge + string.length() > EINGABELAENGE
                        || offset == 0 && Name.anfangverboten(string);
                if (gefiltert) {
                    Toolkit.getDefaultToolkit().beep();
                }
                return gefiltert;
            }
        }

        /** Karte fr menschlichen Spieler. */
        private static final String MENSCH = "Mensch";
        /** Karte fr knstliche Intelligenz. */
        private static final String BOT = "Bot";
        /** Eingabelngenbeschrnkung fr Spielernamen. */
        private static final int EINGABELAENGE = SPALTEN * 7 / 4;

        /** Bentigt zur Aktualisierung des Absprachedialogs aufgrund von Namenseingaben. */
        private final Absprache absprache;
        /** Nameneingabe fr menschlichen Spieler. */
        private final JTextField spielernameneingabe = new JTextField(SPALTEN);
        /** Name der knstlichen Intelligenz. */
        private final JComboBox<String> botnamenauswahl = new JComboBox<>();

        /** Gibt an, ob die knstliche Intelligenz die Rolle als Spieler bernehmen soll. */
        private boolean kuenstlich;

        /**
         * Konstruiert ein Eingabefeld fr die Nameneingabe oder Namenanzeige.
         *
         * @param absprache
         *            Der Dialog, dessen Aussehen an dieses Objekt angepasst werden muss.
         */
        Nominierung(Absprache absprache) {
            super(new CardLayout());
            this.absprache = absprache;
            spielernameneingabe
                    .setToolTipText("<html>" + "<p>Eingabe der Spielernamen, bis zu " + EINGABELAENGE + " Zeichen.</p>"
                            + "<p>Namen knnen nicht mit Leer- oder Anfhrungszeichen beginnen.</p>" + "</html>");

            add(spielernameneingabe, MENSCH);
            add(botnamenauswahl, BOT);
        }

        /** Bearbeitet die Entscheidung zur knstlichen Intelligenz. */
        @Override
        public void actionPerformed(ActionEvent event) {
            String kommando = event.getActionCommand();
            if (kommando.equals("Bot")) {
                botnamenauswahl.removeAllItems();
                kuenstlich = ((JCheckBox) event.getSource()).isSelected();
                CardLayout layout = (CardLayout) getLayout();
                if (kuenstlich) {
                    for (String botname : Bot.BOTNAMEN) {
                        if (absprache.frei(botname)) {
                            botnamenauswahl.addItem(botname);
                        }
                    }
                    layout.show(this, BOT);
                } else {
                    layout.show(this, MENSCH);
                }
                absprache.aktualisiere();
            } else if (kommando.equals("Var")) {
                @SuppressWarnings("unchecked")
                JComboBox<Variante> variantenauswahl = (JComboBox<Variante>) event.getSource();
                if (variantenauswahl.getSelectedItem() == Variante.ERWEITERT) {
                    kuenstlich = false;
                    CardLayout layout = (CardLayout) getLayout();
                    layout.show(this, MENSCH);
                }
            }
        }

        @Override
        public String toString() {
            return kuenstlich ? (String) botnamenauswahl.getSelectedItem() : spielernameneingabe.getText().trim();
        }

        /**
         * Aktiviert die Eingabe der Spielernamen.
         *
         * @param aktiv
         *            Gibt an, ob Eingabe aktiv sein soll oder nicht.
         */
        void aktiviere(boolean aktiv) {
            spielernameneingabe.setEnabled(aktiv);
        }

        /**
         * Entfernt einen Botnamen aus der Namensauswahl.
         *
         * @param botname
         *            Der zu entfernende Botname.
         */
        void entferne(String botname) {
            botnamenauswahl.removeItem(botname);
        }

        /**
         * Erweitert die Namenauswahl um einen Botnamen.
         *
         * @param botname
         *            Der hinzuzufgende Name.
         */
        void erweitere(String botname) {
            botnamenauswahl.addItem(botname);
        }

        /** Installiert die Beobachter fr die Eingabe der Spielernamen (inklusive Bot). */
        void installiere() {
            ((AbstractDocument) spielernameneingabe.getDocument()).setDocumentFilter(new Filter());
            spielernameneingabe.getDocument().addDocumentListener(new DocumentListener() {

                @Override
                public void changedUpdate(DocumentEvent event) {
                    absprache.aktualisiere();
                }

                @Override
                public void insertUpdate(DocumentEvent event) {
                    absprache.aktualisiere();
                }

                @Override
                public void removeUpdate(DocumentEvent event) {
                    absprache.aktualisiere();
                }
            });
            botnamenauswahl.addItemListener(event -> {
                if (event.getStateChange() == ItemEvent.SELECTED) {
                    absprache.entferne(Nominierung.this, event.getItem().toString());
                } else {
                    absprache.erweitere(Nominierung.this, event.getItem().toString());
                }
            });
        }

        /** Gibt an, ob ein Bot die Rolle des Spielers bernehmen soll. */
        boolean kuenstlich() {
            return kuenstlich;
        }

        /**
         * bernimmt angegebenen Werte in die Eingabemasken.
         *
         * @param verwaltungsname
         *            Spielername, eventuell mit vorangestellter Kennzeichnung fr Bot oder Vertreter; jeweils als
         *            interner, nicht angezeigter Verwaltungsname.
         * @param botnamen
         *            Die Liste aller zurzeit freien Botnamen. Aus dieser Liste wird gegebenenfalls ein bergebener
         *            Botname entfernt.
         */
        void uebernehme(String verwaltungsname, List<String> botnamen) {
            CardLayout layout = (CardLayout) getLayout();
            kuenstlich = Name.kuenstlich(verwaltungsname);
            if (kuenstlich) {
                botnamenauswahl.addItem(Name.anzeigename(verwaltungsname));
                for (String botname : Bot.BOTNAMEN) {
                    if (!botnamen.contains(botname)) {
                        botnamenauswahl.addItem(botname);
                    }
                }
                layout.show(this, BOT);
            } else {
                spielernameneingabe.setText(Name.anzeigename(verwaltungsname));
                layout.show(this, MENSCH);
            }
        }
    }

    /**
     * Verwaltet die Herstellung aller wechselseitigen Verbindungen zwischen den Austragungsorten mit Bercksichtigung
     * der Darstellungen der Elemente der Bedienoberflche.
     */
    private class Rezeptionist extends SwingWorker<Boolean, Void> {

        /** Gibt an, ob der hiesige Austragungsort den Vorsitz fr die Turnierorganisation bernommen hat. */
        private final boolean vorsitzend;
        /** Stornier, fr eine Stopp bentigt. */
        private Stornierer stornierer;

        /** Bringt die Oberflchenelemente in den passenden Zustand und den Beobachter im Hintergrund. */
        Rezeptionist() {
            vorsitzend = leitung.vorsitzend();
            waehleAbstimmungssicht(HINWEIS);
            addPropertyChangeListener(ereignis -> {
                switch ((SwingWorker.StateValue) ereignis.getNewValue()) {
                case DONE:
                    uebertrageNamen(leitung.gesamtnamen(), leitung.vorortnamen());
                    break;
                case PENDING:
                case STARTED:
                }
            });
        }

        @Override
        protected Boolean doInBackground() throws InterruptedException {
            return leitung.warte();
        }

        @Override
        protected void done() {
            try {
                setEnabled(true);
                if (get()) {
                    setVisible(vorsitzend);
                    start.setEnabled(true);
                    return;
                }
            } catch (CancellationException ausnahme) {
                // Nicht alle Verbindungen innerhalb der Frist zustande gekommen
            } catch (ExecutionException | InterruptedException ausnahme) {
                ausnahme.printStackTrace();
            }
            stornierer.stoppe();
            stornierer = null;
            SwingUtilities.invokeLater(() -> {
                JOptionPane.showMessageDialog(Absprache.this,
                        "Kein Turnier ber Netzwerk zustande gekommen!\n"
                                + "Verbindungen zwischen den Austragungsorten konnten nicht hergestellt werden.\n"
                                + "Bitte erneut von Beginn an versuchen.",
                        "JaFuffy (Fehler)", JOptionPane.ERROR_MESSAGE);
                storniere();
            });
        }

        /**
         * Stellt Rckbezug zum Stornierer her.
         *
         * @param stornierer
         *            Auf diesen Stornierer wird Bezug genommen.
         */
        void verknuepfe(Stornierer stornierer) {
            this.stornierer = stornierer;
        }

    }

    /** Erlaubt den vorzeitigen Ausstieg aus Vorbereitungen fr ein Turnier ber ein Netzwerk. */
    private abstract class Stornierer extends ProgressMonitor {

        /** Die Wartezeit in Sekunden, nach der automatisch ein Ausstieg initiiert wird. */
        private static final int FRISTZEIT = 30;
        /** Zeit in Millisekunden bis zur Erinnerung an die Festlegung von Turnierbedingungen. */
        private static final int ERINNERUNGSZEIT = 10000;
        /** Die Periode in Millisekunden, mit der die Anzeige zum Wartezustand aktualisiert wird. */
        private static final int AKTUALISIERUNGSPERIODE = 500;
        /** Der Maximalwert fr die Fortschrittsanzeige. */
        private static final int MAXIMUM = FRISTZEIT * 1000 / AKTUALISIERUNGSPERIODE;

        /** Die verstrichene Zeit in Sekunden. */
        private volatile int zeit;

        /** Mit diesem Timer wird die Anzeige zum Wartezustand aktualisiert. */
        private final Timer fortschrittsaktualisierung = new Timer(AKTUALISIERUNGSPERIODE, ereignis -> {
            zeit++;
            if (leitung == null
                    || (abgelaufen() || isCanceled() || leitung.phase() == Phase.BETRIEB) && ueberpruefe()) {
                entferne();
            } else {
                setProgress(zeit);
            }
        });
        /** Beboachter fr den Abbruch-Button. */
        private final ActionListener abbruchbeobachter = ereignis -> {
            entferne();
            deinstalliere();
            if (leitung != null) {
                leitung.stoppe();
            }
            leitung = null;
        };
        /** Beobachter fr den OK-Button. */
        private final ActionListener okbeobachter = ereignis -> {
            entferne();
            deinstalliere();
        };

        /** Meldung und Stornierung nach Fristablauf. */
        protected final Runnable fristablauf;

        /**
         * Fhrt die Basiskonstruktionen durch.
         *
         * @param heimatkomponente
         *            Die Komponente,innerhalb derer die Fortschrittsanzeige platziert wwird.
         * @param information
         *            Der erluternde Text.
         * @param frist
         *            Die Dauer der Frist in Sekunden.
         * @param erinnerung
         *            Die Zeit in Sekunden, nach der die Erinnerung erscheint.
         */
        private Stornierer(Component heimatkomponente, Object information, int frist, int erinnerung) {
            super(heimatkomponente, information, null, 0, frist);
            fristablauf = () -> {
                JOptionPane.showMessageDialog(heimatkomponente,
                        "Frist abgelaufen,\nkein Turnier ber Netzwerk zustande gekommen.", "JaFuffy (Fristablauf)",
                        JOptionPane.WARNING_MESSAGE);
                storniere();
            };
            setMillisToDecideToPopup(erinnerung);
            installiere();
        }

        /** Erinnert den Benutzer an die Festlegung der Turnierbedingungen. */
        Stornierer() {
            this(Absprache.this, "Frist zur Festlegung der Turnierbedingungen luft...\n"
                    + "Abbruch enfternt diese Erinnerung, Frist luft weiter.", MAXIMUM, ERINNERUNGSZEIT);
        }

        /**
         * Sorgt fr die Mglichkeit des Ausstiegs durch den Benutzer, falls die Vorbereitungen fr ein Turnier ber ein
         * Netzwerk fehlschlagen.
         *
         * @param vorsitz
         *            Die textuelle Beschreibung des Austragungsortes, welcher den Vorsitz bernommen hat.
         */
        Stornierer(String vorsitz) {
            this(getOwner(), "Warten auf die Festlegung der Turnierbedingungen durch\n" + "\"" + vorsitz + "\".\n"
                    + "Abbruch beseitigt alle Turniervorbereitungen.", MAXIMUM, 0);
        }

        /** Deinstalliert die Beobachter fr Bedienelemente. */
        private void deinstalliere() {
            abbruch.removeActionListener(abbruchbeobachter);
            start.removeActionListener(okbeobachter);
        }

        /** Entfernt den Stornierer ohne weitere Aktionen zu ttigen. */
        private void entferne() {
            fortschrittsaktualisierung.stop();
            close();
        }

        /** Installiert die Beobachter fr Bedienelemente. */
        private void installiere() {
            abbruch.addActionListener(abbruchbeobachter);
            start.addActionListener(okbeobachter);
            addWindowListener(new WindowAdapter() {
                @Override
                public void windowClosing(WindowEvent ereignis) {
                    removeWindowListener(this);
                    entferne();
                    deinstalliere();
                    storniere();
                }
            });
        }

        /** Gibt an, ob die Frist des Stornierers abgelaufen ist. */
        boolean abgelaufen() {
            return zeit >= getMaximum();
        }

        /** Startet den Stornierer. */
        void starte() {
            fortschrittsaktualisierung.start();
        }

        /** Stoppt den Stornierer. */
        void stoppe() {
            fortschrittsaktualisierung.stop();
            close();
        }

        /**
         * Fhrt die berprfungsaktionen durch, welche im Falle eines Ablauf des Stornierers oder eines Abbruchs durch
         * den Benutzer zu tun sind.
         *
         * @return Gibt an, ob der Stornierer weiterhin aktiv bleiben soll.
         */
        abstract boolean ueberpruefe();

    }

    /** Auswahlicon fr aktive Bots. */
    private static final ImageIcon AKTIVBOT = Grafik.icon("Aktivbot.png");
    /** Auswahlicon fr ruhenden Bot. */
    private static final ImageIcon RUHEBOT = Grafik.icon("Ruhebot.png");
    /** Start-Icon. */
    private static final ImageIcon START = Grafik.icon("Start.png");
    /** Abbruch-Icon. */
    private static final ImageIcon ABBRUCH = Grafik.icon("Abbruch.png");
    /** Netzwerk-Icon. */
    private static final ImageIcon NETZWERK = Grafik.icon("Netzwerk.png");
    /** Abstand vom Feldrahmen zum Inhalt. */
    private static final int FELDRAHMENABSTAND = 4;
    /** Abstand vom Dialograhmen zum Inhalt. */
    private static final int DIALOGRAHMENABSTAND = 8;
    /** Mindestgre fr Anzahlauswahl. */
    private static final int AUSWAHLLAENGE = 240;
    /** Abstand Feldrahmen zu Buttons. */
    private static final int BUTTONABSTAND = 2 * FELDRAHMENABSTAND;
    /** Abstand der Feinticks (Spielanzahlauswahl). */
    private static final int FEINTICKS = 1;
    /** Abstand der Orientierungsticks (Spielanzahlauswahl). */
    private static final int ORIENTIERUNGSTICKS = 5;
    /** Anzahl der Spalten im Texteingabefeld. */
    private static final int SPALTEN = 8;
    /** Anzahl der Tick-Labels (Spielanzahlauswahl). */
    private static final int TICKLABELS = 5;
    /** Die Karte bzw. Sicht zur Abstimmung ber ein Netzwerk. */
    private static final String ABSTIMMUNG = "Abstimmung";
    /** Die Karte bzw. Sicht zum Hinweis zur Abstimmung ber ein Netzwerk. */
    private static final String HINWEIS = "Hinweis";

    /** Maximale Anzahl der Spiele pro Turnier. */
    public static final int ANZAHL = 25;

    /** Einstellung der Spieleanzahl pro Turnier per Schieberegler. */
    private final JSlider anzahlschieberegler = new JSlider(0, ANZAHL);
    /** Beginnmodusauswahl. */
    private final JComboBox<Beginner> beginnmodusauswahl = new JComboBox<>(Beginner.values());
    /** Programm- und Turniereigenschaften. */
    private final Eigenschaften eigenschaften;
    /** Regelvariantenauswahl. */
    private final JComboBox<Variante> variantenauswahl = new JComboBox<>(Variante.values());
    /** Nominierungen der Mitspieler, sowohl menschliche Spieler und knstliche Intelligenzen. */
    private final Nominierung[] nominierungen = new Nominierung[Spieler.SPIELER];
    /** Auswahlboxen fr knstliche Intelligenzen. */
    private final Botselektion[] botselektionen = new Botselektion[Spieler.SPIELER];
    /** Organisation der Durchfhrung eines Turniers ber ein Netzwerk. */
    private final JButton abstimmung = new JButton("Abstimmung ber Netzwerk", NETZWERK);
    /** Das Wechselfeld fr die Abstimmung der Teilnehmer ber ein Netzwerk. */
    private final JPanel abstimmungsfeld = new JPanel(new CardLayout());
    /** Start-Button. */
    private final JButton start = new JButton("Start");
    /** Abbruch-Button. */
    private final JButton abbruch = new JButton("Abbruch");

    /** Die Leitung des Turnierbetriebs ber ein Netzwerk, so wie mit diesem Dialog verbunden. */
    private Leitung leitung;

    /**
     * Konstruktor.
     *
     * @param eigenschaften
     *            Programm- und Turniereigenschaften.
     * @param manager
     *            Der betreuende Manager, welcher sich um das Turnier kmmert, welches aus der Absprache entsteht.
     */
    public Absprache(Eigenschaften eigenschaften, Fenster fenster, Manager manager) {
        super(fenster);
        this.eigenschaften = eigenschaften;
        UIManager.addPropertyChangeListener(this);
        setTitle("JaFuffy (Absprache fr neues Turnier)");
        setIconImages(Oberflaeche.LOGOS);
        setModal(true);
        baue();
        initialisiere();
        installiere(manager);
        aktualisiere();
        pack();
    }

    @Override
    public void propertyChange(PropertyChangeEvent event) {
        if (event.getPropertyName().equals("Beginnen")) {
            setLocationRelativeTo(getOwner());
            setVisible(true);
        } else if (event.getPropertyName().equals("lookAndFeel")) {
            SwingUtilities.updateComponentTreeUI(this);
            pack();
        }
    }

    /** Verteilt die Namen der knstlichen Intelligenzen und passt Elemente des Dialogs an den aktuellen Zustand an. */
    private void aktualisiere() {
        int spieleranzahl = 0;
        int botanzahl = 0;
        for (Nominierung nominierung : nominierungen) {
            if (!nominierung.toString().isEmpty()) {
                spieleranzahl++;
                if (nominierung.kuenstlich()) {
                    botanzahl++;
                }
            }
        }
        boolean mehrfach = false;
        for (int i = 0; i < nominierungen.length; i++) {
            for (int j = i + 1; j < nominierungen.length; j++) {
                mehrfach = mehrfach || !nominierungen[i].toString().isEmpty() && !nominierungen[i].toString().isEmpty()
                        && Objects.equals(nominierungen[i].toString(), nominierungen[j].toString());
            }
        }
        beginnmodusauswahl.setEnabled(spieleranzahl > 1 && (leitung == null || leitung.vorsitzend()));
        start.setEnabled(spieleranzahl > 0 && !mehrfach && (leitung == null || leitung.phase() == Phase.BEREIT));
        abstimmung.setEnabled(spieleranzahl > botanzahl);
    }

    /**
     * Anzahl der Spiele pro Turnier.
     *
     * @return Anzahlauswahlfeld.
     */
    private JPanel anzahlauswahl() {
        JLabel label;
        label = new JLabel("Anzahl der Spiele im Turnier");
        label.setAlignmentX(Component.CENTER_ALIGNMENT);
        anzahlschieberegler.setMinorTickSpacing(FEINTICKS);
        anzahlschieberegler.setMajorTickSpacing(ORIENTIERUNGSTICKS);
        anzahlschieberegler.setPaintTicks(true);
        anzahlschieberegler.setPaintLabels(true);
        anzahlschieberegler.setSnapToTicks(true);
        @SuppressWarnings("unchecked")
        Hashtable<Integer, JLabel> hashtable = anzahlschieberegler.createStandardLabels(TICKLABELS);
        hashtable.put(0, new JLabel("beliebig"));
        anzahlschieberegler.setLabelTable(hashtable);
        JPanel anzahlfeld = new JPanel();
        anzahlfeld.setLayout(new BoxLayout(anzahlfeld, BoxLayout.Y_AXIS));
        anzahlfeld.add(label);
        anzahlfeld.add(anzahlschieberegler);
        anzahlfeld.add(Box.createHorizontalStrut(AUSWAHLLAENGE));
        return anzahlfeld;
    }

    /** Aufbau des Gesamtdialogs. */
    private void baue() {
        JPanel regelnfeld = new JPanel(new BorderLayout());
        regelnfeld.add(regelnfeld(), BorderLayout.CENTER);
        JPanel buttonspalte = new JPanel(new BorderLayout());
        buttonspalte.add(buttonspalte(), BorderLayout.CENTER);
        JPanel feld = new JPanel();
        feld.setBorder(
                new EmptyBorder(DIALOGRAHMENABSTAND, DIALOGRAHMENABSTAND, DIALOGRAHMENABSTAND, DIALOGRAHMENABSTAND));
        feld.setLayout(new BoxLayout(feld, BoxLayout.X_AXIS));
        feld.add(nominierungsfeld());
        feld.add(Box.createHorizontalStrut(FELDRAHMENABSTAND));
        feld.add(regelnfeld);
        feld.add(Box.createHorizontalStrut(FELDRAHMENABSTAND));
        feld.add(buttonspalte);
        setContentPane(feld);
    }

    /**
     * OK-Abbruch-Leiste.
     *
     * @return Buttonspalte.
     */
    private JPanel buttonspalte() {
        abbruch.setIcon(ABBRUCH);
        abbruch.setHorizontalTextPosition(SwingConstants.CENTER);
        abbruch.setVerticalTextPosition(SwingConstants.BOTTOM);
        start.setToolTipText("<html>" + "<p>Beginnt ein neues Turnier.</p>"
                + "<p>Mindestens ein Spielername ist ntig und alle Spielernamen mssen verschieden sein.</p>"
                + "<p>Beim Netzwerkbetrieb mssen sich alle Austragungsorte miteinander verbunden haben.</p>"
                + "</html>");
        start.setIcon(START);
        start.setHorizontalTextPosition(SwingConstants.CENTER);
        start.setVerticalTextPosition(SwingConstants.BOTTOM);
        JPanel buttonspalte = new JPanel(new GridLayout(0, 1, 0, BUTTONABSTAND));
        buttonspalte.setBorder(BorderFactory.createEmptyBorder(2 * BUTTONABSTAND + FELDRAHMENABSTAND, FELDRAHMENABSTAND,
                2 * BUTTONABSTAND, FELDRAHMENABSTAND));
        buttonspalte.add(start);
        buttonspalte.add(abbruch);
        return buttonspalte;
    }

    /**
     * Entfernt einen knftig bentigten Namen eines Verursachers von den Namensauswahlen der anderen Bots.
     *
     * @param verursacher
     *            Der verursachende Bot.
     * @param botname
     *            Der zu entfernende Name.
     */
    private void entferne(Nominierung verursacher, String botname) {
        for (Nominierung nominierung : nominierungen) {
            if (nominierung != verursacher && nominierung.kuenstlich()) {
                nominierung.entferne(botname);
            }
        }
    }

    /**
     * Bestimmt den ersten Spieler, der das Turnier erffnet.
     *
     * @param namen
     *            Die Namen aller teilnehmenden Spieler.
     * @return Der Index des Spielers, welcher das Turnier erffnet.
     */
    private int eroeffnungsindex(ArrayList<Name> namen) {
        int index = 0;
        if (namen.size() > 1) {
            Eroeffnung eroeffnung = new Eroeffnung(namen);
            eroeffnung.addPropertyChangeListener(this);
            eroeffnung.setLocationRelativeTo(this);
            eroeffnung.setVisible(true);
            eroeffnung.dispose();
            index = eroeffnung.eroeffnungsindex();
        }
        return index;
    }

    /**
     * Fgt einen freiwerdenden Namen des Verursachers zur Namensauswahl anderer Bots hinzu.
     *
     * @param verursacher
     *            Der verursachende Bot.
     * @param botname
     *            Der hinzuzufgende Name.
     */
    private void erweitere(Nominierung verursacher, String botname) {
        for (Nominierung nominierung : nominierungen) {
            if (nominierung != verursacher && nominierung.kuenstlich()) {
                nominierung.erweitere(botname);
            }
        }
    }

    /**
     * Ermittelt, ob der gewnschte Botname noch zur Verfgung steht.
     *
     * @param botname
     *            Zu berprfender Name.
     * @return Gibt an, ob der Botname noch zur Verfgung steht.
     */
    private boolean frei(String botname) {
        for (Nominierung nominierung : nominierungen) {
            if (botname.equals(nominierung.toString())) {
                return false;
            }
        }
        return true;
    }

    /** Initialisiert alle Oberflchenelemente mit den Werten, so wie sie in den Eigenschaften vorgefunden werden. */
    private void initialisiere() {
        ArrayList<String> botnamen = new ArrayList<>();
        for (int i = 0; i < Spieler.SPIELER; i++) {
            String verwaltungsname = eigenschaften.verwaltungsname(i);
            if (Name.kuenstlich(verwaltungsname)) {
                botnamen.add(Name.anzeigename(verwaltungsname));
            }
        }
        for (int i = 0; i < Spieler.SPIELER; i++) {
            nominierungen[i].uebernehme(eigenschaften.verwaltungsname(i), botnamen);
            botselektionen[i].setSelected(nominierungen[i].kuenstlich());
        }
        anzahlschieberegler.setValue(Integer.parseInt(eigenschaften.getProperty("Anzahl")));
        beginnmodusauswahl.setSelectedIndex(eigenschaften.beginnmodus());
        variantenauswahl.setSelectedIndex(eigenschaften.variante());
        leitung = null;
    }

    /**
     * Installiert die Reaktionen der Buttonleiste auf Abbruch und OK, sowie die Abstimmung zur Netzwerkkontrolle.
     *
     * @param manager
     *            Der Manager, welcher das abgesprochenen Turnier betreiben soll.
     */
    private void installiere(Manager manager) {
        addWindowListener(new WindowAdapter() {
            @Override
            public void windowClosing(WindowEvent ereignis) {
                abbruch.doClick();
            }
        });
        for (Nominierung nominierung : nominierungen) {
            nominierung.installiere();
        }
        abstimmung.addActionListener(ereignis -> {
            try {
                organisiere(manager);
            } catch (UnknownHostException | SocketException ausnahme) {
                ausnahme.printStackTrace();
                JOptionPane.showMessageDialog(Absprache.this,
                        "Kein Turnier ber Netzwerk mglich!\nKein Netzwerk verfgbar?\nJava-Exception:\n" + ausnahme,
                        "JaFuffy (Fehler)", JOptionPane.ERROR_MESSAGE);
            }
        });
        abstimmung.setToolTipText("<html>" + "<p>Organisiert ein Turnier ber ein lokales Netzwerk.</p>"
                + "<p>Erlaubt Hinzufgen von Mitspielern, die ber das Netzwerk teilnehmen.</p>"
                + "<p>Bots sind von der Teilnahme ausgeschlossen.</p>"
                + "<br><p><small><em>Experimentell! Volle Funktionalitt steht nicht zur Verfgung.</em></small></p>"
                + "</html>");
        abbruch.addActionListener(ereignis -> {
            initialisiere();
            aktualisiere();
            setVisible(false);
            waehleAbstimmungssicht(ABSTIMMUNG);
        });
        start.addActionListener(ereignis -> {
            if (leitung == null) {
                manager.betreibe(plan());
            } else {
                leitung.betreibe(manager, plan());
                leitung = null;
            }
            setVisible(false);
            waehleAbstimmungssicht(ABSTIMMUNG);
        });
        getRootPane().setDefaultButton(start);
    }

    /** Erstellt das Feld zum Zugang zur Netzwerkkontrolle fr ein Turnier, das ber ein Netzwerk gespielt wird. */
    private JPanel konfiguriereAbstimmungsfeld() {
        abstimmungsfeld.setBorder(BorderFactory.createEmptyBorder(BUTTONABSTAND, 0, FELDRAHMENABSTAND / 2, 0));
        abstimmungsfeld.add(abstimmung, ABSTIMMUNG);
        JLabel hinweis = new JLabel("Teilnehmer wie eben zuvor abgestimmt");
        hinweis.setEnabled(false);
        hinweis.setHorizontalAlignment(SwingConstants.CENTER);
        abstimmungsfeld.add(hinweis, HINWEIS);
        return abstimmungsfeld;
    }

    /** Baut das Feld fr die Auswahl des Beginnmodus auf. */
    private void konfiguriereBeginnmodusauswahl() {
        beginnmodusauswahl.setAlignmentX(Component.CENTER_ALIGNMENT);
        beginnmodusauswahl.setToolTipText("Auswahl, welcher Mitspieler jedes neue Spiel beginnt");
    }

    /** Baut das Feld zur Variantenauswahl auf. */
    private void konfiguriereVariantenauswahl() {
        variantenauswahl.setActionCommand("Var");
        variantenauswahl.setAlignmentX(Component.CENTER_ALIGNMENT);
        variantenauswahl.setToolTipText("Nach welchen Spielregeln soll gespielt werden?");
    }

    /**
     * Baut das Nominierungsfeld zusammen, in dem Spieler menschlicher oder knstlicher Natur angemeldet werden.
     *
     * @return Nominierungsfeld.
     */
    private JPanel nominierungsfeld() {
        JPanel nominierungsfeld = new JPanel(new BorderLayout());
        nominierungsfeld.setBorder(BorderFactory.createCompoundBorder(BorderFactory.createTitledBorder("Namen"),
                BorderFactory.createEmptyBorder(FELDRAHMENABSTAND / 2, FELDRAHMENABSTAND, FELDRAHMENABSTAND / 2,
                        FELDRAHMENABSTAND)));
        nominierungsfeld.add(spielerfestlegung(), BorderLayout.PAGE_START);
        nominierungsfeld.add(konfiguriereAbstimmungsfeld(), BorderLayout.CENTER);
        return nominierungsfeld;
    }

    /**
     * Steuert die grafischen Oberflchenelemente, welche zur Organisation eines Turniers ber ein Netzwerk ntig sind,
     * und bereitet die Infrastruktur zur Organisation vor. Herstellung der wechselseitigen Verbindungen zwischen den
     * Austragungsorten, Ermittlung und Verteilung des Turnierplans zur Erzeugung eines neuen Turniers.
     *
     * @param manager
     *            Fr diesen Manager erfolgt die Organisation eines Turniers ber ein Netzwerk.
     * @throws UnknownHostException
     *             Abstimmung ber Netzwerk nicht verfgbar.
     * @throws SocketException
     *             Abstimmung ber Netzwerk nicht verfgbar.
     */
    private void organisiere(Manager manager) throws UnknownHostException, SocketException {
        LinkedHashSet<String> vorortnamen = vorortnamen();
        leitung = new Leitung(() -> {
            waehleAbstimmungssicht(ABSTIMMUNG);
            uebernehmeTeilnehmer(vorortnamen);
            leitung = null;
        }, manager, vorortnamen);
        abstimmung.addActionListener(leitung);
        start.setEnabled(false);
        setEnabled(false);
        Organisation organisation = new Organisation(leitung, vorortnamen.size());
        organisation.setLocationRelativeTo(this);
        organisation.setVisible(true);
        abstimmung.removeActionListener(leitung);
        if (leitung.phase() == Phase.ABBRUCH) {
            leitung = null;
            setEnabled(true);
            start.setEnabled(true);
            return;
        }
        final Rezeptionist rezeptionist = new Rezeptionist();
        final Stornierer stornierer = stornierer(rezeptionist);
        rezeptionist.verknuepfe(stornierer);
        stornierer.starte();
        rezeptionist.execute();
    }

    /**
     * Erstellt einen Plan, nach dem ein Turnier betrieben werden kann.
     *
     * @return Der erstellte Plan.
     */
    private Plan plan() {
        final Plan plan;
        if (leitung == null || leitung.phase() == Phase.BETRIEB) {
            ArrayList<Name> teilnehmer = uebernehmeTeilnehmer();
            plan = new Plan(teilnehmer, uebernehmeBeginnmodus(), uebernehmeVariante(), uebernehmeAnzahl(),
                    eroeffnungsindex(teilnehmer));
        } else if (leitung.vorsitzend()) {
            uebernehmeTeilnehmer(leitung.vorortnamen());
            ArrayList<Name> teilnehmer = uebernehmeTeilnehmer(leitung.gesamtnamen(), leitung.vorortnamen());
            Plan hilfsplan = new Plan(teilnehmer, uebernehmeBeginnmodus(), uebernehmeVariante(), uebernehmeAnzahl(),
                    eroeffnungsindex(teilnehmer));
            try {
                leitung.informiere(hilfsplan);
            } catch (Exception ausnahme) {
                ausnahme.printStackTrace();
                JOptionPane.showMessageDialog(Absprache.this,
                        "Strung im Turnierbetrieb ber Netzwerk!\n" + "Turnier muss neu gestartet werden.\n"
                                + "\nJava-Exception:\n" + ausnahme.getLocalizedMessage(),
                        "JaFuffy (Fehler)", JOptionPane.ERROR_MESSAGE);
                storniere();
                hilfsplan = null;
            }
            plan = hilfsplan;
        } else {
            plan = null; // Bei Erreichen dieser Stelle liegt ein Programmierfehler vor wegen eines Logikfehlers
        }
        return plan;
    }

    /**
     * Erstellt das Feld, in dem die Spielregeln festgelegt werden.
     *
     * @return Regelnfeld.
     */
    private JPanel regelnfeld() {
        konfiguriereVariantenauswahl();
        konfiguriereBeginnmodusauswahl();
        JPanel regelnfeld = new JPanel();
        regelnfeld.setLayout(new BoxLayout(regelnfeld, BoxLayout.Y_AXIS));
        regelnfeld.setBorder(BorderFactory.createCompoundBorder(BorderFactory.createTitledBorder("Bedingungen"),
                BorderFactory.createEmptyBorder(FELDRAHMENABSTAND / 2, FELDRAHMENABSTAND, FELDRAHMENABSTAND / 2,
                        FELDRAHMENABSTAND)));
        regelnfeld.add(beginnmodusauswahl);
        regelnfeld.add(Box.createVerticalStrut(1));
        regelnfeld.add(Box.createVerticalGlue());
        regelnfeld.add(Box.createVerticalStrut(1));
        regelnfeld.add(variantenauswahl);
        regelnfeld.add(Box.createVerticalStrut(FELDRAHMENABSTAND));
        regelnfeld.add(Box.createVerticalGlue());
        regelnfeld.add(Box.createVerticalStrut(FELDRAHMENABSTAND));
        regelnfeld.add(anzahlauswahl());
        regelnfeld.add(Box.createVerticalGlue());
        return regelnfeld;
    }

    private JPanel spielerfestlegung() {
        JPanel spielerfestlegung = new JPanel(new GridBagLayout());
        GridBagConstraints constraints = new GridBagConstraints();
        constraints.gridwidth = 1;
        constraints.weighty = 1;
        constraints.insets = new Insets(1, 0, 1, 0);
        for (int i = 0; i < Spieler.SPIELER; i++) {
            constraints.anchor = GridBagConstraints.LINE_START;
            constraints.gridx = 0;
            constraints.gridy = i;
            constraints.ipadx = 2 * FELDRAHMENABSTAND;
            constraints.fill = GridBagConstraints.VERTICAL;
            constraints.weightx = 0;
            nominierungen[i] = new Nominierung(this);
            variantenauswahl.addActionListener(nominierungen[i]);
            botselektionen[i] = new Botselektion(nominierungen[i], variantenauswahl);
            variantenauswahl.addActionListener(botselektionen[i]);
            spielerfestlegung.add(botselektionen[i], constraints);
            constraints.anchor = GridBagConstraints.CENTER;
            constraints.gridx = 1;
            constraints.ipadx = FELDRAHMENABSTAND;
            constraints.weightx = 0;
            JLabel kopf = new JLabel(i + 1 + ". Spieler: ", SwingConstants.CENTER);
            spielerfestlegung.add(kopf, constraints);
            constraints.anchor = GridBagConstraints.LINE_END;
            constraints.gridx = 2;
            constraints.ipadx = 0;
            constraints.fill = GridBagConstraints.BOTH;
            constraints.weightx = 1;
            spielerfestlegung.add(nominierungen[i], constraints);
        }
        return spielerfestlegung;
    }

    /** Storniert die Vorbereitungen fr ein Turnier ber ein Netzwerk. */
    private void storniere() {
        abbruch.doClick();

    }

    /**
     * Stellt einen Stornierer zur Verfgung.
     *
     * @param rezeptionist
     *            Der Rezepitionist, welcher von diesem Stornierer storniert werden kann.
     * @return Stornierer
     */
    private Stornierer stornierer(Rezeptionist rezeptionist) {
        if (leitung.vorsitzend()) {
            return new Stornierer() {

                @Override
                boolean ueberpruefe() {
                    if (abgelaufen() && leitung.phase() != Phase.BETRIEB) {
                        SwingUtilities.invokeLater(fristablauf);
                    }
                    return abgelaufen();
                }
            };
        } else {
            return new Stornierer(leitung.vorsitz()) {
                @Override
                boolean ueberpruefe() {
                    if (leitung.phase() == Phase.VERBINDUNG) {
                        rezeptionist.cancel(true);
                    } else if (leitung.phase() != Phase.BETRIEB) {
                        if (isCanceled()) {
                            storniere();
                        } else {
                            SwingUtilities.invokeLater(fristablauf);
                        }
                    }
                    return true;
                }
            };
        }
    }

    /**
     * Bestimmt und bernimmt die Anzahl.
     *
     * @return Anzahl der Spiele pro Turnier.
     */
    private int uebernehmeAnzahl() {
        eigenschaften.setProperty("Anzahl", Integer.toString(anzahlschieberegler.getValue()));
        return anzahlschieberegler.getValue();
    }

    /**
     * Bestimmt und bernimmt den Beginnmodus.
     *
     * @return Beginnmodus.
     */
    private Beginner uebernehmeBeginnmodus() {
        Beginner beginnmodus = Beginner.values()[beginnmodusauswahl.getSelectedIndex()];
        eigenschaften.setProperty("Beginnmodus", beginnmodus.name());
        return beginnmodus;
    }

    /**
     * Bestimmt und bernimmt die teilnehmenden Spieler bei gleichzeitiger Speicherung in den Eigenschaften.
     *
     * @return Die Liste der Basisnamen aller an dem Turnier teilnehmenden Spieler.
     */
    private ArrayList<Name> uebernehmeTeilnehmer() {
        ArrayList<Name> namen = new ArrayList<>(Spieler.SPIELER);
        for (int index = 0; index < Spieler.SPIELER; index++) {
            eigenschaften.setProperty("Name" + index, "");
        }
        for (Nominierung nominierung : nominierungen) {
            String anzeigename = nominierung.toString();
            if (!anzeigename.isEmpty()) {
                final Name.Typus typus = nominierung.kuenstlich() ? Name.Typus.BOT : Name.Typus.SPIELER;
                final Name name = new Name(anzeigename, typus, namen.size());
                eigenschaften.setProperty("Name" + namen.size(), name.verwaltungsname());
                namen.add(name);
            }
        }
        return namen;
    }

    /**
     * bernimmt die Spieler vor Ort aus der Abstimmung fr ein Turnier ber ein Netzwerk. Die bernahme erfolgt in die
     * Eigenschaften sowie in in den Dialog der Absprache.
     *
     * @param vorortnamen
     *            Die Liste aller Namen vor Ort.
     */
    private void uebernehmeTeilnehmer(LinkedHashSet<String> vorortnamen) {
        Iterator<String> iterator = vorortnamen.iterator();
        for (int index = 0; index < Spieler.SPIELER; index++) {
            String anzeigename;
            String verwaltungsname;
            if (iterator.hasNext()) {
                anzeigename = iterator.next();
                verwaltungsname = Name.verwaltungsname(anzeigename, Typus.SPIELER);
            } else {
                anzeigename = "";
                verwaltungsname = "";
            }
            nominierungen[index].uebernehme(verwaltungsname, null);
            eigenschaften.setProperty("Name" + index, verwaltungsname);
        }
    }

    /**
     * Bestimmt die vollstndige Namensliste aus der Abstimmung fr ein Turnier ber ein Netzwerk.
     *
     * @param gesamtnamen
     *            Die Liste der Namen, so wie sie aus der Abstimmung fr das Turnier ber ein Netzwerk bestimmt wurde.
     * @param vorortnamen
     *            Die Liste aller Namen vor Ort.
     * @return Alle teilnehmenden Spieler.
     */
    private ArrayList<Name> uebernehmeTeilnehmer(LinkedHashSet<String> gesamtnamen, LinkedHashSet<String> vorortnamen) {
        ArrayList<Name> namen = new ArrayList<>(Spieler.SPIELER);
        for (String anzeigename : gesamtnamen) {
            final Typus typus;
            if (vorortnamen.contains(anzeigename)) {
                typus = Typus.SPIELER;
            } else {
                typus = Typus.VERTRETER;
            }
            namen.add(new Name(anzeigename, typus, namen.size()));
        }
        return namen;
    }

    /**
     * Bestimmt und bernimmt die Regelvariante.
     *
     * @return Regelvariante.
     */
    private Variante uebernehmeVariante() {
        Variante variante = (Variante) variantenauswahl.getSelectedItem();
        eigenschaften.setProperty("Variante", variante.name());
        return variante;
    }

    /**
     * bertrgt die Namens aus der Turniervorbereitung der Leitung fr ein Turnier ber ein Netzwerk, und zwar in die
     * Dialogfelder der Nominierungen.
     *
     * @param gesamtnamen
     *            Die Liste aller Namen, welche an dem Turnier ber das Netzwerk teilnehmen.
     * @param vorortnamen
     *            Die Liste der Namen, welche an dem hiesigen Austragungsort teilnehmen.
     */
    private void uebertrageNamen(LinkedHashSet<String> gesamtnamen, LinkedHashSet<String> vorortnamen) {
        Iterator<String> iterator = gesamtnamen.iterator();
        for (Nominierung nominierung : nominierungen) {
            String name = "";
            if (iterator.hasNext()) {
                name = iterator.next();
                if (vorortnamen.contains(name)) {
                    name = Typus.SPIELER.praefix + name;
                } else {
                    name = Typus.VERTRETER.praefix + name;
                }
            }
            nominierung.uebernehme(name, null);
        }
    }

    /** @return Teilnehmer aus dem Absprachedialog. */
    private LinkedHashSet<String> vorortnamen() {
        LinkedHashSet<String> teilnehmer = new LinkedHashSet<>();
        for (Nominierung nominierung : nominierungen) {
            String name = nominierung.toString();
            if (!name.isEmpty() && !nominierung.kuenstlich()) {
                teilnehmer.add(name);
            }
        }
        return teilnehmer;
    }

    /**
     * Passt den Absprachedialog fr die Spielernamen an die gewhlte Karte fr die Abstimmung ber ein Netzwerk an.
     *
     * @param sicht
     *            Die gewhlte Sicht fr die Abstimmung ber ein Netzwerk.
     */
    private void waehleAbstimmungssicht(String sicht) {
        ((CardLayout) abstimmungsfeld.getLayout()).show(abstimmungsfeld, sicht);
        final boolean aktiv = ABSTIMMUNG.equals(sicht);
        for (JCheckBox botselektor : botselektionen) {
            botselektor.setEnabled(aktiv);
        }
        for (Nominierung nominierung : nominierungen) {
            nominierung.aktiviere(aktiv);
        }
        start.setIcon(aktiv ? START : NETZWERK);
    }

}
