/******************************************************************************
 ** $Id: Verlaufsgrafik.java 1815 2019-07-15 20:57:34Z wmh $
 ** Diese Datei ist Bestandteil der Java-Quelltexte des Wrfelspiels JaFuffy.
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************
 ** Die aktuellste Version von JaFuffy findet sich im Internet unter
 ** <http://jafuffy.3kelvin.de>.
 **
 ** Kommentare, Fehler oder Erweiterungswnsche bitte per E-Mail senden an
 ** <jafuffy@3kelvin.de>.
 ******************************************************************************/
package jafuffy.bedienung;

import java.awt.BasicStroke;
import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Font;

import javax.swing.JPanel;

import com.orsoncharts.Chart3D;
import com.orsoncharts.Chart3DFactory;
import com.orsoncharts.Chart3DPanel;
import com.orsoncharts.Range;
import com.orsoncharts.axis.NumberAxis3D;
import com.orsoncharts.axis.StandardCategoryAxis3D;
import com.orsoncharts.data.DefaultKeyedValues;
import com.orsoncharts.data.category.StandardCategoryDataset3D;
import com.orsoncharts.graphics3d.Offset2D;
import com.orsoncharts.graphics3d.ViewPoint3D;
import com.orsoncharts.graphics3d.swing.DisplayPanel3D;
import com.orsoncharts.interaction.Chart3DMouseEvent;
import com.orsoncharts.interaction.Chart3DMouseListener;
import com.orsoncharts.label.StandardCategoryItemLabelGenerator;
import com.orsoncharts.plot.CategoryPlot3D;
import com.orsoncharts.renderer.category.BarRenderer3D;
import com.orsoncharts.renderer.category.StandardCategoryColorSource;
import com.orsoncharts.table.StandardRectanglePainter;

import jafuffy.logik.Spieler;
import jafuffy.logik.Statistik;

/** Anzeige des Turnierverlaufs in Form eines rumlichen Balkendiagramms. */
@SuppressWarnings("serial")
class Verlaufsgrafik extends JPanel {

    /** Bestimmt Balkenlnge in Richtung der Spielreihe. */
    private static final double ZLAENGE = 0.7;
    /** Bestimmt Balkenlnge in Richtung der Turnierabfolge. */
    private static final double XLAENGE = 0.5;
    /** Wertebereich fr die darzustellenden Punktzahlen. */
    private static final Range WERTEBEREICH = new Range(0, 350);
    /** Drehwinkel der Perspektive. */
    private static final double DREHWINKEL = -12 * Math.PI / 180;
    /** Rollwinkel der Perspektive. */
    private static final double ROLLWINKEL = 8 * Math.PI / 180;
    /** Rho der Perspektive. */
    private static final double RHO = 55;
    /** Verschiebung des Charts. */
    private static final Offset2D VERSCHIEBUNG = new Offset2D(0, -20);
    /** Projektionsdistanz. */
    private static final int PROJEKTIONSDISTANZ = 1000;
    /** Darstellung der Gitterlinien in den Koordinatenwnden. */
    private static final BasicStroke GITTERLINIENSTIL = new BasicStroke(0.25f);
    /** Farbe der Koordinatenwnde. */
    private static final Color WANDFARBE = new Color(176, 192, 192, 64);
    /** Beschriftungsfarbe fr Punktzahlen. */
    private static final Color ZAHLSCHRIFTFARBE = Color.GRAY;
    /** Beschriftungsfarbe fr Achsenbeschriftung. */
    private static final Color ACHSSCHRIFTFARBE = Color.LIGHT_GRAY;
    /** Beschriftungszeichensatz fr Punktzahlen. */
    private static final Font ZAHLSCHRIFTART = new Font(Font.SANS_SERIF, Font.PLAIN, 11);
    /** Punktzahlenzeichensatz fr Punktzahlen. */
    private static final Font ACHSSCHRIFTART = new Font(Font.SANS_SERIF, Font.BOLD, 11);
    /** Abstand der Grafik zum Rand. */
    private static final double RANDABSTAND = 0.17;
    /** Fr Platzhalterberechnung. */
    private static final int P = 250;
    /** Fr Platzhalterberechnung. */
    private static final int N = 5;
    /** Fr Platzhalterberechnung. */
    private static final int S = 3;

    /** Bereitet Diagramm zu einem leeren Datensatz vor, welcher spter ersetzt wird. */
    private static Chart3D chart() {
        Chart3D chart = Chart3DFactory.createBarChart(null, null, platzhalter(), null, "Spielnummer", "Punktzahl");
        chart.setAntiAlias(true);
        chart.setLegendBuilder(null);
        chart.setChartBoxColor(WANDFARBE);
        chart.setProjDistance(PROJEKTIONSDISTANZ);
        chart.setViewPoint(perspektive());
        chart.setBackground(new StandardRectanglePainter(Report.HINTERGRUNDFARBE));
        chart.setTranslate2D(VERSCHIEBUNG);
        return chart;
    }

    /**
     * Liefert einen Datensatz zur Statistik. Der Datensatz besteht aus Datenreihen, die durch den Spieler indiziert
     * sind; eine Datenreihe besteht aus den Punktzahlen fr einen Spieler ber das bisherige Turnier (Series: Spieler,
     * Row: Spieler, Column: Spielnummer).
     */
    private static StandardCategoryDataset3D<String, Integer, Integer> datensatz(Statistik statistik) {
        StandardCategoryDataset3D</* Series */ String, /* Row */ Integer, /* Col */ Integer> datensatz;
        datensatz = new StandardCategoryDataset3D<>();
        for (Spieler spieler : statistik.teilnehmer()) {
            DefaultKeyedValues</* Spielnummer */ Integer, /* Punktzahl */ Integer> reihe = new DefaultKeyedValues<>();
            for (int i = 0; i < statistik.spielanzahl(); i++) {
                reihe.put(/* Spielnummer */ i + 1, /* Punktzahl */ statistik.verlaeufe().get(spieler.index()).get(i));
            }
            datensatz.addSeriesAsRow(spieler.identitaet(), spieler.index(), reihe);
        }
        return datensatz;
    }

    /** Konfiguriert Koordinatensystem und Darstellung. */
    private static void konfiguriere(CategoryPlot3D plot) {
        plot.getRowAxis().setVisible(false);
        plot.setGridlineStrokeForValues(GITTERLINIENSTIL);
        plot.setToolTipGenerator(new StandardCategoryItemLabelGenerator(
                "<html><strong>%1$s</strong> hat im %3$d. Spiel <strong>%4$.0f Punkte</strong> erzielt.</html>"));
        StandardCategoryAxis3D turnierachse = (StandardCategoryAxis3D) plot.getColumnAxis();
        turnierachse.setLabelFont(ACHSSCHRIFTART);
        turnierachse.setLabelColor(ACHSSCHRIFTFARBE);
        turnierachse.setTickLabelFont(ZAHLSCHRIFTART);
        turnierachse.setTickLabelColor(ZAHLSCHRIFTFARBE);
        NumberAxis3D punktzahlachse = (NumberAxis3D) plot.getValueAxis();
        punktzahlachse.setLabelFont(ACHSSCHRIFTART);
        punktzahlachse.setLabelColor(ACHSSCHRIFTFARBE);
        punktzahlachse.setTickLabelFont(ZAHLSCHRIFTART);
        punktzahlachse.setTickLabelColor(ZAHLSCHRIFTFARBE);
        punktzahlachse.setAutoRangeIncludeZero(true);
        punktzahlachse.setAutoRangeStickyZero(true);
        punktzahlachse.setRange(WERTEBEREICH);
        BarRenderer3D renderer = (BarRenderer3D) plot.getRenderer();
        renderer.setColors(Report.BALKENFARBEN);
        renderer.setTopColorSource(new StandardCategoryColorSource(Color.YELLOW));
        renderer.setBarXWidth(XLAENGE);
        renderer.setBarZWidth(ZLAENGE);
    }

    /** Liefert das Zeichenfeld, wo das Diagramm dargestellt wird. */
    private static Chart3DPanel panel(Chart3D chart) {
        Chart3DPanel panel = new Chart3DPanel(chart);
        panel.setMargin(RANDABSTAND);
        panel.zoomToFit();
        panel.addMouseListener(new Hinweissteuerung());
        panel.addChartMouseListener(new Chart3DMouseListener() {

            @Override
            public void chartMouseClicked(Chart3DMouseEvent event) {
                if (event.getTrigger().getClickCount() > 1) {
                    chart.setViewPoint(perspektive());
                    panel.zoomToFit();
                }
            }

            @Override
            public void chartMouseMoved(Chart3DMouseEvent event) {
            }
        });
        return panel;
    }

    /** Liefert die Standardperspektive. */
    private static ViewPoint3D perspektive() {
        ViewPoint3D perspektive = ViewPoint3D.createAboveRightViewPoint(RHO);
        perspektive.panLeftRight(DREHWINKEL);
        perspektive.roll(ROLLWINKEL);
        return perspektive;
    }

    /** Berechnet initialen Platzhalter fr die rumlich Darstellung zwecks Grenabschtzung. */
    private static StandardCategoryDataset3D<String, Integer, Integer> platzhalter() {
        StandardCategoryDataset3D<String, Integer, Integer> platzhalter = new StandardCategoryDataset3D<>();
        DefaultKeyedValues</* Spielnummer */ Integer, /* Punktzahl */ Integer> reihe;
        for (int s = 1; s <= S; s++) {
            reihe = new DefaultKeyedValues<>();
            for (int n = 1; n <= N; n++) {
                reihe.put(n, P);
            }
            platzhalter.addSeriesAsRow("Name " + s + " (Spieler " + s + ")", s, reihe);
        }
        return platzhalter;
    }

    /** Verwaltung aller Diagrammbestandteile. */
    private final Chart3D chart;
    /** Rumliche Darstellung der Werte zu zwei Kategorien (Spieler, Spieler). */
    private final CategoryPlot3D plot;
    /** Das Zeichenfeld zur Swing-Integration. */
    private final Chart3DPanel panel;
    /** Zeichenfeld mit Toolbar und Kontextmen. */
    private final DisplayPanel3D display;

    /** Datensatz zur rumlichen Darstellung. */
    private StandardCategoryDataset3D<String, Integer, Integer> datensatz;

    /** Konstruktor. */
    Verlaufsgrafik() {
        super(new BorderLayout());
        chart = chart();
        plot = (CategoryPlot3D) chart.getPlot();
        konfiguriere(plot);
        panel = panel(chart);
        display = new DisplayPanel3D(panel);
        add(display);
    }

    /**
     * Aktualisiert die Darstellung zum Stand der letzten vollendeten Runde.
     * 
     * @param statistik
     *            Die Statistik, welche den Turnierverlauf enthlt.
     */
    void aktualisiere(Statistik statistik) {
        for (Spieler spieler : statistik.teilnehmer()) {
            int punktzahl = statistik.verlaeufe().get(spieler.index()).get(statistik.spielanzahl() - 1);
            datensatz.setValue(punktzahl, spieler.identitaet(), spieler.index(), statistik.spielanzahl());
        }
    }

    /**
     * Passt die Darstellung an das Ende eines Spiels im Turnier an.
     * 
     * @param statistik
     *            Die Statistik, welche den Turnierverlauf enthlt.
     */
    void beende(Statistik statistik) {
        if (statistik.rest() > 0) {
            for (Spieler spieler : statistik.teilnehmer()) {
                datensatz.setValue(0, spieler.identitaet(), spieler.index(), statistik.spielanzahl());
            }
        }
    }

    /**
     * Startet eine neue Darstellung mit der gegebenen Statistik.
     *
     * @param statistik
     *            Die darzustellende Statistik.
     */
    void starte(Statistik statistik) {
        datensatz = datensatz(statistik);
        plot.setDataset(datensatz);
        panel.zoomToFit();
    }

}
