/******************************************************************************
 ** $Id: Menue.java 1625 2019-01-03 12:00:21Z wmh $
 ** Diese Datei ist Bestandteil der Java-Quelltexte des Wrfelspiels JaFuffy.
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************
 ** Die aktuellste Version von JaFuffy findet sich im Internet unter
 ** <http://jafuffy.3kelvin.de>.
 **
 ** Kommentare, Fehler oder Erweiterungswnsche bitte per E-Mail senden an
 ** <jafuffy@3kelvin.de>.
 ******************************************************************************/

package jafuffy.bedienung;

import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.WindowEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import javax.swing.ButtonGroup;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JRadioButtonMenuItem;
import javax.swing.KeyStroke;
import javax.swing.SwingUtilities;
import javax.swing.ToolTipManager;
import javax.swing.UIManager;
import javax.swing.UIManager.LookAndFeelInfo;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import jafuffy.Eigenschaften;
import jafuffy.Ressource;
import jafuffy.logik.CEAblauf;
import jafuffy.logik.CEJaFuffy;
import jafuffy.logik.Turnier;
import jafuffy.logik.Update;

/** Erzeugung der Menzeile und Einrichtung der Ereignislauscher der Meneintrge. */
@SuppressWarnings("serial")
public class Menue extends JMenuBar implements PropertyChangeListener, ChangeListener {

    /** Wechseln von Eigenschaften. */
    private class Wechsel implements ActionListener {
        private final String schluessel;

        Wechsel(String schluessel) {
            this.schluessel = schluessel;
        }

        @Override
        public void actionPerformed(ActionEvent e) {
            eigenschaften.wechsle(schluessel);
        }

        boolean aktiv() {
            return eigenschaften.aktiv(schluessel);
        }
    }

    /** Fehlermeldung fr Applet. */
    private static final String SANDBOX = "<html>"
            + "<p><strong>Funktion auer Betrieb:</strong> <i>JaFuffy</i> luft mit eingeschrnkten Rechten.</p>"
            + "<p>Dies ist u.a. der Fall bei der Ausfhrung von <i>JaFuffy</i> per Java Web Start.</p>" + "<br>"
            + "<p>Zur Bereitstellung der Funktion bitte <i>JaFuffy</i> ber Homepage installieren.</p>" + "</html>";

    /** Dialog zur Anzeige des Turnierverlaufs. */
    private final Report report;
    /** Dialoge zur Anzeige der Bestenliste. */
    private final Bestenliste bestenliste;
    /** Dialog zur Fortsetzung von Turnieren. */
    private final Fortsetzung fortsetzung;
    /** berprfung auf Update von JaFuffy. */
    private final Update update;
    /** Dialog zur Einstellungen der Spielstrken der einzelnen Bots. */
    private final Botstaerke botstaerke;
    /** Rechtliche Informationen, Versionsangabe. */
    private final Info info = new Info();
    /** Hinweise zur Lizenzierung. */
    private final Lizenzierung lizenzierung = new Lizenzierung();

    private final JMenuItem speichern = new JMenuItem("Speichern");
    private final JMenuItem speichernUnter = new JMenuItem("Speichern unter...");
    private final JMenuItem berichten = new JMenuItem("Report");
    private final JMenuItem rueckgaengig = new JMenuItem("Rckgngig");
    private final JMenuItem beenden = new JMenuItem("Beenden");
    /** Die Eigenschaften, mit denen JaFuffy betrieben wird. */
    private final Eigenschaften eigenschaften;
    /** Meneintrag zur Erstellung eines Vorschlags fr die Auswahl eines Eintrages und dazu passender Wrfel. */
    private final Vorschlagen vorschlagen;
    /** Meneintrag zum Setzen der Wrfelauswahl. */
    private final Setzen setzen;

    /** Im Spielbetrieb oder noch Intro? */
    private boolean betrieb = false;

    /**
     * Erstellt Drop-Down-Men.
     *
     * @param eigenschaften
     *            Die Eigenschaften, mit denen JaFuffy betrieben wird.
     * @param vorschlagen
     *            Meneintrag zur Erstellung eines Vorschlags fr die Auswahl eines Eintrages und dazu passender Wrfel.
     * @param setzen
     *            Meneintrag zum Setzen der Wrfelauswahl.
     * @param report
     *            Dialog zur Anzeige des Turnierverlaufs.
     * @param bestenliste
     *            Dialoge zur Anzeige der Bestenliste.
     * @param fortsetzung
     *            Dialog zur Fortsetzung von Turnieren.
     * @param update
     *            berprfung auf Update von JaFuffy.
     */
    public Menue(Eigenschaften eigenschaften, Vorschlagen vorschlagen, Setzen setzen, Report report,
            Bestenliste bestenliste, Fortsetzung fortsetzung, Update update) {
        this.eigenschaften = eigenschaften;
        this.vorschlagen = vorschlagen;
        this.setzen = setzen;
        this.report = report;
        this.bestenliste = bestenliste;
        this.fortsetzung = fortsetzung;
        this.update = update;
        botstaerke = new Botstaerke(eigenschaften);

        UIManager.addPropertyChangeListener(this);

        erzeugeTurniermenue();
        erzeugeStatistikmenue();
        erzeugeExtrasmenue();
        erzeugeEinstellungenmenue();
        erzeugeHilfemenue();
    }

    @Override
    public void propertyChange(PropertyChangeEvent event) {
        if (event.getPropertyName().equals("lookAndFeel")) {
            SwingUtilities.updateComponentTreeUI(this);
        } else if (event.getPropertyName().equals("Speichern")) {
            speichern.setEnabled((boolean) event.getNewValue());
        }
    }

    @Override
    public void stateChanged(ChangeEvent ce) {
        if (CEJaFuffy.adressiert(ce, CEAblauf.class)) {
            CEJaFuffy<CEAblauf> cej = CEJaFuffy.ereignisbehaelter(ce);
            Turnier turnier = (Turnier) cej.quelle(Turnier.class);
            switch (cej.ereignis()) {
            case START:
                meldeTurnierAn(turnier);
                break;
            case RESULTAT:
                aktualisiere(false, false, false);
                break;
            case ABBRUCH:
            case ENDE:
                meldeTurnierAb(turnier);
                break;
            case RUECKGAENGIG:
            case GEWUERFELT:
                aktualisiere(true, false, false);
                break;
            case GESETZT:
                aktualisiere(false, false, turnier.ruecknahmefaehig());
                break;
            default:
                break;
            }
        }
    }

    /**
     * Aktualisiert einige Meneintrge.
     * 
     * @param vorschlagbar
     *            Gibt an, ob Vorschlge angefordet werden knnen.
     * @param setzbar
     *            Gibt an, ob ein Wurf setzbar ist.
     * @param ruecknahmefaehig
     *            Gibt an, ob ein Setzvorgang rckgngig gemacht werden kann.
     */
    private void aktualisiere(boolean vorschlagbar, boolean setzbar, boolean ruecknahmefaehig) {
        vorschlagen.setEnabled(vorschlagbar);
        setzen.setEnabled(setzbar);
        rueckgaengig.setEnabled(ruecknahmefaehig);
    }

    /**
     * Erzeugt das Einstellungsmen.
     */
    private void erzeugeEinstellungenmenue() {
        JMenu menu = new JMenu("Einstellungen");
        menu.setMnemonic('E');
        add(menu);
        JMenu subMenu;
        JMenuItem menuItem;
        // Einfachklick
        menuItem = menu.add(new JCheckBoxMenuItem("Setzen durch Einfachklick"));
        menuItem.setToolTipText("<html>" + "Wahl zwischen einmaligem und zweimaligem Klicken" + "<br>"
                + "zum Setzen des Wurfs in der Tabelle" + "</html>");
        menuItem.setSelected(eigenschaften.aktiv("Einfachklick"));
        menuItem.addActionListener(new Wechsel("Einfachklick"));
        // Botkontrolle
        subMenu = new JMenu("Botverhalten");
        subMenu.setToolTipText(
                "<html>Ein Bot ist ein Gegenspieler, dessen Rolle von <i>JaFuffy</i> bernommen wird.</html>");
        menu.add(subMenu);
        menuItem = subMenu.add(new JMenuItem("Spielstrke der Bots..."));
        menuItem.setToolTipText("Spielstrke der einzelnen Bots einstellen oder ansehen");
        menuItem.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                botstaerke.setLocationRelativeTo(SwingUtilities.getRootPane(Menue.this));
                botstaerke.setVisible(true);
            }
        });
        subMenu.addSeparator();
        menuItem = subMenu.add(new JCheckBoxMenuItem("berlegungen nur andeuten"));
        menuItem.setToolTipText(
                "<html>" + "Falls ausgewhlt werden die berlegungen des Bots zur Wrfelauswahl nur kurz dargestellt,"
                        + "<br>" + " mssen aber nicht besttigt werden." + "</html>");
        menuItem.setSelected(eigenschaften.aktiv("Ueberlegungsdarstellungsbeschleunigung"));
        menuItem.addActionListener(new Wechsel("Ueberlegungsdarstellungsbeschleunigung"));
        menuItem = subMenu.add(new JCheckBoxMenuItem("Beschleunigung der Aktionskontrolle"));
        menuItem.setToolTipText("Falls ausgewhlt werden Aktionen des Bots krzer dargestellt.");
        menuItem.setSelected(eigenschaften.aktiv("Aktionsdarstellungsbeschleunigung"));
        menuItem.addActionListener(new Wechsel("Aktionsdarstellungsbeschleunigung"));
        //
        menu.addSeparator();
        //
        // Zwischenstand anzeigen
        menuItem = menu.add(new JCheckBoxMenuItem("Zwischenstandsanzeige"));
        menuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_Z, 0));
        menuItem.setToolTipText("<html>" + "Die Punktezahl des derzeitigen Wurfs" + "<br>"
                + "fr jeden freien Tabellenplatz anzeigen?" + "</html>");
        menuItem.setSelected(eigenschaften.aktiv("Zwischenstand"));
        menuItem.addActionListener(new Wechsel("Zwischenstand"));
        menuItem.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                SwingUtilities.getRootPane(Menue.this).repaint();
            }
        });
        // Nachfragen
        subMenu = new JMenu("Meldungen");
        subMenu.setToolTipText("Meldungen ein/ausschalten bei...");
        menu.add(subMenu);
        menuItem = subMenu.add(new JCheckBoxMenuItem("Hinweise"));
        menuItem.setToolTipText("Allgemeine Hinweise (\"ToolTips\" in Ksten) anzeigen?");
        menuItem.setSelected(eigenschaften.aktiv("Hinweis"));
        ToolTipManager.sharedInstance().setEnabled(eigenschaften.aktiv("Hinweis"));
        menuItem.addActionListener(new Wechsel("Hinweis") {
            @Override
            public void actionPerformed(ActionEvent e) {
                super.actionPerformed(e);
                ToolTipManager.sharedInstance().setEnabled(aktiv());
            }
        });
        menuItem = subMenu.add(new JCheckBoxMenuItem("Spicker"));
        menuItem.setToolTipText("Spicker fr Wrfelpunkte in Tabelle?");
        menuItem.setSelected(eigenschaften.aktiv("Spicker"));
        menuItem.addActionListener(new Wechsel("Spicker"));
        menuItem = subMenu.add(new JCheckBoxMenuItem("Erinnerung abgebrochenes Turnier"));
        if (fortsetzung != null) {
            menuItem.setToolTipText("Erinnerung an abgebrochenes Turnier zum Weiterspielen?");
            menuItem.setSelected(eigenschaften.aktiv("Erinnerung"));
            menuItem.addActionListener(new Wechsel("Erinnerung"));
        } else {
            menuItem.setEnabled(false);
            menuItem.setToolTipText(SANDBOX);
        }
        // Effekte
        subMenu = new JMenu("Effekte");
        subMenu.setToolTipText("Grafik- und Toneffekte");
        menu.add(subMenu);
        menuItem = subMenu.add(new JCheckBoxMenuItem("Eingangsgrafik"));
        menuItem.setToolTipText("Eingangsgrafik ein/ausschalten");
        menuItem.setSelected(eigenschaften.aktiv("Intro"));
        menuItem.addActionListener(new Wechsel("Intro") {
            @Override
            public void actionPerformed(ActionEvent e) {
                if (!betrieb) {
                    super.actionPerformed(e);
                    if (aktiv()) {
                        firePropertyChange("Karte", Spielplatz.TEXT, Spielplatz.INTRO);
                    } else {
                        firePropertyChange("Karte", Spielplatz.INTRO, Spielplatz.TEXT);
                    }
                }
            }
        });
        menuItem = subMenu.add(new JCheckBoxMenuItem("Gerusche"));
        menuItem.setEnabled(Ressource.soundsystem());
        menuItem.setToolTipText("Gerusche ein/ausschalten");
        menuItem.setSelected(Ressource.soundsystem() && eigenschaften.aktiv("Geraeusche"));
        menuItem.addActionListener(new Wechsel("Geraeusche"));
        // Look & Feel
        subMenu = new JMenu("Look and Feel");
        subMenu.setToolTipText("<html>" + "<p>Look & Feel der Benutzeroberflche.</p>"
                + "<p>Je nach Auswahl kann ein Neustart erforderlich sein, siehe Extras-Men.</p>" + "</html>");
        menu.add(subMenu);
        ButtonGroup oberflaechengruppe = new ButtonGroup();
        for (final LookAndFeelInfo laf : lafs()) {
            menuItem = new JRadioButtonMenuItem(laf.getName());
            String alt = eigenschaften.getProperty("LookAndFeel");
            menuItem.setSelected(alt.equals(laf.getClassName()));
            menuItem.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent event) {
                    String alt = eigenschaften.getProperty("LookAndFeel");
                    String neu = laf.getClassName();
                    firePropertyChange("LookAndFeel", alt, neu);
                }
            });
            subMenu.add(menuItem);
            oberflaechengruppe.add(menuItem);
        }
        //
        menu.addSeparator();
        //
        // Pfad zu Turnierstnden
        menuItem = menu.add(new JMenuItem("Pfad..."));
        if (eigenschaften.pfad() != null) {
            menuItem.setToolTipText("<html>" + "<p>Pfad zu Statistikdaten und Turnierstnden einstellen.</p>"
                    + "<p>Aktuelle Einstellung: " + "<strong>" + eigenschaften.pfad() + "</strong></p>" + "</html>");
            menuItem.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent e) {
                    new Pfad(eigenschaften, SwingUtilities.getRootPane(Menue.this));
                }
            });
        } else {
            menuItem.setEnabled(false);
            menuItem.setToolTipText(SANDBOX);
        }
        //
        menu.addSeparator();
        //
        // Nach neueren Versionen suchen
        menuItem = menu.add(new JCheckBoxMenuItem("Nach neuerer Version schauen"));
        if (eigenschaften.pfad() != null) {
            menuItem.setToolTipText("<html>Bei jedem Programmstart automatisch "
                    + "nach neuerer Version auf <i>JaFuffy</i>-Homepage suchen<html>");
            menuItem.setSelected(eigenschaften.aktiv("Nachschauen"));
            menuItem.addActionListener(new Wechsel("Nachschauen"));
        } else {
            menuItem.setEnabled(false);
            menuItem.setToolTipText(SANDBOX);
        }
    }

    /** Erzeugung eines Mens. */
    private void erzeugeExtrasmenue() {
        JMenu menu = new JMenu("Extras");
        menu.setMnemonic('X');
        add(menu);

        JMenuItem menuItem;

        menu.add(rueckgaengig);
        rueckgaengig.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_R, 0));
        rueckgaengig.setActionCommand("Rueckgaengig");
        rueckgaengig.setEnabled(false);
        rueckgaengig.setToolTipText("Letztes Setzen rckgngig machen");
        menu.add(setzen);
        //
        menu.addSeparator();
        //
        menuItem = menu.add(new JMenuItem("Fenster anpassen"));
        menuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_A, 0));
        menuItem.setToolTipText("Fenstergre optimal anpassen " + "an Tabelle, Wrfelfeld und Look & Feel");
        menuItem.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                SwingUtilities.invokeLater(new Runnable() {
                    @Override
                    public void run() {
                        Window umgebung = SwingUtilities.getWindowAncestor(Menue.this);
                        umgebung.pack();
                    }
                });
            }
        });
        //
        menu.addSeparator();
        //
        menuItem = menu.add(new JMenuItem("Neu zeichnen"));
        menuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_N, 0));
        menuItem.setToolTipText("Alles neu zeichnen zur Beseitigung von Darstellungsfehlern");
        menuItem.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                SwingUtilities.getRootPane(Menue.this).repaint();
            }
        });
        menuItem = menu.add(new JMenuItem("Neustart"));
        if (eigenschaften.pfad() != null) {
            menuItem.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent event) {
                    eigenschaften.merkeNeustartAnforderung(betrieb);
                    beenden.doClick();
                }
            });
            menuItem.setToolTipText("<html>" + "<p>Neustart durchfhren mit automatischer Turnierfortsetzung.</p>"
                    + "<p>Hilfreich weil Look & Feel-Wechsel nicht immer funktionieren.</p>" + "</html>");
        } else {
            menuItem.setEnabled(false);
            menuItem.setToolTipText(SANDBOX);
        }

    }

    /** Erzeugung eines Mens. */
    private void erzeugeHilfemenue() {
        JMenu menu = new JMenu("Hilfe");
        menu.setMnemonic('H');
        add(menu);

        JMenuItem menuItem;

        menu.add(vorschlagen);
        vorschlagen.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                setzen.setEnabled(true);
            }
        });
        //
        menu.addSeparator();
        //
        menuItem = menu.add(new JMenuItem("Hilfethemen"));
        menuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_H, 0));
        menuItem.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent event) {
                try {
                    Hilfe hilfe = new Hilfe();
                    hilfe.setLocationRelativeTo(SwingUtilities.getRootPane(Menue.this));
                    hilfe.setVisible(true);
                } catch (Exception exception) {
                    exception.printStackTrace();
                    Signal.WARNUNG.play();
                    JOptionPane.showMessageDialog(SwingUtilities.getRootPane(Menue.this),
                            "Hilfetexte knnen nicht angezeigt werden!\n"
                                    + "Fehlerhafte Installation? Keine Internetverbindung?\n" + "Java-Exception: "
                                    + exception,
                            "JaFuffy (Warnung)", JOptionPane.WARNING_MESSAGE);
                }
            }
        });
        //
        menu.addSeparator();
        //
        // ber JaFuffy
        menuItem = menu.add(new JMenuItem("<html>ber <i>JaFuffy</i></html>"));
        menuItem.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                info.setLocationRelativeTo(SwingUtilities.getRootPane(Menue.this));
                info.setVisible(true);
            }
        });
        // Lizenzierung
        menuItem = menu.add(new JMenuItem("Lizenzierung"));
        menuItem.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                lizenzierung.setLocationRelativeTo(SwingUtilities.getRootPane(Menue.this));
                lizenzierung.setVisible(true);
            }
        });
        //
        menu.addSeparator();
        //
        // Aktualisieren
        menuItem = menu.add(new JMenuItem("Aktualisieren..."));
        menuItem.setToolTipText("<html>" + "Sucht auf der <i>JaFuffy</i>-Homepage nach neuerer Version" + "<html>");
        menuItem.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                Menue.this.update.ueberpruefe(SwingUtilities.getRootPane(Menue.this));
            }
        });
    }

    /** Erzeugung eines Mens. */
    private void erzeugeStatistikmenue() {
        JMenu menu = new JMenu("Statistik");
        menu.setMnemonic('S');
        add(menu);

        JMenuItem menuItem;

        menu.add(berichten);
        berichten.setToolTipText("(Zischen)Stand & Turnierverlauf");
        berichten.setEnabled(false);
        berichten.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                Menue.this.report.setLocationRelativeTo(SwingUtilities.getRootPane(Menue.this));
                Menue.this.report.setVisible(true);
            }
        });
        menuItem = menu.add(new JMenuItem("Bestenliste"));
        menuItem.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                Menue.this.bestenliste.setLocationRelativeTo(SwingUtilities.getRootPane(Menue.this));
                Menue.this.bestenliste.setVisible(true);
            }
        });
    }

    /** Erzeugung eines Mens. */
    private void erzeugeTurniermenue() {
        JMenu menu = new JMenu("Turnier");
        menu.setMnemonic('T');
        add(menu);

        JMenuItem menuItem;

        menuItem = menu.add(new JMenuItem("Beginnen..."));
        menuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_B, 0));
        menuItem.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                Menue.this.firePropertyChange("Beginnen", null, SwingUtilities.getRootPane(Menue.this));
            }
        });
        //
        menu.addSeparator();
        //
        menu.add(speichern);
        speichern.setEnabled(false);
        speichern.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_S, 0));
        if (fortsetzung != null) {
            speichern.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent e) {
                    Menue.this.fortsetzung.setLocationRelativeTo(SwingUtilities.getRootPane(Menue.this));
                    Menue.this.fortsetzung.speichere();
                }
            });
        } else {
            speichern.setToolTipText(SANDBOX);
        }
        menu.add(speichernUnter);
        speichernUnter.setEnabled(false);
        if (fortsetzung != null) {
            speichernUnter.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent e) {
                    Menue.this.fortsetzung.setLocationRelativeTo(SwingUtilities.getRootPane(Menue.this));
                    Menue.this.fortsetzung.speichereUnter();
                    Menue.this.fortsetzung.setVisible(true);
                }
            });
        } else {
            speichernUnter.setToolTipText(SANDBOX);
        }
        menuItem = menu.add(new JMenuItem("Fortsetzen..."));
        if (fortsetzung != null) {
            menuItem.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent e) {
                    Menue.this.fortsetzung.setLocationRelativeTo(SwingUtilities.getRootPane(Menue.this));
                    Menue.this.fortsetzung.setzeFort();
                    Menue.this.fortsetzung.setVisible(true);
                }
            });
        } else {
            menuItem.setEnabled(false);
            menuItem.setToolTipText(SANDBOX);
        }
        //
        menu.addSeparator();
        //
        menu.add(beenden);
        beenden.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                Window umgebung = SwingUtilities.getWindowAncestor(Menue.this);
                WindowEvent ereignis = new WindowEvent(umgebung, WindowEvent.WINDOW_CLOSING);
                umgebung.dispatchEvent(ereignis);
            }
        });
    }

    /** @return Eine alphabetisch sortierte Liste aller Look & Feel. */
    private List<LookAndFeelInfo> lafs() {
        List<LookAndFeelInfo> lafs = Arrays.asList(UIManager.getInstalledLookAndFeels());
        Collections.sort(lafs, new Comparator<LookAndFeelInfo>() {
            @Override
            public int compare(LookAndFeelInfo laf1, LookAndFeelInfo laf2) {

                return laf1.getName().compareToIgnoreCase(laf2.getName());
            }
        });
        return lafs;
    }

    /**
     * Meldet Turnier ab.
     *
     * @param turnier
     *            Abzumeldendes Turnier.
     */
    private void meldeTurnierAb(Turnier turnier) {
        betrieb = false;
        aktualisiere(false, false, false);
        speichern.setEnabled(false);
        speichernUnter.setEnabled(false);
        vorschlagen.removeActionListener(turnier);
        rueckgaengig.removeActionListener(turnier);
    }

    /**
     * Meldet Turnier an.
     *
     * @param turnier
     *            Anzumeldendes Turnier.
     */
    private void meldeTurnierAn(Turnier turnier) {
        betrieb = true;
        aktualisiere(turnier.vorschlagbar(), false, false);
        vorschlagen.addActionListener(turnier);
        rueckgaengig.addActionListener(turnier);
        speichern.setEnabled(false);
        speichernUnter.setEnabled(fortsetzung != null);
        berichten.setEnabled(true);
    }

}
