/******************************************************************************
 ** $Id: Analyse.java 1646 2019-01-12 05:33:11Z wmh $
 ** Diese Datei ist Bestandteil der Java-Quelltexte des Wrfelspiels JaFuffy.
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************
 ** Die aktuellste Version von JaFuffy findet sich im Internet unter
 ** <http://jafuffy.3kelvin.de>.
 **
 ** Kommentare, Fehler oder Erweiterungswnsche bitte per E-Mail senden an
 ** <jafuffy@3kelvin.de>.
 ******************************************************************************/
package jafuffy.logik.analyse;

import java.io.Serializable;
import java.util.Arrays;
import java.util.HashMap;
import java.util.IdentityHashMap;
import java.util.Map;
import java.util.Random;

import jafuffy.logik.Kombinatorik;
import jafuffy.logik.Spieler;
import jafuffy.logik.Kategorie;
import jafuffy.logik.Turnier;
import jafuffy.logik.Wuerfel;

/** Hlt Eintrge, erstellt Analysen und berechnet Prognose ber die erzielbare Punktzahl bei vorgelegtem Wurf. */
public abstract class Analyse implements Serializable {

    /** Eintrag in Tabelle. */
    protected abstract class Eintrag implements Serializable {

        /** Zur Serialisierung. */
        private static final long serialVersionUID = -1491939796847928172L;

        /** Wichtung fr freie Felder im Intervall [0,1]. */
        private double wichtung;
        /** Zufallsgenerator wird zur Umsetzung von Spielstrken bentigt zur Einfhrung einer Unsicherheit. */
        private final Random spielstaerkenunsicherheit = new Random();

        /** Karte der Entscheidungen, welcher Wrfel erneut gewrfelt werden soll. */
        protected final IdentityHashMap<Wuerfel, Boolean> neuwurfkarte = new IdentityHashMap<>(Turnier.WUERFEL);
        {
            nominiereNeuwurf(true);
        }

        /** Durch den Wurf tatschlich erzielte Punktzahl. */
        protected int wert;
        /** Voraussichtlicher Endstand, wenn der Wurf fr den aktuellen Eintrag weiter verfolgt wird. */
        protected double endsumme;
        /** Erwarteter mittlerer Punktstand pro Eintrag (Startmittel). */
        protected double mittel;

        /**
         * Berechnet den erwarteten, individuellen Zugewinn fr die vorliegende objektive Prognose, angepasst an die
         * Spielstrke.
         *
         * @param unsicherheit
         *            Die Unsicherheit des Spielers bzw. Bots, die auf die objektive Prognose angewendet wird.
         * @param prognose
         *            Die objektive Prognose nach bestmglichem Wissen.
         * @return Der an die Fhigkeiten des Spieler bzw. Bots angepasste individuelle Zugewinn.
         */
        private double individualzugewinn(double unsicherheit, double prognose) {
            return (2 * unsicherheit * spielstaerkenunsicherheit.nextDouble() + 1 - unsicherheit) * prognose;
        }

        /**
         * Bewertet den vorliegenden Wurf durch Ermittlung der tatschlich erwrfelten Punktzahl und deren Speicherung.
         *
         * @param spieler
         *            Spieler, fr welchen der vorliegende Wurf bewertet wird.
         */
        protected abstract void bewerte(Spieler spieler);

        /**
         * Bestimmt ber den Neuwurfstatus des gesamten Wrfelsatzes.
         *
         * @param istNeuwurf
         *            Entscheidet, ob ein Neuwurf anliegt.
         */
        protected void nominiereNeuwurf(boolean istNeuwurf) {
            for (Wuerfel wuerfel : wuerfelsatz) {
                neuwurfkarte.put(wuerfel, istNeuwurf);
            }
        }

        /**
         * @param spieler
         *            Der Spieler, der gerade an der Reihe ist.
         * @param zuwachs
         *            Erwarteter Zuwachs fr oberen und unteren Teil der Tabelle.
         * @return Prognose fr das zu erwartende Resultat fr den laufenden Eintrag zur Ermittlung des
         *         Gesamtergebnisses.
         */
        protected abstract double prognose(Spieler spieler, Zuwachs zuwachs);

        /**
         * Vorschlag zur Wrfelauswahl ermitteln. Wird vor der Rckgabe des Prognosewertes mittels {@link #prognose}
         * ausgefhrt.
         *
         * @param spieler
         *            Der Spieler, der an der Reihe ist.
         */
        protected abstract void waehle(Spieler spieler);

        /** @return Voraussichtlicher Punktendstand. */
        double endsumme() {
            return endsumme;
        }

        /**
         * Berechnet die erwartete Endsumme, welche sich aus der gegenwrtigen Endsumme und dem erwarteten Zuwachs
         * ergibt; einschlielich Bonus.
         *
         * @param spieler
         *            Der Spieler, der gerade an der Reihe ist.
         * @param zuwachs
         *            Der erwartete Zuwachs.
         * @return Erwartete Endsumme.
         */
        abstract double endsumme(Spieler spieler, Zuwachs zuwachs);

        /**
         * @param wuerfel
         *            Referenz auf Wrfel.
         * @return Soll Wrfel erneut gewrfelt werden?
         */
        boolean erneut(Wuerfel wuerfel) {
            return neuwurfkarte.get(wuerfel);
        }

        /**
         * Wichtung eintragen.
         *
         * @param wichtung
         *            Setzt die Wichtung eines einzelnen Eintrags.
         */
        void gewichte(double wichtung) {
            this.wichtung = wichtung;
        }

        /**
         * @param spieler
         *            Spielstrke des Spielers beeinflusst mittlere Punktzahl.
         * @return Mittlere Punktzahl des Eintrags bei 3maligem Wrfeln.
         */
        double mittel(Spieler spieler) {
            return individualzugewinn(spieler.prognoseunsicherheit(), mittel);
        }

        /**
         * Ermittle Prognose sprich tippe auf Endsumme, und erstelle Wrfelauswahl fr Vorschlag.
         *
         * @param spieler
         *            Spieler, der an der Reihe ist.
         * @param zuwachs
         *            Mittlerer erwarteter Zuwachs (pro Feld 3mal Wrfeln angenommen).
         */
        void tippe(Spieler spieler, Zuwachs zuwachs) {
            endsumme = endsumme(spieler, zuwachs);
            if (spieler.rest() > 0) {
                nominiereNeuwurf(false);
                waehle(spieler);
                endsumme += individualzugewinn(spieler.prognoseunsicherheit(), prognose(spieler, zuwachs));
            } else {
                // alle Wrfe verbraucht; vorgeben, als ob alle Wrfel nochmals zu werfen sind
                nominiereNeuwurf(true);
                endsumme += wert;
            }
        }

        /** @return Erzielte Punktzahl fr Wurf. */
        int wert() {
            return wert;
        }

        /** @return Wichtung */
        double wichtung() {
            return wichtung;
        }

    }

    /** Eintrag im oberen Tabellenteil. Mglicher Bonus wird bercksichtigt. */
    class Oben extends Eintrag {
        /** Zur Serialisierung. */
        private static final long serialVersionUID = -5829769579272736329L;

        /** Augenzahl. */
        private final int augen;
        /**
         * Index fr Eintrag (0, 1, 2, 3, 4 oder 5) im oberen Teil der Tabelle, abgeleitet aus der Augenzahl.
         */
        private final int ind;

        /**
         * Konstruktor.
         *
         * @param index
         *            Der Index des oberen Eintrags (0, 1, 2, 3, 4, 5).
         */
        Oben(Index index) {
            ind = index.ordinal();
            augen = index.augen();
            mittel = augen * Kombinatorik.fixgleichenmittel(Turnier.WUERFEL, Turnier.RUNDEN);
        }

        @Override
        protected void bewerte(Spieler spieler) {
            wert = augenhaeufigkeiten[ind] * augen;
        }

        @Override
        protected double prognose(Spieler spieler, Zuwachs zuwachs) {
            // freie Wrfel
            int n = Turnier.WUERFEL - augenhaeufigkeiten[ind];
            // Mindestpunktwert, schon erwrfelt
            double min = augen * augenhaeufigkeiten[ind];
            // Prognose fr Zuwchse (nochmals k gleiche Augenzahl)
            double punkte = min;
            // ermittle Beitrge fr weitere Anzahlen von gleichen Augenzahlen
            for (int k = 1; k <= n; k++) {
                double p = Kombinatorik.fixgleichenwahrscheinlichkeit(n, k, spieler.rest());
                double neu = k * augen;
                punkte += p * neu;
            }
            return punkte;
        }

        @Override
        protected void waehle(Spieler spieler) {
            for (Wuerfel wuerfel : wuerfelsatz) {
                neuwurfkarte.put(wuerfel, wuerfel.augen() != augen);
            }
        }

        @Override
        double endsumme(Spieler spieler, Zuwachs zuwachs) {
            return spieler.endsumme()
                    + zuwachs.zuwachs(spieler.gesamt(), augenhaeufigkeiten[ind], spieler.rest(), augen)
                    - mittel(spieler);
        }
    }

    /** Eintrag im unteren Tabellenteil. */
    public abstract class Unten extends Eintrag {
        /** Zur Serialisierung bentigt. */
        private static final long serialVersionUID = 8894740065051562154L;

        /** @return Maximale Anzahl von Wrfeln mit gleichen Augenzahlen, von eins bis fnf. */
        protected int paschlaenge() {
            int max = 1;
            for (Map.Entry<Paschaufzaehlung, Boolean> pasch : paschkarte.entrySet()) {
                int groesse = pasch.getKey().groesse();
                boolean istPaschVorhanden = pasch.getValue();
                if (istPaschVorhanden && groesse > max) {
                    max = groesse;
                }
            }
            return max;
        }

        @Override
        double endsumme(Spieler spieler, Zuwachs zuwachs) {
            return spieler.endsumme() + zuwachs.zuwachs() - mittel(spieler);
        }
    }

    /** Zhlt alle Pasche von Zwei bis Fnf auf, wobei alle Wrfel zu einem festen Augenwert herangezogen werden. */
    protected enum Paschaufzaehlung {
        /** Zweierpasch. */
        ZWEIERPASCH,
        /** Dreierpasch und kein Zweierpasch. */
        DREIERPASCH,
        /** Viererpasch und weder Dreipasch noch Zweierpasch. */
        VIERERPASCH,
        /** Fnferpasch und weder Viererpasch noch Dreierpasch noch Zweierpasch. */
        FUENFERPASCH;
        /**
         * @return Anzahl der Wrfel im Pasch, wobei der Maximalwert fr einen festen Augenwert zurckgegeben wird.
         */
        public int groesse() {
            return ordinal() + 2;
        }
    }

    /** Paschgrundlagen. */
    public abstract class Pasch extends Unten {
        /** Zur Serialisierung. */
        private static final long serialVersionUID = 8666151619656864766L;

        /** Art des Pasches (3=klein: drei Gleiche, 4=gro: vier Gleiche). */
        private final int paschgroesse;
        /** Mit dieser Augenzahl wird der Pasch versucht. */
        private int paschaugenzahl;

        /**
         * Konstruktor.
         *
         * @param paschgroesse
         *            Art des Pasches (drei: klein, vier: gro)
         */
        protected Pasch(int paschgroesse) {
            this.paschgroesse = paschgroesse;
            mittel = augensumme(Turnier.RUNDEN);
        }

        /**
         * @param r
         *            Anzahl der Restwrfe.
         * @return Mittlere erwartete Augensumme.
         */
        private double augensumme(int r) {
            if (r > 0) {
                double augensumme;
                augensumme = Kombinatorik.gleichenwahrscheinlichkeit(1) * augensumme(r - 1);
                for (int k = 2; k < paschgroesse; k++) {
                    for (int a = 1; a <= 6; a++) {
                        augensumme += Kombinatorik.mehrlingswahrscheinlichkeit(a, k) * augensumme(a, r - 1, k);
                    }
                }
                augensumme += (3.5 * paschgroesse + (Turnier.WUERFEL - paschgroesse) * Kombinatorik.augenzahl(r))
                        * Kombinatorik.mindestgleichenwahrscheinlichkeit(paschgroesse);
                return augensumme;
            } else {
                return 0;
            }
        }

        /**
         * @param augen
         *            Mit dieser Augenzahl wird Pasch versucht.
         * @param r
         *            Anzahl der verbleibenden Restwrfe (0...3).
         * @param max
         *            Maximale Anzahl der schon vorliegenden Wrfel gleicher Augenzahl.
         * @return Mittlere erreichbare Punktzahl.
         */
        private double prognose(int augen, int r, int max) {
            double punkte = 0;
            if (max < paschgroesse) {
                if (max > 1) {
                    punkte = augensumme(augen, r, max);
                } else {
                    punkte = augensumme(r);
                }
            } else {
                for (Wuerfel wuerfel : wuerfelsatz) {
                    if (neuwurfkarte.get(wuerfel)) {
                        punkte += Kombinatorik.augenzahl(r);
                    } else {
                        punkte += wuerfel.augen();
                    }
                }
            }
            return punkte;
        }

        /**
         * @param a
         *            Fest vorgegebene Augenzahl fr einen Mehrling, wobei bei gleich langen Sequenzen diejenige mit der
         *            greren Augenzahl zu whlen ist.
         * @param r
         *            Anzahl der Restwrfe.
         * @param k
         *            Maximale Anzahl von Wrfeln zur vorgegebenen Augenzahl
         */
        protected abstract double augensumme(int a, int r, int k);

        @Override
        protected void bewerte(Spieler spieler) {
            wert = pasch() ? augensumme : 0;
        }

        /** @return Liegt ein erwrfelter Pasch vor? */
        protected abstract boolean pasch();

        @Override
        protected double prognose(Spieler spieler, Zuwachs zuwachs) {
            return prognose(paschaugenzahl, spieler.rest(), paschlaenge());
        }

        @Override
        protected void waehle(Spieler spieler) {
            paschaugenzahl = Wuerfel.MAXIMALAUGENAUGENZAHL;
            while (augenhaeufigkeiten[paschaugenzahl - 1] < paschlaenge()) {
                paschaugenzahl--;
            }
            int haeufigkeit = 0;
            for (Wuerfel wuerfel : wuerfelsatz) {
                if (wuerfel.augen() == paschaugenzahl) {
                    haeufigkeit++;
                    neuwurfkarte.put(wuerfel,
                            haeufigkeit > paschgroesse && Kombinatorik.augenzahl(spieler.rest()) > wuerfel.augen());
                } else {
                    neuwurfkarte.put(wuerfel, !pasch() || Kombinatorik.augenzahl(spieler.rest()) > wuerfel.augen());
                }
            }
        }
    }

    /** Chance. */
    class Chance extends Unten {

        /** Zur Serialisierung. */
        private static final long serialVersionUID = -3411668587180538841L;
        /** Erwartete Punkte (alle Augen aufsummiert). */
        private double punkte;

        /** Konstruktor. */
        Chance() {
            mittel = Turnier.WUERFEL * Kombinatorik.augenzahl(Turnier.RUNDEN);
        }

        @Override
        protected void bewerte(Spieler spieler) {
            wert = augensumme;
        }

        @Override
        protected double prognose(Spieler spieler, Zuwachs zuwachs) {
            return punkte;
        }

        @Override
        protected void waehle(Spieler spieler) {
            punkte = 0;
            for (Wuerfel wuerfel : wuerfelsatz) {
                double m = Kombinatorik.augenzahl(wuerfel.augen(), spieler.rest());
                neuwurfkarte.put(wuerfel, m > wuerfel.augen());
                punkte += m;
            }
        }

    }

    /** Zhlt alle Eintrge aus dem oberen Teil der Tabelle auf. */
    protected enum Index {
        /** Einser-Eintrag im oberen Tabellenteil. */
        EINSER,
        /** Zweier-Eintrag im oberen Tabellenteil. */
        ZWEIER,
        /** Dreier-Eintrag im oberen Tabellenteil. */
        DREIER,
        /** Vierer-Eintrag im oberen Tabellenteil. */
        VIERER,
        /** Fnfer-Eintrag im oberen Tabellenteil. */
        FUENFER,
        /** Sechser-Eintrag im oberen Tabellenteil. */
        SECHSER;
        /** @return Liefert die Augenzahl zum Eintrag zurck. */
        public int augen() {
            return ordinal() + 1;
        }

        /**
         * @param haeufigkeiten
         *            Feld mit allen Augenhufigkeiten.
         * @return Anzahl der Wrfel mit dieser Augenzahl.
         */
        public int vorkommen(int[] haeufigkeiten) {
            return haeufigkeiten[ordinal()];
        }
    }

    /** Alle Standardeintrge als Tabelle zusammengefasst. */
    protected abstract class Eintragungen extends HashMap<Kategorie, Eintrag> {
        /** Erlaubt Serialisierung. */
        private static final long serialVersionUID = 8130365096301351531L;

        /** Belegt die Standardeintrge, welche fr alle Varianten der Spielregeln gilt. */
        protected Eintragungen() {
            super(Kategorie.ALLE.length, 1);
            put(Kategorie.EINS, new Oben(Index.EINSER));
            put(Kategorie.ZWEI, new Oben(Index.ZWEIER));
            put(Kategorie.DREI, new Oben(Index.DREIER));
            put(Kategorie.VIER, new Oben(Index.VIERER));
            put(Kategorie.FUENF, new Oben(Index.FUENFER));
            put(Kategorie.SECHS, new Oben(Index.SECHSER));
            put(Kategorie.CHANCE, new Chance());
        }
    }

    /** Erwarteter Zuwachs fr diverse Tabellenteile. */
    protected static class Zuwachs {
        /** Erwarteter Bonuszuwachs. */
        private final double mittlererBonusZuwachs;
        /** Erwarteter Zuwachs fr obere und untere Tabelle, aber ohne Bonus. */
        private final double mittlererZuwachs;
        /** Beitragswahrscheinlichkeit eines einzelnen oberen freien Eintrags. */
        private final double p;
        /** Mindestanzahl der Wrfel fr jeden einzelnen oberen freien Eintrags. */
        private final int teilung;

        /**
         * Berechnung grundlegender Zuwachsgren.
         *
         * @param gesamt
         *            Die Gesamtpunktzahl aus dem oberen Tabellenteil.
         * @param mittlererObererZuwachs
         *            Erwarteter Zuwachs, aufsummiert ber alle oberen Eintrge.
         * @param augensummeObereFreieEintraege
         *            Die Augensumme der Augen aller freien oberen Eintrge.
         * @param anzahlFreieObereEintraege
         *            Die Anzahl der oberen freien Eintrge.
         * @param mittlererUntererZuwachs
         *            Erwarteter Zuwachs, aufsummiert ber alle unteren Eintrge.
         */
        Zuwachs(int gesamt, double mittlererObererZuwachs, int augensummeObereFreieEintraege,
                int anzahlFreieObereEintraege, double mittlererUntererZuwachs) {
            if (gesamt < Spieler.GRENZE && augensummeObereFreieEintraege > 0) {
                teilung = (Spieler.GRENZE - gesamt + augensummeObereFreieEintraege - 1) / augensummeObereFreieEintraege;
                double q = Kombinatorik.mindestfixgleichenwahrscheinlichkeit(teilung);
                mittlererBonusZuwachs = Math.pow(q, anzahlFreieObereEintraege) * Spieler.BONUS;
                p = q != 0 ? q : 1;
            } else {
                teilung = 0;
                p = 1;
                mittlererBonusZuwachs = 0;
            }
            mittlererZuwachs = mittlererObererZuwachs + mittlererUntererZuwachs;
        }

        /** Liefert den erwarteten Zuwachs fr die gesamte Tabelle unter Einbezug des Bonus. */
        double zuwachs() {
            return mittlererZuwachs + mittlererBonusZuwachs;
        }

        /**
         * @param gesamt
         *            Die Gesamtpunktzahl aus dem oberen Tabellenteil.
         * @param guenstige
         *            Die Anzahl der gnstigen Wrfel fr einen Eintrag.
         * @param r
         *            Die Anzahl der verbleibenden Restwrfe.
         * @param augen
         *            Die Augenzahl fr den Eintrag, aus dem der Zuwachs resultiert.
         * @return Der erwarte Zuwachs fr die gesamte Tabelle.
         */
        double zuwachs(int gesamt, int guenstige, int r, int augen) {
            double q = 0;
            int punkteBisZumBonus = Spieler.GRENZE - gesamt;
            if (punkteBisZumBonus > 0) {
                punkteBisZumBonus -= guenstige * augen;
                q = 1;
                if (punkteBisZumBonus > 0) {
                    int n = Turnier.WUERFEL - guenstige;
                    int min = guenstige <= teilung ? teilung - guenstige : 0;
                    q = Kombinatorik.mindestfixgleichenwahrscheinlichkeit(n, min, r);
                }
            }
            return mittlererZuwachs + q / p * mittlererBonusZuwachs;
        }
    }

    /** Zur Serialisierung. */
    private static final long serialVersionUID = -6325984568972413728L;

    /** Neutrale Wichtung, falls alle Wichtungen gleich sind. */
    private static final double NEUTRALWICHTUNG = 1;

    /** Zufallsgenerator zur Auswahl von Tipp-Strategien bei Mehrdeutigkeiten. */
    protected static final Random ZUFALL = new Random();

    /** Voraussichtlicher mittlerer Punktstand pro Feld, wobei alle drei Wrfe genutzt werden. */
    private final Eintragungen eintragungen;

    /** Bester Eintrag. */
    private Kategorie optimalzeile;

    /** Anzahl der Wrfel pro Augenzahl. */
    protected final int[] augenhaeufigkeiten = new int[Wuerfel.MAXIMALAUGENAUGENZAHL];
    /** Die Karte ordnet einem Pasch (2, 3, 4, 5 gleiche Wrfel) zu, ob genau ein solcher im Wurf vorhanden ist. */
    protected final Map<Paschaufzaehlung, Boolean> paschkarte = new HashMap<>(Paschaufzaehlung.values().length, 1);
    /** Im Turnier benutzte Wrfel. */
    protected final Wuerfel[] wuerfelsatz;

    /** Summe aller Augen. */
    protected int augensumme;

    /**
     * Konstruktor.
     *
     * @param wuerfelsatz
     *            Die im Turnier benutzten Wrfel.
     */
    protected Analyse(Wuerfel[] wuerfelsatz) {
        this.wuerfelsatz = wuerfelsatz;
        eintragungen = eintragungen();
    }

    /** @return bester Eintrag (-1 heit keiner vorhanden) */
    public Kategorie bester() {
        return optimalzeile;
    }

    /**
     * @param kategorie
     *            Eintrag in der Tabelle.
     * @param wuerfel
     *            Referenz auf Wrfel.
     * @return Wrfel nochmals wrfeln fr Vorschlag?
     */
    public boolean erneut(Kategorie kategorie, Wuerfel wuerfel) {
        return eintrag(kategorie).erneut(wuerfel);
    }

    /**
     * Tatschlich erwrfelte Punkte fr alle Eintrge errechnen.
     *
     * @param spieler
     *            Fr diesen Spieler werden die Berechnungen durchgefhrt.
     */
    public void errechne(Spieler spieler) {
        // Zhle Wrfel pro Augenzahl, Summe bestimmen
        Arrays.fill(augenhaeufigkeiten, 0);
        augensumme = 0;
        for (Wuerfel wuerfel : wuerfelsatz) {
            augensumme += wuerfel.augen();
            augenhaeufigkeiten[wuerfel.augen() - 1]++;
        }
        // Genau 2, 3, 4, 5 gleiche Wrfel?
        for (Paschaufzaehlung pasch : Paschaufzaehlung.values()) {
            boolean istPaschVorhanden = false;
            for (int augenhaeufigkeit : augenhaeufigkeiten) {
                istPaschVorhanden = augenhaeufigkeit == pasch.groesse() || istPaschVorhanden;
            }
            paschkarte.put(pasch, istPaschVorhanden);
        }
        // erwrfelte Punktzahl fr alle Eintrge
        for (Kategorie kategorie : Kategorie.ALLE) {
            eintrag(kategorie).bewerte(spieler);
        }
    }

    /**
     * @param kategorie
     *            Der zu untersuchende Eintrag.
     * @return Wrfelauswahl entspricht Vorschlag fr Eintrag.
     */
    public boolean istVorschlagEntsprechendWuerfelauswahl(Kategorie kategorie) {
        int[] anzahlenJeAugenAusgewaehlte = new int[Wuerfel.MAXIMALAUGENAUGENZAHL];
        int[] anzahlenJeAugenVorgeschlagene = new int[Wuerfel.MAXIMALAUGENAUGENZAHL];
        for (Wuerfel wuerfel : wuerfelsatz) {
            int a = wuerfel.augen() - 1;
            if (wuerfel.darstellungsmodell().isSelected()) {
                anzahlenJeAugenAusgewaehlte[a]++;
            }
            if (eintrag(kategorie).erneut(wuerfel)) {
                anzahlenJeAugenVorgeschlagene[a]++;
            }
        }
        for (int a = 0; a < Wuerfel.MAXIMALAUGENAUGENZAHL; a++) {
            if (anzahlenJeAugenAusgewaehlte[a] != anzahlenJeAugenVorgeschlagene[a]) {
                return false;
            }
        }
        return true;
    }

    /**
     * Punktprognosen aus Wurf ableiten.
     *
     * @param spieler
     *            Spieler, der an Reihe ist.
     */
    public void prognostiziere(Spieler spieler) {
        tippe(spieler, zuwachs(spieler));
        double max = findeMaximum(spieler);
        double min = findeMinimum(spieler, max);
        gewichte(spieler, min, max);
    }

    /**
     * @param kategorie
     *            Eintragsnummer.
     * @return Erwrfelter Punktstand.
     */
    public final int wert(Kategorie kategorie) {
        return eintrag(kategorie).wert();
    }

    /**
     * @param kategorie
     *            Eintragsnummer.
     * @return Wichtung fr erstellte Vorschlge.
     */
    public double wichtung(Kategorie kategorie) {
        return eintrag(kategorie).wichtung();
    }

    /**
     * @param kategorie
     *            Die Zeile in der Tabelle, zu der ein Eintrag zurckgegeben werden soll.
     * @return Eintrag zur Tabellenzeile
     */
    private Eintrag eintrag(Kategorie kategorie) {
        return eintragungen.get(kategorie);
    }

    /**
     * Ermittelt das Maximum ber alle Endstnde fr alle setzbaren Eintrge, und merkt sich einen Eintrag mit dem
     * voraussichtlich besten Endstand.
     *
     * @param spieler
     *            Der Spieler, der gerade an der Reihe ist.
     * @return Das Maximum aller Endstnde.
     */
    private double findeMaximum(Spieler spieler) {
        double max = 0;
        optimalzeile = null;
        for (Kategorie kategorie : Kategorie.ALLE) {
            if (spieler.setzbar(kategorie)) {
                double endstand = eintrag(kategorie).endsumme();
                if (endstand > max) {
                    optimalzeile = kategorie;
                    max = endstand;
                }
            }
        }
        return max;
    }

    /**
     * Ermittelt das Minimum ber alle Endstnde fr alle setzbaren Eintrge.
     *
     * @param spieler
     *            Der Spieler, der gerade an der Reihe ist.
     * @param max
     *            Der maximale erwartete Endstand fr alle setzbaren Eintrge.
     * @return Das Minimum aller Endstnde.
     */
    private double findeMinimum(Spieler spieler, double max) {
        double min = max;
        for (Kategorie kategorie : Kategorie.ALLE) {
            if (spieler.setzbar(kategorie)) {
                double endstand = eintrag(kategorie).endsumme();
                if (endstand < min) {
                    min = endstand;
                }
            }
        }
        return min;
    }

    /**
     * Gewichtet die setzbaren Eintrge nach erwarteter Hhe des Endstandes. Das Gewicht je Eintrag entspricht der
     * linearen Abbildung des Intervals zwischen dem minimalen und maximalen Endstand, abgebildet auf das Interval [0,
     * 1].
     *
     * @param spieler
     *            Der Spieler, der gerade an der Reihe ist.
     * @param min
     *            Der minimale erwartete Endstand fr alle setzbaren Eintrge.
     * @param max
     *            Der maximale erwartete Endstand fr alle setzbaren Eintrge.
     */
    private void gewichte(Spieler spieler, double min, double max) {
        for (Kategorie kategorie : Kategorie.ALLE) {
            if (spieler.setzbar(kategorie)) {
                if (max - min > 0) {
                    double endstand = eintrag(kategorie).endsumme();
                    eintrag(kategorie).gewichte((endstand - min) / (max - min));
                } else {
                    eintrag(kategorie).gewichte(NEUTRALWICHTUNG);
                }
            } else {
                eintrag(kategorie).gewichte(0);
            }
        }
    }

    /**
     * Tippt fr jeden freien Eintrag auf die erwartete Endsumme des aktiven Spielers.
     * 
     * @param spieler
     *            Der Spieler, welcher gerade an der Reihe ist.
     * @param zuwachs
     *            Der erwartetet Zuwachs, wenn fr jeden freien Eintrag dreimal gewrfelt wird ohne Bercksichtigung des
     *            vorliegenden Wurfs.
     */
    private void tippe(Spieler spieler, Zuwachs zuwachs) {
        for (Kategorie kategorie : Kategorie.ALLE) {
            if (spieler.setzbar(kategorie)) {
                eintrag(kategorie).tippe(spieler, zuwachs);
            }
        }
    }

    /**
     * Errechnet unabhngig vom vorliegenden Wurf die mittleren Zuwchse durch Einbeziehung aller setzbaren Eintrge.
     *
     * @param spieler
     *            Der Spieler, der an der Reihe ist.
     * @return Mittlere Zuwchse
     */
    private Zuwachs zuwachs(Spieler spieler) {
        double mittlererObererZuwachs = 0;
        int anzahlObereFreieEintraege = 0;
        int augensummeObereFreieEintraege = 0;
        for (Kategorie kategorie : Kategorie.OBEN) {
            if (spieler.setzbar(kategorie)) {
                Oben oben = (Oben) eintrag(kategorie);
                mittlererObererZuwachs += oben.mittel(spieler);
                anzahlObereFreieEintraege++;
                augensummeObereFreieEintraege += oben.augen;
            }
        }
        double mittlererUntererZuwachs = 0;
        for (Kategorie kategorie : Kategorie.UNTEN) {
            if (spieler.setzbar(kategorie)) {
                mittlererUntererZuwachs += eintrag(kategorie).mittel(spieler);
            }
        }
        return new Zuwachs(spieler.gesamt(), mittlererObererZuwachs, augensummeObereFreieEintraege,
                anzahlObereFreieEintraege, mittlererUntererZuwachs);
    }

    /** @return Alle Eintragungen in der Tabelle. */
    protected abstract Eintragungen eintragungen();

}
