/******************************************************************************
 ** $Id: Verlaufsgrafik.java 1212 2017-04-18 23:48:18Z wmh $
 ** Diese Datei ist Bestandteil der Java-Quelltexte des Wrfelspiels JaFuffy.
 ** Lauffhig ab Java 7.
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************
 ** Die aktuellste Version von JaFuffy findet sich im Internet unter
 ** <http://jafuffy.3kelvin.de>.
 **
 ** Kommentare, Fehler oder Erweiterungswnsche bitte per E-Mail senden an
 ** <jafuffy@3kelvin.de>.
 ******************************************************************************/
package jafuffy.bedienung;

import java.awt.BasicStroke;
import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Font;

import javax.swing.JPanel;
import javax.swing.SwingUtilities;

import com.orsoncharts.Chart3D;
import com.orsoncharts.Chart3DFactory;
import com.orsoncharts.Chart3DPanel;
import com.orsoncharts.Range;
import com.orsoncharts.axis.NumberAxis3D;
import com.orsoncharts.axis.StandardCategoryAxis3D;
import com.orsoncharts.graphics3d.Offset2D;
import com.orsoncharts.graphics3d.ViewPoint3D;
import com.orsoncharts.graphics3d.swing.DisplayPanel3D;
import com.orsoncharts.interaction.Chart3DMouseEvent;
import com.orsoncharts.interaction.Chart3DMouseListener;
import com.orsoncharts.label.StandardCategoryItemLabelGenerator;
import com.orsoncharts.plot.CategoryPlot3D;
import com.orsoncharts.renderer.category.BarRenderer3D;
import com.orsoncharts.renderer.category.StandardCategoryColorSource;
import com.orsoncharts.table.StandardRectanglePainter;

import jafuffy.logik.Statistik;

/** Anzeige des Turnierverlaufs in Form eines rumlichen Balkendiagramms. */
@SuppressWarnings("serial")
class Verlaufsgrafik extends JPanel {

    /** Bestimmt Balkenlnge in Richtung der Spielreihe. */
    private static final double ZLAENGE = 0.8;
    /** Bestimmt Balkenlnge in Richtung der Turnierabfolge. */
    private static final double XLAENGE = 0.5;
    /** Verschiebung im Fenster. */
    private static final Offset2D VERSCHIEBUNG = new Offset2D(0, -15);
    /** Wertebereich fr die darzustellenden Punktzahlen. */
    private static final Range WERTEBEREICH = new Range(0, 350);
    /** Kameraabstand. */
    private static final int RHO = 25;
    /** Kameraperspektive. */
    private static final ViewPoint3D PERSPEKTIVE = ViewPoint3D.createAboveViewPoint(RHO);
    static {
        PERSPEKTIVE.panLeftRight(Math.PI / 12);
        PERSPEKTIVE.moveUpDown(-Math.PI / 30);
        PERSPEKTIVE.roll(Math.PI / 24);
    }
    /** Projektionseigenschaft. */
    private static final int PROJEKTIONSDISTANZ = 2000;
    /** Darstellung der Gitterlinien in den Koordinatenwnden. */
    private static final BasicStroke GITTERLINIENSTIL = new BasicStroke(0.25f);
    /** Farbe der Koordinatenwnde. */
    private static final Color WANDFARBE = new Color(176, 192, 192, 64);
    /** Beschriftungsfarbe fr Punktzahlen. */
    private static final Color ZAHLSCHRIFTFARBE = Color.GRAY;
    /** Beschriftungsfarbe fr Achsenbeschriftung. */
    private static final Color ACHSSCHRIFTFARBE = Color.LIGHT_GRAY;
    /** Beschriftungszeichensatz fr Punktzahlen. */
    private static final Font ZAHLSCHRIFTART = new Font(Font.SANS_SERIF, Font.PLAIN, 12);
    /** Punktzahlenzeichensatz fr Punktzahlen. */
    private static final Font ACHSSCHRIFTART = new Font(Font.SANS_SERIF, Font.BOLD, 12);

    /** Darstellung des rumlichen Balkendiagramms. */
    private DisplayPanel3D display;

    /** Konstruktor. */
    Verlaufsgrafik() {
        super(new BorderLayout());
    }

    /**
     * Startet eine neue Darstellung mit der gegebenen Statistik.
     *
     * @param statistik
     *            Die darzustellende Statistik.
     */
    void starte(Statistik statistik) {
        if (display != null) {
            remove(display);
        }
        Chart3D chart =
                Chart3DFactory.createBarChart(null, null, statistik.datensatz(), null, "Spielnummer", "Punktzahl");
        chart.setAntiAlias(true);
        chart.setLegendBuilder(null);
        chart.setChartBoxColor(WANDFARBE);
        chart.setProjDistance(PROJEKTIONSDISTANZ);
        chart.setViewPoint(new ViewPoint3D(PERSPEKTIVE.getPoint(), PERSPEKTIVE.calcRollAngle()));
        chart.setTranslate2D(VERSCHIEBUNG);
        chart.setBackground(new StandardRectanglePainter(Report.HINTERGRUNDFARBE));
        Chart3DPanel panel = new Chart3DPanel(chart);
        panel.addMouseListener(new Hinweissteuerung());
        panel.addChartMouseListener(new Chart3DMouseListener() {

            @Override
            public void chartMouseMoved(Chart3DMouseEvent event) {
            }

            @Override
            public void chartMouseClicked(Chart3DMouseEvent event) {
                if (event.getTrigger().getClickCount() > 1) {
                    chart.setViewPoint(new ViewPoint3D(PERSPEKTIVE.getPoint(), PERSPEKTIVE.calcRollAngle()));
                    chart.setTranslate2D(VERSCHIEBUNG);
                    panel.zoomToFit();
                }
            }
        });
        display = new DisplayPanel3D(panel, true, false);

        CategoryPlot3D plot = (CategoryPlot3D) chart.getPlot();
        plot.getRowAxis().setVisible(false);
        plot.setGridlineStrokeForValues(GITTERLINIENSTIL);
        plot.setToolTipGenerator(new StandardCategoryItemLabelGenerator(
                "<html><strong>%1$s</strong> hat im %3$d. Spiel <strong>%4$.0f Punkte</strong> erzielt.</html>"));

        StandardCategoryAxis3D turnierachse = (StandardCategoryAxis3D) plot.getColumnAxis();
        turnierachse.setLabelFont(ACHSSCHRIFTART);
        turnierachse.setLabelColor(ACHSSCHRIFTFARBE);
        turnierachse.setTickLabelFont(ZAHLSCHRIFTART);
        turnierachse.setTickLabelColor(ZAHLSCHRIFTFARBE);
        NumberAxis3D punktzahlachse = (NumberAxis3D) plot.getValueAxis();
        punktzahlachse.setLabelFont(ACHSSCHRIFTART);
        punktzahlachse.setLabelColor(ACHSSCHRIFTFARBE);
        punktzahlachse.setTickLabelFont(ZAHLSCHRIFTART);
        punktzahlachse.setTickLabelColor(ZAHLSCHRIFTFARBE);
        punktzahlachse.setAutoRangeIncludeZero(true);
        punktzahlachse.setAutoRangeStickyZero(true);
        punktzahlachse.setRange(WERTEBEREICH);
        plot.setValueAxis(punktzahlachse);

        BarRenderer3D renderer = (BarRenderer3D) plot.getRenderer();
        renderer.setColors(Report.BALKENFARBEN);
        renderer.setTopColorSource(new StandardCategoryColorSource(Color.YELLOW));
        renderer.setBarXWidth(XLAENGE);
        renderer.setBarZWidth(ZLAENGE);
        add(display);

        SwingUtilities.invokeLater(new Runnable() {

            @Override
            public void run() {
                panel.zoomToFit();
            }
        });

    }

}
