/******************************************************************************
 ** $Id: ImHieb.java 1017 2016-05-28 20:35:09Z wmh $
 ** Diese Datei ist Bestandteil der Java-Quelltexte des Wrfelspiels JaFuffy.
 ** Lauffhig ab Java 7.
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************
 ** Die aktuellste Version von JaFuffy findet sich im Internet unter
 ** <http://jafuffy.3kelvin.de>.
 **
 ** Kommentare, Fehler oder Erweiterungswnsche bitte per E-Mail senden an
 ** <jafuffy@3kelvin.de>.
 ******************************************************************************/
package jafuffy.logik.analyse;

import jafuffy.logik.Analyse;
import jafuffy.logik.Spieler;
import jafuffy.logik.Tabzeile;
import jafuffy.logik.Wuerfel;

/** Erstellt Analysen und berechnet Prognosen fr die Im-Hieb-Variante der Spielregeln. */
public class ImHieb extends Klassisch {

    /** Dreierpasch mit der Im-Hieb-Regel. */
    class Dreierpasch extends Klassisch.Dreierpasch {
        /** Zur Serialisierung. */
        private static final long serialVersionUID = 5789475987451295504L;
        /** Mittlerer Im-Hieb-Bonus. */
        private static final double IM_HIEB_BONUS_MITTEL = IM_HIEB_BONUS_MIN * 23. / 108.;

        /** Ermittlung des korrigierten mittleren Punktstandes um den Im-Hieb-Bonus. */
        Dreierpasch() {
            mittel += IM_HIEB_BONUS_MITTEL;
        }

        @Override
        protected void bewerte(Spieler spieler) {
            super.bewerte(spieler);
            if (istImHieb(spieler, wert)) {
                wert += IM_HIEB_BONUS_MIN;
            }
        }
    }

    /** Viererpasch mit der Im-Hieb-Regel. */
    class Viererpasch extends Klassisch.Viererpasch {
        /** Zur Serialisierung. */
        private static final long serialVersionUID = 5628384303623911087L;
        /** Mittlerer Im-Hieb-Bonus. */
        private static final double IM_HIEB_BONUS_MITTEL = IM_HIEB_BONUS_MIN * 13. / 648.;

        /** Ermittlung des korrigierten mittleren Punktstandes um den Im-Hieb-Bonus. */
        Viererpasch() {
            mittel += IM_HIEB_BONUS_MITTEL;
        }

        @Override
        protected void bewerte(Spieler spieler) {
            super.bewerte(spieler);
            if (istImHieb(spieler, wert)) {
                wert += IM_HIEB_BONUS_MIN;
            }
        }

    }

    /** Full House mit der Im-Hieb-Regel. */
    class FullHouse extends Klassisch.FullHouse {
        /** Zur Serialisierung. */
        private static final long serialVersionUID = -3227165375021646796L;
        /** Mittlerer Im-Hieb-Bonus. */
        private static final double IM_HIEB_BONUS_MITTEL = IM_HIEB_BONUS_MIN * 17. / 432.;

        /** Ermittlung des korrigierten mittleren Punktstandes um den Im-Hieb-Bonus. */
        FullHouse() {
            mittel += IM_HIEB_BONUS_MITTEL;
        }

        @Override
        protected void bewerte(Spieler spieler) {
            super.bewerte(spieler);
            if (istImHieb(spieler, wert)) {
                wert += IM_HIEB_BONUS_MIN;
            }
        }
    }

    /** Kleine Strae mit der Im-Hieb-Regel. */
    class KleineStrasse extends Klassisch.KleineStrasse {
        /** Zur Serialisierung. */
        private static final long serialVersionUID = 1874639430274154259L;
        /** Mittlerer Im-Hieb-Bonus. */
        private static final double IM_HIEB_BONUS_MITTEL = IM_HIEB_BONUS_MIN * 25. / 162.;

        /** Ermittlung des korrigierten mittleren Punktstandes um den Im-Hieb-Bonus. */
        KleineStrasse() {
            mittel += IM_HIEB_BONUS_MITTEL;
        }

        @Override
        protected void bewerte(Spieler spieler) {
            super.bewerte(spieler);
            if (istImHieb(spieler, wert)) {
                wert += IM_HIEB_BONUS_MIN;
            }
        }

        @Override
        protected void waehle(Spieler spieler) {
            waehle(liegtKleineStrasseVor());
        }
    }

    /** Groe Strae mit der Im-Hieb-Regel. */
    class GrosseStrasse extends Klassisch.GrosseStrasse {
        /** Zur Serialisierung. */
        private static final long serialVersionUID = -2371420977396166725L;
        /** Mittlerer Im-Hieb-Bonus. */
        private static final double IM_HIEB_BONUS_MITTEL = IM_HIEB_BONUS_MIN * 5. / 162.;

        /** Ermittlung des korrigierten mittleren Punktstandes um den Im-Hieb-Bonus. */
        GrosseStrasse() {
            mittel += IM_HIEB_BONUS_MITTEL;
        }

        @Override
        protected void bewerte(Spieler spieler) {
            super.bewerte(spieler);
            if (istImHieb(spieler, wert)) {
                wert += IM_HIEB_BONUS_MIN;
            }
        }
    }

    /** JaFuffy mit der Im-Hieb-Regel. */
    class JaFuffy extends Klassisch.JaFuffy {
        /** Zur Serialisierung. */
        private static final long serialVersionUID = -4640248861129674120L;
        /** Mittlerer Im-Hieb-Bonus. */
        private static final double IM_HIEB_BONUS_MITTEL = IM_HIEB_BONUS_MAX / 1296.;

        /** Ermittlung des korrigierten mittleren Punktstandes um den Im-Hieb-Bonus. */
        JaFuffy() {
            mittel += IM_HIEB_BONUS_MITTEL;
        }

        @Override
        protected void bewerte(Spieler spieler) {
            super.bewerte(spieler);
            if (istImHieb(spieler, wert)) {
                wert += IM_HIEB_BONUS_MAX;
            }
        }
    }

    /** Alle Eintrge fr die klassische Variante der Spielregeln als Tabelle zusammengefasst. */
    class Eintragungen extends Analyse.Eintragungen {
        /** Erlaubt Serialisierung. */
        private static final long serialVersionUID = -226404933174352144L;

        /** Konstruktion der Tabelle. */
        Eintragungen() {
            put(Tabzeile.DREIERPASCH, new Dreierpasch());
            put(Tabzeile.VIERERPASCH, new Viererpasch());
            put(Tabzeile.FULLHOUSE, new FullHouse());
            put(Tabzeile.KLEINESTRASSE, new KleineStrasse());
            put(Tabzeile.GROSSESTRASSE, new GrosseStrasse());
            put(Tabzeile.JAFUFFY, new JaFuffy());
        }
    }

    /** Erlaubt Serialisierung. */
    private static final long serialVersionUID = 447027133112163377L;

    /** Normaler Im-Hieb-Bonus, auer JaFuffy. */
    private static final int IM_HIEB_BONUS_MIN = 5;
    /** Maximal Im-Hieb-Bonus, bei JaFuffy. */
    private static final int IM_HIEB_BONUS_MAX = 30;

    /**
     * Konstruktor.
     *
     * @param wuerfel
     *            Wrfel, mit denen das Turnier durchgefhrt wird.
     */
    public ImHieb(Wuerfel[] wuerfel) {
        super(wuerfel);
    }

    /**
     * @param spieler
     *            Der Spieler, dessen Wurf berprft wird.
     * @param eintrag
     *            Der Eintrag, der berprft wird.
     * @return Gibt an, ob fr den angegebenen Eintrag ein Ergebnis im Hieb erzielt wurde.
     */
    public boolean istImHieb(Spieler spieler, Tabzeile eintrag) {
        return eintrag.unten() && eintrag != Tabzeile.CHANCE && istImHieb(spieler, wert(eintrag));
    }

    /**
     * @param spieler
     *            Der Spieler, dessen Wurf berprft wird.
     * @param wert
     *            Die erzielte Punktzahl fr einen Eintrag.
     * @return Gibt an, ob fr den angegebenen Eintrag ein Ergebnis im Hieb erzielt wurde.
     */
    private boolean istImHieb(Spieler spieler, int wert) {
        return spieler.einmalig() && wert > 0;
    }

    @Override
    protected Analyse.Eintragungen eintragungen() {
        return new Eintragungen();
    }

}
