/******************************************************************************
 **  $Id: Variante.java 1167 2017-04-09 23:50:10Z wmh $
 ** Diese Datei ist Bestandteil der Java-Quelltexte des Wrfelspiels JaFuffy.
 ** Lauffhig ab Java 7.
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************
 ** Die aktuellste Version von JaFuffy findet sich im Internet unter
 ** <http://jafuffy.3kelvin.de>.
 **
 ** Kommentare, Fehler oder Erweiterungswnsche bitte per E-Mail senden an
 ** <jafuffy@3kelvin.de>.
 ******************************************************************************/
package jafuffy.logik;

import jafuffy.bedienung.Auswahl;
import jafuffy.logik.auswertung.Erweitert;
import jafuffy.logik.auswertung.ImHieb;
import jafuffy.logik.auswertung.Klassisch;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;

/**
 * Varianten mit unterschiedlichen Spielregeln und passenden Mglichkeiten zur Auswertung und Auswahl.
 */
public enum Variante {

    /** Klassische Regeln. */
    KLASSISCH(Klassisch.REGEL, Klassisch.class, Klassisch.AUSWAHLKLASSE),
    /** Im-Hieb-Regeln. */
    IMHIEB(ImHieb.REGEL, ImHieb.class, ImHieb.AUSWAHLKLASSE),
    /** Erweiterte Regeln mit Joker und Bonus. */
    ERWEITERT(Erweitert.REGEL, Erweitert.class, Erweitert.AUSWAHLKLASSE);

    /** Anzahl der untersttzten Varianten. */
    public static final int ANZAHL = values().length;

    /** Spielregelname. */
    private final String spielregelname;
    /** Konstruktor, welcher das passende Auswertungsobjekt erzeugt. */
    private final Constructor<? extends Auswertung> auswertungskonstruktor;
    /** Konstruktor, welcher das passende Auswahlobjekt erzeugt. */
    private final Constructor<? extends Auswahl> auswahlkonstruktor;

    /**
     * Erstellt eine Spielregelvariante.
     *
     * @param spielregelname
     *            Name der Spielregel.
     * @param auswertungsklasse
     *            Auswertungsklasse.
     * @param auswahlklasse
     *            Auswahlklasse.
     */
    Variante(String spielregelname, Class<? extends Auswertung> auswertungsklasse,
            Class<? extends Auswahl> auswahlklasse) {
        this.spielregelname = spielregelname;
        this.auswertungskonstruktor = auswertungskonstruktor(auswertungsklasse);
        this.auswahlkonstruktor = auswahlkonstruktor(auswahlklasse);
    }

    /**
     * Erzeugt Auswahlobjekt.
     *
     * @param turnier
     *            Turnier, fr das die Auswahl Auswirkung zeigt.
     *
     * @return Auswahlobjekt.
     */
    public Auswahl auswahl(Turnier turnier) {
        try {
            return auswahlkonstruktor.newInstance(new Object[] { turnier });
        } catch (InstantiationException | IllegalAccessException | IllegalArgumentException
                | InvocationTargetException exception) {
            exception.printStackTrace();
            return null;
        }
    }

    /**
     * Erzeugt Auswertungsobjekt.
     *
     * @param wuerfel
     *            Die Wrfel, mit denen gespielt wird.
     *
     * @return Auswertungsobjekt.
     */
    public Auswertung auswertung(Wuerfel[] wuerfel) {
        try {
            return auswertungskonstruktor.newInstance(new Object[] { wuerfel });
        } catch (InstantiationException | IllegalAccessException | IllegalArgumentException
                | InvocationTargetException exception) {
            exception.printStackTrace();
            return null;
        }
    }

    @Override
    public String toString() {
        return spielregelname;
    }

    /**
     * @param klasse
     *            Klasse, aus welcher der Konstruktor erzeugt wird.
     * @return Konstruktor.
     */
    private Constructor<? extends Auswahl> auswahlkonstruktor(Class<? extends Auswahl> klasse) {
        try {
            return klasse.getConstructor(Turnier.class);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * @param klasse
     *            Klasse, aus welcher der Konstruktor erzeugt wird.
     * @return Konstruktor.
     */
    private Constructor<? extends Auswertung> auswertungskonstruktor(Class<? extends Auswertung> klasse) {
        try {
            return klasse.getConstructor(Wuerfel[].class);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }

}
