/******************************************************************************
 ** $Id: Spieler.java 1180 2017-04-14 23:53:55Z wmh $
 ** Diese Datei ist Bestandteil der Java-Quelltexte des Wrfelspiels JaFuffy.
 ** Lauffhig ab Java 7.
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************
 ** Die aktuellste Version von JaFuffy findet sich im Internet unter
 ** <http://jafuffy.3kelvin.de>.
 **
 ** Kommentare, Fehler oder Erweiterungswnsche bitte per E-Mail senden an
 ** <jafuffy@3kelvin.de>.
 ******************************************************************************/
package jafuffy.logik;

import java.io.Serializable;
import java.util.HashMap;

import jafuffy.logik.Punkte.Meldung;

/** Spieler; enthlt Punktzahlen und weitere Daten, die den Stand beschreiben. */
public class Spieler implements Comparable<Spieler>, Cloneable, Serializable {

    /** Objekt ist serialisierbar. */
    private static final long serialVersionUID = -5170511106942700515L;

    /** Maximale Anzahl der Spieler. */
    public static final int SPIELER = 4;
    /** Grenzpunktzahl fr Bonus. */
    static final int GRENZE = 63;
    /** Bonus. */
    static final int BONUS = 35;

    /** Spielername. */
    private final String name;
    /** Die maximale Anzahl der Versuche, die einem Spieler zum Wrfeln zur Verfgung stehen. */
    private final int max;
    /** Punktstnde inklusive Zwischensummen und Endsumme. */
    private final HashMap<Tabzeile, Integer> punktzahlen = new HashMap<>(Tabzeile.ANZAHL);
    /** Karte zur Ermittlung der freien, setzbaren Eintrge. */
    private final HashMap<Tabzeile, Boolean> setzbarkeiten = new HashMap<>(Tabzeile.ANZAHL);
    /** Anzahl der dem Spieler zur Verfgung stehenden Restwrfe. */
    private int rest;
    /** Gibt an, ob der Spieler schon gewuerfelt hat. Nur der aktive Spieler kann wrfeln! . */
    private boolean gewuerfelt;
    /** Spieler hat genau einmal gewrfelt. */
    private boolean einmalig;
    /** Zurckmeldung ber nderungen von Daten an Punktemodell. */
    private transient Meldung meldung;

    /**
     * Konstruktor.
     *
     * @param name
     *            Name des Spielers.
     * @param max
     *            Die maximale Anzahl der Versuche, die einem Spieler zum Wrfeln zur Verfgung stehen.
     */
    public Spieler(String name, int max) {
        this.name = name;
        this.max = max;
        initialisiere();
    }

    /** @return Vorzeichen bei Vergleich fr Endsummen von Spielern. */
    @Override
    public int compareTo(Spieler spieler) {
        if (endsumme() < spieler.endsumme()) {
            return -1;
        }
        if (endsumme() > spieler.endsumme()) {
            return +1;
        }
        return 0;
    }

    /** @return Hat der Spieler genau einmal gewrfelt? */
    public boolean einmalig() {
        return einmalig;
    }

    /** @return Endsumme */
    public int endsumme() {
        return punktzahlen.get(Tabzeile.ESUMME);
    }

    /**
     * @return Gibt an, ob der Spieler fertig ist mit seinen Versuchen hinsichtlich der Anzahl von Wrfen.
     */
    public boolean fertig() {
        return 0 == rest;
    }

    /** @return Hat der Spieler gewrfelt? */
    public boolean gewuerfelt() {
        return gewuerfelt;
    }

    /** @return Gibt an, ob Spieler menschlich ist. */
    public boolean menschlich() {
        return true;
    }

    /** @return Gibt an, ob dem Spieler noch alle Versuche zum Wrfeln zur Verfgung stehen. */
    public boolean neu() {
        return max == rest;
    }

    /**
     * @param eintrag
     *            Nummer des Eintrags auf dem Punktezettel
     * @return Schon erwrfelte Punktzahl.
     */
    public int punkte(Tabzeile eintrag) {
        return punktzahlen.get(eintrag);
    }

    /** @return Wie oft darf der Spieler noch wrfeln? */
    public int rest() {
        return rest;
    }

    /**
     * Setzbarkeit fr einen Eintrag zurckholen.
     *
     * @param eintrag
     *            Nummer des Eintrags auf dem Punktezettel
     * @return Ist der Eintrag noch setzbar?
     */
    public boolean setzbar(Tabzeile eintrag) {
        return setzbarkeiten.get(eintrag);
    }

    /**
     * Schreibe erwrfelte Punktzahl und warte auf nchste Spieleraktion.
     *
     * @param eintrag
     *            Eintragszeile auf dem Punktezettel.
     * @param wert
     *            Einzutragender Wert.
     * @param weiter
     *            Gibt an, ob der Spieler die Wrfel zum nchsten Spieler weiter gibt.
     */
    public void setze(Tabzeile eintrag, int wert, boolean weiter) {
        int oben = 0;
        int bonus = 0;
        int unten = 0;

        punktzahlen.put(eintrag, wert);
        setzbarkeiten.put(eintrag, false);

        for (Tabzeile z : Tabzeile.OBEN) {
            oben += punktzahlen.get(z);
        }
        punktzahlen.put(Tabzeile.GESAMT, oben);
        if (punktzahlen.get(Tabzeile.GESAMT) >= GRENZE) {
            bonus = BONUS;
        }
        punktzahlen.put(Tabzeile.BONUS, bonus);
        punktzahlen.put(Tabzeile.OSUMME, oben + bonus);
        for (Tabzeile z : Tabzeile.UNTEN) {
            unten += punktzahlen.get(z);
        }
        punktzahlen.put(Tabzeile.USUMME, unten);
        punktzahlen.put(Tabzeile.ESUMME, oben + bonus + unten);

        if (weiter) {
            warte();
        } else {
            aktualisiere();
        }
    }

    /** * @return Spielername */
    @Override
    public String toString() {
        return name;
    }

    /** Dieser Spieler wird aktiviert, um fr diese Runde seine Spielaktionen durchzufhren. */
    void aktiviere() {
        gewuerfelt = false;
        einmalig = false;
        rest = max;
        aktualisiere();
    }

    /** nderungen ber Daten an verbundenes Punktemodell zurckmelden. */
    void aktualisiere() {
        if (meldung != null) {
            meldung.mitteilen();
        }
    }

    /** @return Gesamtsumme */
    int gesamt() {
        return punktzahlen.get(Tabzeile.GESAMT);
    }

    /** Setze auf Initialzustand zurck, um neue Spielrunde beginnen. */
    void initialisiere() {
        for (Tabzeile z : Tabzeile.values()) {
            punktzahlen.put(z, 0);
            switch (z) {
            case GESAMT:
            case BONUS:
            case OSUMME:
            case USUMME:
            case ESUMME:
                setzbarkeiten.put(z, false);
                break;
            default:
                setzbarkeiten.put(z, true);
            }
        }
        warte();
    }

    /**
     * @return Gewnschte Unsicherheit der Prognose zur Steuerung der Spielstrke des Bots. 0 ist ganz sicher, 1 ist
     *         vllig unsicher.
     */
    double prognoseunsicherheit() {
        return 0;
    }

    /**
     * Sichert Spielerdaten.
     *
     * @param
     */
    void sichere(Spieler sicherung) {
        sicherung.punktzahlen.putAll(punktzahlen);
        sicherung.setzbarkeiten.putAll(setzbarkeiten);
        sicherung.rest = 0;
        sicherung.gewuerfelt = true;
        sicherung.einmalig = einmalig;
        sicherung.meldung = meldung;
    }

    /** Reagiert auf einen Wurf des Spielers. */
    void reagiere() {
        gewuerfelt = true;
        einmalig = neu();
        rest--;
        aktualisiere();
    }

    /**
     * Verbinde mit dem Punktemodell, welches Spielerdaten beobachten mchte.
     *
     * @param meldung
     *            Das Meldungsobjekt, welches von dem Punktemodell verwendet wird.
     */
    void verbinde(Meldung meldung) {
        this.meldung = meldung;
    }

    /** Der Spieler wartet, bis er an die Reihe kommt. */
    void warte() {
        gewuerfelt = false;
        einmalig = false;
        rest = 0;
        aktualisiere();
    }

}
