/******************************************************************************
 ** $Id: Zwischenstand.java 1017 2016-05-28 20:35:09Z wmh $
 ** Diese Datei ist Bestandteil der Java-Quelltexte des Wrfelspiels JaFuffy.
 ** Lauffhig ab Java 7.
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************
 ** Die aktuellste Version von JaFuffy findet sich im Internet unter
 ** <http://jafuffy.3kelvin.de>.
 **
 ** Kommentare, Fehler oder Erweiterungswnsche bitte per E-Mail senden an
 ** <jafuffy@3kelvin.de>.
 ******************************************************************************/
package jafuffy.bedienung;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.RenderingHints;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.awt.event.MouseEvent;
import java.awt.event.MouseMotionAdapter;
import java.awt.geom.Rectangle2D;

import javax.swing.JPanel;

import jafuffy.logik.Spieler;
import jafuffy.logik.Statistik;

/** Balkendiagramm zur Anzeige aller erwrfelten Punkte, welches nach jeder Runde aktualisiert werden kann. */
@SuppressWarnings("serial")
class Zwischenstand extends JPanel {

    /** Der Standardhilfetext im Balkendiagramm, falls kein Balken berfahren wird. */
    private static final String STANDARDHILFE = "Alle bisher im Turnier erspielten Punkte der einzelnen Spieler";
    /** Gesamtbreite des Balkendiagramms. */
    private static final int BREITE = 400;
    /** Dicke der Balken. */
    private static final int BALKENDICKE = 15;
    /** Strke des rumlichen Effektes bei der Darstellung der Balken. */
    private static final int RAUMEFFEKT = 2;
    /** Lcke zwischen den Balken. */
    private static final int LUECKE = 6 + RAUMEFFEKT;
    /** Schriftgre der Balkenbeschriftung. */
    private static final int SCHRIFTGROESSE = 12;
    /** Font fr die Schriftgre. */
    private static final Font FONT = new Font("Dialog", Font.BOLD, SCHRIFTGROESSE);

    /** Zu dieser Statistik wird der Zwischenstand dargestellt. */
    private final Statistik statistik;
    /** Erwartete Punktzahl eines Spieler fr das ganze Turnier zur relativen Darstellung. */
    private int ausschnitt;
    /**
     * Zwischenspeicher fr die Stnde der einzelnen Spieler, kann nach jeder Runde auf die tatschliche Zwischenstnde
     * gebracht werden.
     */
    private int[] staende;
    /** Die Namen der Spieler. */
    private String[] namen;
    /** Die Rechtecke, in denen sich die sichtbaren Vorderseiten der Balken befinden. */
    private Rectangle2D[] balkenrechtecke;
    /** Die Rechtecke, in denen sich die Beschriftungen befinden. */
    private Rectangle2D[] beschriftungsrechtecke;

    /**
     * Konstruktor.
     *
     * @param statistik
     *            Die Statistik, mit der das Balkendiagramm aktualisiert wird.
     */
    Zwischenstand(Statistik statistik) {
        this.statistik = statistik;
        setBackground(Color.WHITE);
        setToolTipText(STANDARDHILFE);
        addComponentListener(new ComponentAdapter() {
            @Override
            public void componentResized(ComponentEvent event) {
                verteile();
                repaint();
            }
        });
        addMouseMotionListener(new MouseMotionAdapter() {
            @Override
            public void mouseMoved(MouseEvent event) {
                beschreibe(event);
            }
        });
    }

    @Override
    public Dimension getPreferredSize() {
        if (staende != null) {
            // Die tatschliche Gre fr das laufende Turnier.
            return new Dimension(BREITE, staende.length * (BALKENDICKE + LUECKE) + LUECKE);
        } else {
            // Vorhalt fr die maximal mgliche Gre, welche bei Konstruktion bestimmt wird.
            return new Dimension(BREITE, Spieler.SPIELER * (BALKENDICKE + LUECKE) + LUECKE);
        }
    }

    @Override
    public void paintComponent(Graphics g) {
        super.paintComponent(g);
        if (balkenrechtecke != null) {
            g.setFont(FONT);
            ((Graphics2D) g).setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
            final int n = staende.length;
            for (int i = 0; i < n; i++) {
                final int l = (int) balkenrechtecke[i].getWidth();
                final int b = (int) balkenrechtecke[i].getHeight();
                final int x = (int) balkenrechtecke[i].getX();
                final int y = ((int) balkenrechtecke[i].getY()) - (RAUMEFFEKT - 1);
                zeichne(g, x, y, l, b, Report.FARBEN[i]);
                final int d = (int) beschriftungsrechtecke[i].getX(); // Distanz
                final int m = (int) beschriftungsrechtecke[i].getY() + (int) beschriftungsrechtecke[i].getHeight() - 1;
                g.drawString(String.valueOf(staende[i]), d, m);
            }
        }
    }

    /** Passt den Ausschnitt an den aktuellen Stand an. */
    private void anpasse() {
        boolean istAusschnittUeberschritten = false;
        for (int i = 0; i < staende.length; i++) {
            staende[i] = statistik.stand(i);
            if (staende[i] > ausschnitt) {
                istAusschnittUeberschritten = true;
                ausschnitt = staende[i];
            }
        }
        if (istAusschnittUeberschritten) {
            if (statistik.anzahl() == 0) {
                ausschnitt *= 2;
            } else {
                int anzahl = statistik.rest();
                ausschnitt += statistik.abweichung() + statistik.mittelwert() * anzahl
                        + 2 * statistik.abweichung() * ((int) Math.sqrt(anzahl));
            }
        }
    }

    /**
     * Beschreibt mittels Kontexthilfe die Balken in Abhngigkeit der Mausposition.
     *
     * @param event
     *            Das Mausereignis, welches die Mausposition angibt.
     */
    private void beschreibe(MouseEvent event) {
        setToolTipText(STANDARDHILFE);
        for (int i = 0; i < staende.length; i++) {
            if (balkenrechtecke[i].contains(event.getX(), event.getY())
                    || beschriftungsrechtecke[i].contains(event.getX(), event.getY())) {
                setToolTipText("<html>" + "<p>" + "Alle bisher im Turnier erspielten Punkte von " + "<strong>"
                        + namen[i] + "</strong>" + "</p>" + "</html>");
            }
        }
    }

    /** Verteilt die Balken ber das Zeichenfeld und merkt sich die Balkenrechtecke. */
    private void verteile() {
        if (balkenrechtecke != null) {
            final int n = staende.length;
            for (int i = 0; i < n; i++) {
                final FontMetrics fm = getFontMetrics(FONT);
                final int s = fm.stringWidth(String.valueOf(ausschnitt));
                final int w = fm.charWidth(' ');
                final int a = fm.getAscent();
                final int l = (getWidth() - s - 2 * w - RAUMEFFEKT) * staende[i] / ausschnitt;
                final int y = (getHeight() - n * BALKENDICKE) * (i + 1) / (n + 1) + i * BALKENDICKE;
                balkenrechtecke[i].setRect(0, y + RAUMEFFEKT - 1, l, BALKENDICKE);
                beschriftungsrechtecke[i].setRect(l + RAUMEFFEKT + w, y + RAUMEFFEKT - 2 + (BALKENDICKE - a + 1) / 2, s,
                        a);
            }
        }
    }

    /**
     * Zeichnet einen Balken mit angedeutetem rumlichen Effekt.
     *
     * @param g
     *            Das Grafikobjekt, auf dem die Aktion stattfindet.
     * @param x
     *            Linke obere Ecke.
     * @param y
     *            Rechte obere Ecke.
     * @param b
     *            Breite ohne Effekte.
     * @param l
     *            Lnge ohne Effekte.
     * @param farbe
     *            Balkenfarbe.
     */
    private void zeichne(Graphics g, int x, int y, int b, int l, Color farbe) {
        g.setColor(farbe);
        for (int i = 0; i < RAUMEFFEKT; i++) {
            g.fill3DRect(x + RAUMEFFEKT - 1 - i, y + i, b, l, true);
        }
    }

    /** Aktualisiert das Balkendiagramm mit den aktuellen Gesamtpunktstnden. */
    void aktualisiere() {
        anpasse();
        verteile();
    }

    /** Startet die Darstellung von vorn, nmlich fr einen Neubeginn eines Turniers. */
    void starte() {
        final int n = statistik.spieler().size();
        staende = new int[n];
        namen = new String[staende.length];
        balkenrechtecke = new Rectangle2D[staende.length];
        beschriftungsrechtecke = new Rectangle2D[staende.length];
        final int anzahl = Integer.max(1, statistik.anzahl());
        for (int i = 0; i < staende.length; i++) {
            namen[i] = statistik.spieler().get(i).toString();
            balkenrechtecke[i] = new Rectangle2D.Double();
            beschriftungsrechtecke[i] = new Rectangle2D.Double();
        }
        // Ausschnitt berechnet sich aus Mittelwert plus doppelter Standardabweichung
        ausschnitt = statistik.mittelwert() * anzahl + 2 * statistik.abweichung() * ((int) Math.sqrt(anzahl));
        aktualisiere();
        revalidate();
    }

}
