/******************************************************************************
 ** $Id: Report.java 1017 2016-05-28 20:35:09Z wmh $
 ** Diese Datei ist Bestandteil der Java-Quelltexte des Wrfelspiels JaFuffy.
 ** Lauffhig ab Java 7.
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************
 ** Die aktuellste Version von JaFuffy findet sich im Internet unter
 ** <http://jafuffy.3kelvin.de>.
 **
 ** Kommentare, Fehler oder Erweiterungswnsche bitte per E-Mail senden an
 ** <jafuffy@3kelvin.de>.
 ******************************************************************************/
package jafuffy.bedienung;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.GridLayout;
import java.awt.Insets;
import java.awt.Rectangle;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTabbedPane;
import javax.swing.ScrollPaneConstants;
import javax.swing.SwingConstants;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;
import javax.swing.border.Border;
import javax.swing.border.TitledBorder;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.MouseInputAdapter;
import javax.swing.event.MouseInputListener;

import jafuffy.logik.CEJaFuffy;
import jafuffy.logik.CEStatistik;
import jafuffy.logik.Statistik;

/** Report ber den Verlauf des Turniers. */
@SuppressWarnings("serial")
public class Report extends JDialog implements PropertyChangeListener, ChangeListener {

    /** Gewnschte Breite des Reports. */
    private static final int BREITE = 480;
    /** Gewnschte Hhe des Reports. */
    private static final int HOEHE = 480;
    /** Abstand von Elemente zum Dialograhmen. */
    private static final int DIALOGRAHMENABSTAND = 8;
    /** Abstand von Element zu Feldrahmen. */
    private static final int FELDRAHMENABSTAND = 4;
    /** Rahmen fr Hauptgrafiken, also Zwischenstand und Verlauf. */
    private static final Border RAHMEN = BorderFactory.createCompoundBorder(BorderFactory.createRaisedBevelBorder(),
            BorderFactory.createLoweredBevelBorder());

    /** Farben fr Balkengrafiken, Index entspricht Spieler im Turnier. */
    static final Color[] FARBEN =
            { Color.MAGENTA.darker(), Color.GREEN.darker(), Color.ORANGE.darker(), Color.CYAN.darker() };

    /** Statistische Daten, die hier im Verlauf fr ein Turnier dargestellt werden sollen. */
    private final Statistik statistik;
    /** Die Legende mit dem Farbcode zu den Darstellungen. */
    private final JPanel legende = new JPanel(new GridLayout(0, 1, 0, DIALOGRAHMENABSTAND));
    /** Grafische Darstellung des Turnierverlaufs. */
    private final Verlaufsgrafik verlaufsgrafik;
    /** Tabellarische Darstellung des Turnierverlaufs. */
    private final Verlaufstabelle verlaufstabelle;
    /** Zwischenstand ber den Gesamtpunktstand. */
    private final Zwischenstand zwischenstand;

    /**
     * Konstruktor.
     *
     * @param statistik
     *            Statistische Daten, die hier im Verlauf fr ein Turnier dargestellt werden sollen.
     */
    public Report(Statistik statistik) {

        this.statistik = statistik;
        UIManager.addPropertyChangeListener(this);

        zwischenstand = new Zwischenstand(statistik);

        JScrollPane zwischenstandscrolling = new JScrollPane(zwischenstand);
        zwischenstandscrolling.setBorder(
                BorderFactory.createTitledBorder(RAHMEN, "(Zwischen)Stand im bisherigen Verlauf des Turniers",
                        TitledBorder.DEFAULT_JUSTIFICATION, TitledBorder.ABOVE_TOP));

        JScrollPane verlaufsgrafikscrolling = new JScrollPane();
        verlaufsgrafikscrolling.setBorder(RAHMEN);
        verlaufsgrafik = verlaufsgrafik(verlaufsgrafikscrolling);

        verlaufstabelle = new Verlaufstabelle(statistik);
        JPanel verlaufstabellenfeld = verlaufstabellenfeld();

        JTabbedPane verlauf = new JTabbedPane();
        verlauf.addTab("Balkendiagramm zum Verlauf", verlaufsgrafikscrolling);
        verlauf.addTab("Tabellenauflistung des Verlaufs", verlaufstabellenfeld);
        verlauf.setBorder(BorderFactory.createEmptyBorder(0, FELDRAHMENABSTAND, 0, FELDRAHMENABSTAND));

        setContentPane(feld(zwischenstandscrolling, verlauf));
        setTitle(titel());
        setIconImages(Oberflaeche.LOGOS);
        setPreferredSize(new Dimension(BREITE, HOEHE));
        pack();

    }

    @Override
    public void propertyChange(PropertyChangeEvent evt) {
        if (evt.getPropertyName().equals("lookAndFeel")) {
            SwingUtilities.updateComponentTreeUI(this);
        }
    }

    @Override
    public void stateChanged(ChangeEvent ce) {
        if (CEJaFuffy.adressiert(ce, CEStatistik.class)) {
            CEJaFuffy<CEStatistik> cej = CEJaFuffy.ereignisbehaelter(ce);
            switch (cej.ereignis()) {
            case TURNIER_START:
            case TURNIER_WEITER:
                starte();
                zwischenstand.starte();
                verlaufsgrafik.starte();
                verlaufstabelle.starte();
                revalidate();
                break;
            case SPIEL_ENDE:
                verlaufsgrafik.beende();
                verlaufstabelle.beende();
                break;
            case STAND:
                zwischenstand.aktualisiere();
                verlaufsgrafik.aktualisiere();
                verlaufstabelle.aktualisiere();
                break;
            default:
                break;
            }
            setTitle(titel());
            repaint();
        }
    }

    /**
     * Baut das Hauptfeld fr den Report zusammen, welche die Verlaufsdarstellungen enthlt.
     *
     * @param zwischenstandscrolling
     *            Stellt den Zwischenstand dar.
     * @param verlauf
     *            Enthlt die Darstellungen fr den Verlauf in grafischer und tabellarischer Form dar.
     * @return Das Hauptfeld.
     */
    private JPanel feld(JScrollPane zwischenstandscrolling, JTabbedPane verlauf) {
        JPanel feld = new JPanel(new BorderLayout(0, FELDRAHMENABSTAND));
        feld.add(zwischenstandscrolling, BorderLayout.PAGE_START);
        feld.add(verlauf, BorderLayout.CENTER);
        feld.add(leiste(), BorderLayout.PAGE_END);
        feld.setBorder(BorderFactory.createEmptyBorder(DIALOGRAHMENABSTAND, DIALOGRAHMENABSTAND, DIALOGRAHMENABSTAND,
                DIALOGRAHMENABSTAND));
        return feld;
    }

    /**
     * Erstellt die Besttigungsleiste fr die Kontrolle durch den Benutzer.
     *
     * @return Leistenfeld.
     */
    private JPanel leiste() {
        JButton schliessen = new JButton("Report schlieen");
        schliessen.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent event) {
                setVisible(false);
            }
        });
        getRootPane().setDefaultButton(schliessen);
        JButton zeigen = new JButton("Legende zeigen");
        zeigen.setToolTipText("Zeigt die Zuordnung von Farben zu Spielern");
        zeigen.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent event) {
                JOptionPane.showMessageDialog(Report.this, legende, "JaFuffy (Legende)",
                        JOptionPane.INFORMATION_MESSAGE);
            }
        });
        JPanel leiste = new JPanel(new GridLayout(1, 2, DIALOGRAHMENABSTAND, 0));
        leiste.add(schliessen);
        leiste.add(zeigen);
        return leiste;
    }

    /** Passt die Legende an ein neues Turnier an. */
    private void starte() {
        JLabel name;
        legende.removeAll();
        for (int i = 0; i < statistik.spieler().size(); i++) {
            name = new JLabel(statistik.spieler().get(i).toString(), SwingConstants.CENTER);
            legende.add(name);
            name.setOpaque(true);
            name.setForeground(FARBEN[i]);
            name.setBackground(Color.white);
            name.setBorder(BorderFactory.createCompoundBorder(BorderFactory.createLineBorder(FARBEN[i], 2),
                    BorderFactory.createEmptyBorder(FELDRAHMENABSTAND / 2, FELDRAHMENABSTAND, FELDRAHMENABSTAND / 2,
                            FELDRAHMENABSTAND)));
        }
    }

    /**
     * Ermittelt den Titel des Dialogfensters.
     *
     * @return Fenstertitel.
     */
    private String titel() {
        if (statistik.verlauf() == null) {
            return "JaFuffy (Report)";
        } else if (statistik.istTurnierLaufend()) {
            if (statistik.anzahl() == 0 || statistik.anzahl() > 1) {
                return "JaFuffy (Report: " + statistik.verlauf()[0].size() + ". Spiel im Turnier)";
            } else {
                return "JaFuffy (Report: Einziges Spiel im Turnier)";
            }
        } else {
            return "JaFuffy (Report: Turnierendergebnis)";
        }
    }

    /**
     * Baut die Verlaufsgrafik zusammen.
     *
     * @param verlaufsgrafikscrolling
     *            bernimmt das Scrolling fr das Grafikfeld.
     * @return Die Verlaufsgrafik.
     */
    private Verlaufsgrafik verlaufsgrafik(JScrollPane verlaufsgrafikscrolling) {
        MouseInputListener mouseInputListener = new MouseInputAdapter() {
            private int x;
            private int y;

            @Override
            public void mouseDragged(MouseEvent event) {
                Rectangle rechteck = verlaufsgrafikscrolling.getViewport().getViewRect();
                rechteck.translate(x - event.getX(), y - event.getY());
                ((JPanel) event.getSource()).scrollRectToVisible(rechteck);
            }

            @Override
            public void mousePressed(MouseEvent e) {
                x = e.getX();
                y = e.getY();
            }
        };
        Verlaufsgrafik grafik = new Verlaufsgrafik(statistik);
        grafik.addMouseMotionListener(mouseInputListener);
        grafik.addMouseListener(mouseInputListener);
        verlaufsgrafikscrolling.setViewportView(grafik);
        return grafik;
    }

    /**
     * Stellt das Feld (samt Scrolleinrichtung) zusammen, welches die Verlaufstabelle enthlt.
     *
     * @return Verlaufstabellenfeld.
     */
    private JPanel verlaufstabellenfeld() {
        JPanel verlaufstabellenfeld = new JPanel();
        verlaufstabellenfeld.setBorder(RAHMEN);
        JScrollPane verlaufstabellenscrolling = new JScrollPane(verlaufstabelle) {
            @Override
            public Dimension getPreferredSize() {
                Insets rand = getInsets();
                Insets verlaufstabellenfeldrand = verlaufstabellenfeld.getInsets();
                Dimension verlaufstabellengroesse = verlaufstabelle.getPreferredSize();
                Dimension verlaufstabellenfeldgroesse = verlaufstabellenfeld.getSize();
                return new Dimension(
                        verlaufstabellengroesse.width + getVerticalScrollBar().getWidth() + rand.left + rand.right,
                        verlaufstabellenfeldgroesse.height - verlaufstabellenfeldrand.top
                                - verlaufstabellenfeldrand.bottom - rand.top - 1 - rand.bottom - 1);
            }
        };
        verlaufstabellenscrolling.setVerticalScrollBarPolicy(ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS);
        verlaufstabellenfeld.add(verlaufstabellenscrolling);
        return verlaufstabellenfeld;
    }

}
