/******************************************************************************
 ** $Id: Verlaufstabelle.java 1017 2016-05-28 20:35:09Z wmh $
 ** Diese Datei ist Bestandteil der Java-Quelltexte des Wrfelspiels JaFuffy.
 ** Lauffhig ab Java 7.
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software; you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation; either version 2 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program; if not, write to the Free Software
 ** Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 ******************************************************************************
 ** Die aktuellste Version von JaFuffy findet sich im Internet unter
 ** <http://jafuffy.3kelvin.de>.
 **
 ** Kommentare, Fehler oder Erweiterungswnsche bitte per E-Mail senden an
 ** <jafuffy@3kelvin.de>.
 ******************************************************************************/
package jafuffy.bedienung;

import java.awt.Color;
import java.awt.Component;
import java.awt.Font;
import java.awt.event.MouseEvent;
import java.util.ArrayList;
import java.util.List;

import javax.swing.BorderFactory;
import javax.swing.JLabel;
import javax.swing.JTable;
import javax.swing.RowSorter.SortKey;
import javax.swing.SwingConstants;
import javax.swing.SwingUtilities;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.JTableHeader;
import javax.swing.table.TableCellRenderer;

import jafuffy.logik.Statistik;

/** Stellt den Turnierverlauf als Tabelle dar. */
@SuppressWarnings("serial")
class Verlaufstabelle extends JTable {

    /** Wandelt die Statistik in ein Modell fr die Tabelle um. */
    private static class Adapter extends AbstractTableModel {
        /** Statistische Daten, die hier im Verlauf fr ein Turnier dargestellt werden sollen. */
        private final Statistik statistik;

        /**
         * Konstruktor.
         *
         * @param statistik
         *            Statistische Daten, welche in ein Tabellenmodell umgewandelt werden soll.
         */
        Adapter(Statistik statistik) {
            this.statistik = statistik;
        }

        @Override
        public Class<?> getColumnClass(int spalte) {
            return Integer.class;
        }

        @Override
        public int getColumnCount() {
            // bercksichtigt erste Spalte, welche die laufende Spielnummer enthlt
            return verlauf() != null ? 1 + verlauf().length : 1;
        }

        @Override
        public String getColumnName(int spalte) {
            if (spalte > 0) {
                return statistik.spieler().get(spalte - 1).toString();
            } else {
                return "Nummer";
            }
        }

        @Override
        public int getRowCount() {
            return verlauf() != null ? verlauf()[0].size() : 0;
        }

        @Override
        public Object getValueAt(int zeile, int spalte) {
            if (spalte > 0) {
                return verlauf()[spalte - 1].get(zeile);
            } else {
                return zeile + 1;
            }
        }

        /** @return Verlauf */
        ArrayList<Integer>[] verlauf() {
            return statistik.verlauf();
        }
    }

    /** Darstellung des Zeilenkopfes. */
    private static class Kopf extends JLabel implements TableCellRenderer {

        /** Standardschriftfarbe. */
        private final Color schriftfarbe = getForeground();

        /** Grafische Grundeinstellungen. */
        Kopf() {
            setHorizontalAlignment(SwingConstants.RIGHT);
            setBorder(BorderFactory.createCompoundBorder(BorderFactory.createEtchedBorder(),
                    BorderFactory.createEmptyBorder(0, ZWISCHENRAUM, 0, ZWISCHENRAUM)));
        }

        @Override
        public Component getTableCellRendererComponent(JTable tabelle, Object wert, boolean selektiert,
                boolean fokussiert, int zeile, int spalte) {
            if (spalte > 0) {
                setFont(getFont().deriveFont(Font.PLAIN));
                setForeground(schriftfarbe);
            } else {
                setFont(getFont().deriveFont(Font.ITALIC));
                setForeground(Color.GRAY);
            }
            setText("<html>" + sortierrichtung(tabelle, spalte) + wert.toString() + "</html>");
            return this;
        }

        /**
         * Ermittelt die Sortierrichtung der Verlaufstabelle.
         *
         * @param tabelle
         *            Die Verlaufstabelle.
         * @param spalte
         *            Die Spalte, welche auf Sortierung berprft wird.
         * @return Pfeilsymbol in HTML-Notation.
         */
        private String sortierrichtung(JTable tabelle, int spalte) {
            @SuppressWarnings("unchecked")
            List<SortKey> sortierschluessel = (List<SortKey>) tabelle.getRowSorter().getSortKeys();
            if (sortierschluessel.isEmpty() || sortierschluessel.get(0).getColumn() != spalte) {
                return "";
            }
            switch (sortierschluessel.get(0).getSortOrder()) {
            case ASCENDING:
                return "&uarr;";
            case DESCENDING:
                return "&darr;";
            default:
                return "";
            }
        }

    }

    /** Konfiguriert die Darstellung der laufenden Nummern der Spiele im Turnier. */
    private static class Nummer extends DefaultTableCellRenderer {
        /** Konstruktor bernimmt grafische Konfiguration. */
        Nummer() {
            setBorder(BorderFactory.createEmptyBorder(0, ZWISCHENRAUM, 0, ZWISCHENRAUM));
            setHorizontalAlignment(SwingConstants.RIGHT);
            setFont(getFont().deriveFont(Font.ITALIC));
            setForeground(Color.GRAY);
        }

        @Override
        public Component getTableCellRendererComponent(JTable tabelle, Object wert, boolean selektiert,
                boolean fokussiert, int zeile, int spalte) {
            setText(wert.toString());
            return this;
        }
    }

    /** Abstand der Zellen zu horizontalen Spaltengrenzen. */
    private static final int ZWISCHENRAUM = 8;

    /**
     * Konstruktor.
     *
     * @param statistik
     *            Statistische Daten, die hier im Verlauf fr ein Turnier dargestellt werden sollen.
     */
    Verlaufstabelle(Statistik statistik) {
        getTableHeader().setDefaultRenderer(new Kopf());
        getTableHeader().setReorderingAllowed(false);
        getTableHeader().setResizingAllowed(false);
        setAutoResizeMode(AUTO_RESIZE_OFF);
        setAutoCreateRowSorter(true);
        setRowSelectionAllowed(false);
        setToolTipText("Die Tabelle ist sortierbar durch Anklicken der Spaltenkpfe");
        setModel(new Adapter(statistik));
    }

    @Override
    public TableCellRenderer getCellRenderer(int zeile, int spalte) {
        if (spalte > 0) {
            return super.getCellRenderer(zeile, spalte);
        } else {
            return new Nummer();
        }
    }

    @Override
    public Component prepareRenderer(TableCellRenderer renderer, int zeile, int spalte) {
        DefaultTableCellRenderer zelle = (DefaultTableCellRenderer) super.prepareRenderer(renderer, zeile, spalte);
        zelle.setBorder(BorderFactory.createEmptyBorder(0, ZWISCHENRAUM, 0, ZWISCHENRAUM));
        return zelle;
    }

    /**
     * Springt mit der Tabellensicht zu einer logischen Zeile unter Bercksichtigung von Umsortierungen.
     *
     * @param zeile
     *            Die logische Zeile, zu welcher der sichtbare Ausschnitt verschoben wird.
     */
    private void springe(int zeile) {
        SwingUtilities.invokeLater(new Runnable() {
            @Override
            public void run() {
                scrollRectToVisible(getCellRect(getRowSorter().convertRowIndexToView(zeile), 0, true));
            }
        });
    }

    @Override
    protected JTableHeader createDefaultTableHeader() {
        return new JTableHeader(columnModel) {
            @Override
            public String getToolTipText(MouseEvent event) {
                int geometrischerIndex = columnModel.getColumnIndexAtX(event.getPoint().x);
                int logischerIndex = columnModel.getColumn(geometrischerIndex).getModelIndex();
                return logischerIndex == 0 ? "Laufende Nummern des Spiels im Turniers"
                        : "Endsummen fr den genannten Spieler zu Spielen mit den laufenden Nummern";
            }
        };
    }

    /** Neuer Spielstand. */
    void aktualisiere() {
        Adapter adapter = (Adapter) getModel();
        int zeile = adapter.getRowCount() - 1;
        adapter.fireTableRowsUpdated(zeile, zeile);
    }

    /** Beende ein Spiel. */
    void beende() {
        Adapter adapter = (Adapter) getModel();
        int zeile = adapter.getRowCount() - 1;
        adapter.fireTableRowsInserted(zeile, zeile);
        springe(zeile);
    }

    /** Neues Turnier gestartet oder Turnier fortgesetzt. */
    void starte() {
        Adapter adapter = (Adapter) getModel();
        adapter.fireTableStructureChanged();
        int zeile = adapter.getRowCount() - 1;
        springe(zeile);
    }

}
