/******************************************************************************
 ** $Id: ImHieb.java 899 2016-02-21 17:25:15Z wmh $
 ** Diese Datei ist Bestandteil der Java-Quelltexte des Wrfelspiels JaFuffy.
 ** Lauffhig ab Java 6.
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************
 ** Die aktuellste Version von JaFuffy findet sich im Internet unter
 ** <http://jafuffy.3kelvin.de>.
 **
 ** Kommentare, Fehler oder Erweiterungswnsche bitte per E-Mail senden an
 ** <jafuffy@3kelvin.de>.
 ******************************************************************************/
package jafuffy.logik.auswertung;

import jafuffy.bedienung.Auswahl;
import jafuffy.logik.Tabzeile;
import jafuffy.logik.Wuerfel;

/**
 * Im-Hieb-Auswertung. Ermittelt die Punktzahl eines Spielers zum vorliegenden Wurf. Im-Hieb-Erkennung.
 */
public class ImHieb extends Klassisch {

    /** Ermglicht Serialisierung. */
    private static final long serialVersionUID = 6730182688848850948L;

    /** Die Klasse der Auswahl, die zu dieser Auswertung passt. */
    public static final Class<? extends Auswahl> AUSWAHLKLASSE = jafuffy.bedienung.auswahl.ImHieb.class;
    /** Regelname. */
    public static final String REGEL = "Im-Hieb-Regeln";
    /** Normaler Im-Hieb-Bonus, auer JaFuffy. */
    public static final int IM_HIEB_BONUS_MIN = 5;
    /** Maximal Im-Hieb-Bonus, bei JaFuffy. */
    public static final int IM_HIEB_BONUS_MAX = 30;

    /** Standardabweichung als grobe Schtzung fr das mittlere Spielergebnis. */
    private static final int ABWEICHUNG = 45;
    /** Erwartete Punktzahl pro Spiel als grobe Schtzung. */
    private static final int MITTELWERT = 235;

    /** Soll Nachricht "Im Hieb!" in Tabelle erscheinen? */
    private boolean istImHiebNachrichtSichtbar = false;
    /** Der Eintrag, fr den die Nachricht "Im Hieb!" erscheinen soll. */
    private Tabzeile imHiebEintrag;

    /**
     * Konstruktor.
     *
     * @param wuerfel
     *            Die im Turnier benutzten Wrfel.
     */
    public ImHieb(Wuerfel[] wuerfel) {
        super(wuerfel, new jafuffy.logik.analyse.ImHieb(wuerfel));
    }

    @Override
    public int abweichung() {
        return ABWEICHUNG;
    }

    /**
     * Aktiviert die Im-Hieb-Nachricht.
     *
     * @param eintrag
     *            Die Nachricht wird fr diesen Eintrag aktiviert.
     */
    public void aktiviereImHiebNachricht(Tabzeile eintrag) {
        this.imHiebEintrag = eintrag;
        istImHiebNachrichtSichtbar = true;
    }

    /** Deaktiviert die Im-Hieb-Nachricht. */
    public void deaktiviereImHiebNachricht() {
        istImHiebNachrichtSichtbar = false;
    }

    /**
     * @param eintrag
     *            Nummer des Eintrags auf dem Zettel.
     *
     * @return Gibt an, ob fr den angegebenen Eintrag ein Ergebnis im Hieb erzielt wurde.
     */
    public boolean istImHieb(Tabzeile eintrag) {
        return eintrag.unten() && eintrag != Tabzeile.CHANCE && aktiver.einmalig() && super.wert(eintrag) > 0;
    }

    @Override
    public int mittelwert() {
        return MITTELWERT;
    }

    @Override
    public String text(Tabzeile eintrag) {
        return istImHiebNachrichtSichtbar && eintrag == imHiebEintrag ? "Im Hieb!" : null;
    }

    @Override
    public String toString() {
        return REGEL;
    }

    @Override
    public int wert(Tabzeile eintrag) {
        if (istImHieb(eintrag)) {
            return super.wert(eintrag) + (eintrag == Tabzeile.JAFUFFY ? IM_HIEB_BONUS_MAX : IM_HIEB_BONUS_MIN);
        } else {
            return super.wert(eintrag);
        }
    }

}
