/******************************************************************************
 ** $Id: Erweitert.java 876 2016-02-08 02:18:27Z wmh $
 ** Diese Datei ist Bestandteil der Java-Quelltexte des Wrfelspiels JaFuffy.
 ** Lauffhig ab Java 6.
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************
 ** Die aktuellste Version von JaFuffy findet sich im Internet unter
 ** <http://jafuffy.3kelvin.de>.
 **
 ** Kommentare, Fehler oder Erweiterungswnsche bitte per E-Mail senden an
 ** <jafuffy@3kelvin.de>.
 ******************************************************************************/
package jafuffy.logik.auswertung;

import jafuffy.Eigenschaften;
import jafuffy.bedienung.Auswahl;
import jafuffy.logik.Tabzeile;
import jafuffy.logik.Wuerfel;

/** Erweiterte Auswertung. */
public class Erweitert extends Klassisch {

    /** Ermglicht Serialisierung. */
    private static final long serialVersionUID = -5463902030609191338L;

    /** Die Klasse der Auswahl, die zu dieser Auswertung passt. */
    public static final Class<? extends Auswahl> AUSWAHLKLASSE = jafuffy.bedienung.auswahl.Erweitert.class;
    /** Regelname. */
    public static final String REGEL = "Erweiterte Regeln";

    /** Standardabweichung als grobe Schtzung fr das mittlere Spielergebnis. */
    private static final int ABWEICHUNG = 60;
    /** Erwartete Punktzahl pro Spiel als grobe Schtzung. */
    private static final int MITTELWERT = 240;
    /** Joker-Bonus. */
    public static final int JOKER = 100;

    /** Gibt an, ob eine Bonus-Nachricht angefordert ist. */
    private boolean istBonusnachrichtAngefordert = false;
    /** Gibt an, ob eine Joker-Nachricht angefordert ist. */
    private boolean istJokerNachrichtAngefordert = false;
    /** Gibt an, ob eine Setzzwang-Nachricht angefordert ist. */
    private boolean istSetzzwangNachrichtAngefordert = false;

    /**
     * Konstruktor.
     *
     * @param wuerfel
     *            Die im Turnier benutzten Wrfel.
     */
    public Erweitert(Wuerfel[] wuerfel) {
        super(wuerfel);
    }

    @Override
    public int abweichung() {
        return ABWEICHUNG;
    }

    /** @return Ermittelt, ob ein Joker erwrfelt wurde. */
    public boolean istJokerVorliegend() {
        return super.wert(Tabzeile.JAFUFFY) > 0 && aktiver.punkte(Tabzeile.JAFUFFY) > 0
                && !aktiver.setzbar(Tabzeile.oben(wuerfel[0].augen()));
    }

    @Override
    public int mittelwert() {
        return MITTELWERT;
    }

    /** @return Gibt an, ob ein Zwang zum Setzen vorliegt (JaFuffy erwrfelt, JaFuffy-Feld belegt, oben frei). */
    public boolean setzzwang() {
        return super.wert(Tabzeile.JAFUFFY) > 0 && aktiver.punkte(Tabzeile.JAFUFFY) > 0
                && aktiver.setzbar(Tabzeile.oben(wuerfel[0].augen()));
    }

    @Override
    public String spicker(Tabzeile eintrag) {
        if (Eigenschaften.GLOBAL.aktiv("Spicker") && aktiver.gewuerfelt() && istJokerVorliegend()
                && eintrag == Tabzeile.JAFUFFY) {
            String text;
            text = "<p align=center><em>Joker (+ 100 Bonuspunkte)!</em></p>"
                    + "<p>Bitte Wurf in freies Feld eintragen.</p>";
            if (sindTippsAngefordert()) {
                return "<html>" + text + "<p>Kontextmen!</p></html>";
            } else {
                return "<html>" + text + "</html>";
            }
        } else {
            return super.spicker(eintrag);
        }
    }

    @Override
    public String text(Tabzeile eintrag) {
        if (eintrag == Tabzeile.JAFUFFY && istJokerNachrichtAngefordert) {
            return "Joker!";
        } else if (eintrag == Tabzeile.JAFUFFY && istBonusnachrichtAngefordert) {
            return "Bonus!";
        } else if (istSetzzwangNachrichtAngefordert && eintrag == Tabzeile.oben(wuerfel[0].augen())) {
            return "Gesetzt!";
        } else {
            return null;
        }
    }

    @Override
    public String toString() {
        return REGEL;
    }

    @Override
    public int wert(Tabzeile eintrag) {
        switch (eintrag) {
        case KLEINESTRASSE:
            return istJokerVorliegend() || super.wert(eintrag) > 0
                    ? jafuffy.logik.analyse.Klassisch.KleineStrasse.PUNKTE : 0;
        case GROSSESTRASSE:
            return istJokerVorliegend() || super.wert(eintrag) > 0
                    ? jafuffy.logik.analyse.Klassisch.GrosseStrasse.PUNKTE : 0;
        case JAFUFFY:
            if (super.wert(eintrag) > 0 && aktiver.punkte(eintrag) > 0) {
                return aktiver.punkte(eintrag) + JOKER;
            }
        default:
            return super.wert(eintrag);
        }
    }

    /**
     * Bestimmt das Anzeigen der Textnachricht "Bonus!".
     *
     * @param istBonusErzielt
     *            Gibt an, ob ein JaFuffy-Bonus erzielt wurde und damit die Nachricht gezeigt werden soll.
     */
    public void zeigeBonusnachricht(boolean istBonusErzielt) {
        istBonusnachrichtAngefordert = istBonusErzielt;
    }

    /**
     * Bestimmt das Anzeigen der Textnachricht "Joker!".
     *
     * @param istJokerVerfuegbar
     *            Gibt an, ob ein Joker vorliegt und damit die Nachricht gezeigt werden soll.
     */
    public void zeigeJokernachricht(boolean istJokerVerfuegbar) {
        if (istJokerVerfuegbar) {
            // Spieler darf keinen Wrfel mehr in den Becher legen
            for (Wuerfel w : wuerfel) {
                w.setEnabled(false);
            }
        }
        istJokerNachrichtAngefordert = istJokerVerfuegbar;
    }

    /**
     * Bestimmt das Anzeigen der Textnachricht "Gesetzt!".
     *
     * @param istSetzzwangGegeben
     *            Gibt an, ob Setzzwang vorliegt und damit die Nachricht gezeigt werden soll.
     */
    public void zeigeSetzzwangnachricht(boolean istSetzzwangGegeben) {
        if (istSetzzwangGegeben) {
            // Spieler darf keinen Wrfel mehr in den Becher legen
            for (Wuerfel w : wuerfel) {
                w.setEnabled(false);
            }
        }
        istSetzzwangNachrichtAngefordert = istSetzzwangGegeben;
    }

}
