/******************************************************************************
 ** $Id: ImHieb.java 899 2016-02-21 17:25:15Z wmh $
 ** Diese Datei ist Bestandteil der Java-Quelltexte des Wrfelspiels JaFuffy.
 ** Lauffhig ab Java 6.
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************
 ** Die aktuellste Version von JaFuffy findet sich im Internet unter
 ** <http://jafuffy.3kelvin.de>.
 **
 ** Kommentare, Fehler oder Erweiterungswnsche bitte per E-Mail senden an
 ** <jafuffy@3kelvin.de>.
 ******************************************************************************/
package jafuffy.logik.analyse;

import jafuffy.logik.Kombinatorik;
import jafuffy.logik.Tabzeile;
import jafuffy.logik.Turnier;
import jafuffy.logik.Wuerfel;

/** Erstellt Analysen und berechnet Prognosen fr die klassische Variante der Spielregeln. */
public class ImHieb extends Klassisch {

    /** Erlaubt Serialisierung. */
    private static final long serialVersionUID = 447027133112163377L;

    /** Kleiner und groer Pasch mit der Im-Hieb-Regel. */
    class Pasch extends Klassisch.Pasch {
        /** Zur Serialisierung. */
        private static final long serialVersionUID = -626035053170491772L;

        /**
         * @param pasch
         *            Bestimmt die Paschart, nmlich Dreierpasch oder Viererpasch.
         * @see Klassisch.Pasch#Pasch
         */
        Pasch(int pasch) {
            super(pasch);
            double s = 0;
            for (int k = pasch; k <= Turnier.WUERFEL; k++) {
                s += Kombinatorik.binom(Turnier.WUERFEL, k) * Wuerfel.AUGEN
                        * Math.pow(Turnier.WUERFEL, Turnier.WUERFEL - k);
            }
            mittel += s / Kombinatorik.VARIATIONEN;
        }
    }

    /** Full House mit der Im-Hieb-Regel. */
    class FullHouse extends Klassisch.FullHouse {
        /** Zur Serialisierung. */
        private static final long serialVersionUID = -3227165375021646796L;
        /** Mittlere Im-Hieb-Bonus. */
        private static final double IM_HIEB_BONUS_MITTEL = jafuffy.logik.auswertung.ImHieb.IM_HIEB_BONUS_MIN * 17.
                / 432.;

        /** Ermittlung des korrigierten mittleren Punktstandes um den Im-Hieb-Bonus. */
        FullHouse() {
            mittel += IM_HIEB_BONUS_MITTEL;
        }
    }

    /** Kleine Strae mit der Im-Hieb-Regel. */
    class KleineStrasse extends Klassisch.KleineStrasse {
        /** Zur Serialisierung. */
        private static final long serialVersionUID = 1874639430274154259L;
        /** Mittlere Im-Hieb-Bonus. */
        private static final double IM_HIEB_BONUS_MITTEL = jafuffy.logik.auswertung.ImHieb.IM_HIEB_BONUS_MIN * 25.
                / 162.;

        /** Ermittlung des korrigierten mittleren Punktstandes um den Im-Hieb-Bonus. */
        KleineStrasse() {
            mittel += IM_HIEB_BONUS_MITTEL;
        }
    }

    /** Groe Strae mit der Im-Hieb-Regel. */
    class GrosseStrasse extends Klassisch.GrosseStrasse {
        /** Zur Serialisierung. */
        private static final long serialVersionUID = -2371420977396166725L;
        /** Mittlere Im-Hieb-Bonus. */
        private static final double IM_HIEB_BONUS_MITTEL = jafuffy.logik.auswertung.ImHieb.IM_HIEB_BONUS_MIN * 5.
                / 162.;

        /** Ermittlung des korrigierten mittleren Punktstandes um den Im-Hieb-Bonus. */
        GrosseStrasse() {
            mittel += IM_HIEB_BONUS_MITTEL;
        }
    }

    /** JaFuffy mit der Im-Hieb-Regel. */
    class JaFuffy extends Klassisch.JaFuffy {
        /** Zur Serialisierung. */
        private static final long serialVersionUID = -4640248861129674120L;
        /** Mittlere Im-Hieb-Bonus. */
        private static final double IM_HIEB_BONUS_MITTEL = jafuffy.logik.auswertung.ImHieb.IM_HIEB_BONUS_MAX / 1296.;

        /** Ermittlung des korrigierten mittleren Punktstandes um den Im-Hieb-Bonus. */
        JaFuffy() {
            mittel += IM_HIEB_BONUS_MITTEL;
        }
    }

    /** Alle Eintrge fr die klassische Variante der Spielregeln als Tabelle zusammengefasst. */
    class Eintragungen extends Basiseintragungen {
        /** Erlaubt Serialisierung. */
        private static final long serialVersionUID = -226404933174352144L;

        /** Konstruktion der Tabelle. */
        Eintragungen() {
            put(Tabzeile.DREIERPASCH, new Pasch(Pasch.KLEIN));
            put(Tabzeile.VIERERPASCH, new Pasch(Pasch.GROSS));
            put(Tabzeile.FULLHOUSE, new FullHouse());
            put(Tabzeile.KLEINESTRASSE, new KleineStrasse());
            put(Tabzeile.GROSSESTRASSE, new GrosseStrasse());
            put(Tabzeile.JAFUFFY, new JaFuffy());
        }
    }

    /**
     * Konstruktor.
     *
     * @param wuerfel
     *            Wrfel, mit denen das Turnier durchgefhrt wird.
     */
    public ImHieb(Wuerfel[] wuerfel) {
        super(wuerfel);
    }

    @Override
    protected Basiseintragungen eintragungen() {
        return new Eintragungen();
    }

}
