/******************************************************************************
 ** $Id: Analyse.java 894 2016-02-20 17:27:56Z wmh $
 ** Diese Datei ist Bestandteil der Java-Quelltexte des Wrfelspiels JaFuffy.
 ** Lauffhig ab Java 6.
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************
 ** Die aktuellste Version von JaFuffy findet sich im Internet unter
 ** <http://jafuffy.3kelvin.de>.
 **
 ** Kommentare, Fehler oder Erweiterungswnsche bitte per E-Mail senden an
 ** <jafuffy@3kelvin.de>.
 ******************************************************************************/
package jafuffy.logik;

import java.io.Serializable;
import java.util.Arrays;
import java.util.HashMap;
import java.util.IdentityHashMap;
import java.util.Map;
import java.util.Random;

/** Hlt Eintrge, erstellt Analysen und berechnet Prognose ber die erzielbare Punktzahl bei vorgelegtem Wurf. */
public abstract class Analyse implements Serializable {

    /** Zur Serialisierung. */
    private static final long serialVersionUID = -6325984568972413728L;

    /** Eintrag in Tabelle. */
    protected abstract class Eintrag implements Serializable {

        /** Zur Serialisierung. */
        private static final long serialVersionUID = -1491939796847928172L;

        /** Wichtung fr freie Felder im Intervall [0,1]. */
        private double wichtung;

        /** Karte, welcher Wrfel erneut gewrfelt werden soll. */
        protected final IdentityHashMap<Wuerfel, Boolean> neuwurfkarte = new IdentityHashMap<>(Turnier.WUERFEL);

        {
            for (Wuerfel w : wuerfel) {
                neuwurfkarte.put(w, false);
            }
        }

        /** Voraussichtlicher Endstand, wenn der Wurf fr den aktuellen Eintrag weiter verfolgt wird. */
        protected double endsumme;
        /** Erwarteter mittlerer Punktstand pro Eintrag (Startmittel). */
        protected double mittel;
        /** Erzielte Punktzahl. */
        protected int wert;

        /** @return Maximale Anzahl gleicher Augen. */
        protected int maxgleich() {
            int max = 0;
            for (Map.Entry<Paschaufzaehlung, Boolean> pasch : paschkarte.entrySet()) {
                int groesse = pasch.getKey().groesse();
                boolean istPaschVorhanden = pasch.getValue();
                if (istPaschVorhanden && groesse > max) {
                    max = groesse;
                }
            }
            return max;
        }

        /**
         * @param spieler
         *            Der Spieler, der gerade an der Reihe ist.
         * @param zuwachs
         *            Erwarteter Zuwachs fr oberen und unteren Teil der Tabelle.
         * @return Prognose fr das zu erwartende Resultat fr den laufenden Eintrag zur Ermittlung des
         *         Gesamtergebnisses.
         */
        protected abstract double prognose(Spieler spieler, Zuwachs zuwachs);

        /**
         * Tatschlich erzielte Punktzahl ermitteln.
         *
         * @return Momentan erreichte Punktzahl fr einen Eintrag.
         */
        protected abstract int punkte();

        /**
         * Vorschlag zur Wrfelauswahl ermitteln.
         *
         * @param spieler
         *            Der Spieler, der an der Reihe ist.
         */
        protected abstract void waehle(Spieler spieler);

        /** Tatschlich erwrfelte Punktzahl ermitteln und merken. */
        void bewerte() {
            wert = punkte();
        }

        /** @return Voraussichtlicher Punktendstand. */
        double endsumme() {
            return endsumme;
        }

        /**
         * Berechnet die erwartete Endsumme, welche sich aus der gegenwrtigen Endsumme und dem erwarteten Zuwachs
         * ergibt; einschlielich Bonus.
         *
         * @param spieler
         *            Der Spieler, der gerade an der Reihe ist.
         * @param zuwachs
         *            Der erwartete Zuwachs.
         * @return Erwartete Endsumme.
         */
        abstract double endsumme(Spieler spieler, Zuwachs zuwachs);

        /**
         * @param w
         *            Referenz auf Wrfel.
         * @return Soll Wrfel erneut gewrfelt werden?
         */
        boolean erneut(Wuerfel w) {
            return neuwurfkarte.get(w);
        }

        /**
         * Wichtung eintragen.
         *
         * @param wichtung
         *            Setzt die Wichtung eines einzelnen Eintrags.
         */
        void gewichte(double wichtung) {
            this.wichtung = wichtung;
        }

        /** @return Mittlere Punktzahl des Eintrags bei 3maligem Wrfeln. */
        double mittel() {
            return mittel;
        }

        /**
         * Prognose ermitteln, und Wrfelauswahl fr Vorschlag treffen.
         *
         * @param spieler
         *            Spieler, der an der Reihe ist.
         * @param zuwachs
         *            Mittlerer erwarteter Zuwachs (pro Feld 3mal Wrfeln angenommen).
         */
        void tippe(Spieler spieler, Zuwachs zuwachs) {
            endsumme = endsumme(spieler, zuwachs);
            if (spieler.rest() > 0) {
                waehle(spieler);
                endsumme += prognose(spieler, zuwachs);
            } else {
                // alle Wrfe verbraucht; vorgeben, als ob alle Wrfel nochmals zu werfen sind
                for (Wuerfel w : wuerfel) {
                    neuwurfkarte.put(w, true);
                }
                endsumme += punkte();
            }
        }

        /** @return Erzielte Punktzahl fr Wurf. */
        int wert() {
            return wert;
        }

        /** @return Wichtung */
        double wichtung() {
            return wichtung;
        }

    }

    /** Zhlt alle Eintrge aus dem oberen Teil der Tabelle auf. */
    protected enum Index {
        /** Einser-Eintrag im oberen Tabellenteil. */
        EINSER,
        /** Zweier-Eintrag im oberen Tabellenteil. */
        ZWEIER,
        /** Dreier-Eintrag im oberen Tabellenteil. */
        DREIER,
        /** Vierer-Eintrag im oberen Tabellenteil. */
        VIERER,
        /** Fnfer-Eintrag im oberen Tabellenteil. */
        FUENFER,
        /** Sechser-Eintrag im oberen Tabellenteil. */
        SECHSER;
        /** @return Liefert die Augenzahl zum Eintrag zurck. */
        public int augen() {
            return ordinal() + 1;
        }

        /**
         * @param haeufigkeiten
         *            Feld mit allen Augenhufigkeiten.
         * @return Anzahl der Wrfel mit dieser Augenzahl.
         */
        public int vorkommen(int[] haeufigkeiten) {
            return haeufigkeiten[ordinal()];
        }
    }

    /** Eintrag im oberen Tabellenteil. Mglicher Bonus wird bercksichtigt. */
    class Oben extends Eintrag {
        /** Zur Serialisierung. */
        private static final long serialVersionUID = -5829769579272736329L;

        /** Augenzahl. */
        private final int augen;
        /** Index fr Eintrag (0, 1, 2, 3, 4 oder 5) im oberen Teil der Tabelle, abgeleitet aus der Augenzahl. */
        private final int ind;

        /**
         * Konstruktor.
         *
         * @param index
         *            Der Index des oberen Eintrags (0, 1, 2, 3, 4, 5).
         */
        Oben(Index index) {
            ind = index.ordinal();
            augen = index.augen();
            mittel = augen * Kombinatorik.gleichenmittel(Turnier.WUERFEL, Turnier.RUNDEN);
        }

        @Override
        protected double prognose(Spieler spieler, Zuwachs zuwachs) {
            // freie Wrfel
            int n = Turnier.WUERFEL - augenhaeufigkeiten[ind];
            // Mindestpunktwert, schon erwrfelt
            double min = augen * augenhaeufigkeiten[ind];
            // Prognose fr Zuwchse (nochmals k gleiche Augenzahl)
            double punkte = min;
            // ermittle Beitrge fr weitere Anzahlen von gleichen Augenzahlen
            for (int k = 1; k <= n; k++) {
                double p = Kombinatorik.gleichenwahrscheinlichkeit(n, k, spieler.rest());
                double neu = k * augen;
                punkte += p * neu;
            }
            return punkte;
        }

        @Override
        protected int punkte() {
            return augenhaeufigkeiten[ind] * augen;
        }

        @Override
        protected void waehle(Spieler spieler) {
            for (Wuerfel w : wuerfel) {
                neuwurfkarte.put(w, w.augen() != augen);
            }
        }

        @Override
        double endsumme(Spieler spieler, Zuwachs zuwachs) {
            return spieler.endsumme()
                    + zuwachs.zuwachs(spieler.gesamt(), augenhaeufigkeiten[ind], spieler.rest(), augen) - mittel;
        }
    }

    /** Zhlt alle Pasche von Zwei bis Fnf auf, wobei alle Wrfel zu einem festen Augenwert herangezogen werden. */
    protected enum Paschaufzaehlung {
        /** Zweierpasch. */
        ZWEIERPASCH,
        /** Dreierpasch und kein Zweierpasch. */
        DREIERPASCH,
        /** Viererpasch und weder Dreipasch noch Zweierpasch. */
        VIERERPASCH,
        /** Fnferpasch und weder Viererpasch noch Dreierpasch noch Zweierpasch. */
        FUENFERPASCH;
        /** @return Anzahl der Wrfel im Pasch, wobei der Maximalwert fr einen festen Augenwert zurckgegeben wird. */
        public int groesse() {
            return ordinal() + 2;
        }
    }

    /** Eintrag im oberen Tabellenteil. */
    public abstract class Unten extends Eintrag {
        /** Zur Serialisierung bentigt. */
        private static final long serialVersionUID = 8894740065051562154L;

        @Override
        double endsumme(Spieler spieler, Zuwachs zuwachs) {
            return spieler.endsumme() + zuwachs.zuwachs() - mittel;
        }
    }

    /** Chance. */
    class Chance extends Unten {
        /** Zur Serialisierung. */
        private static final long serialVersionUID = -3411668587180538841L;
        /** Erzielte Punkte (alle Augen aufsummiert). */
        private double punkte;

        /** Konstruktor. */
        Chance() {
            mittel = Turnier.WUERFEL * Kombinatorik.augenzahl(Turnier.RUNDEN);
        }

        @Override
        protected double prognose(Spieler spieler, Zuwachs zuwachs) {
            return punkte;
        }

        @Override
        protected int punkte() {
            return augensumme;
        }

        @Override
        protected void waehle(Spieler spieler) {
            punkte = 0;
            for (Wuerfel w : wuerfel) {
                double m = Kombinatorik.augenzahl(spieler.rest());
                int a = w.augen();
                neuwurfkarte.put(w, m > a);
                punkte += neuwurfkarte.get(w) ? m : a;
            }
        }
    }

    /** Erwarteter Zuwachs fr diverse Tabellenteile. */
    protected static class Zuwachs {
        /** Erwarteter Bonuszuwachs. */
        private final double mittlererBonusZuwachs;
        /** Erwarteter Zuwachs fr obere und untere Tabelle, aber ohne Bonus. */
        private final double mittlererZuwachs;
        /** Beitragswahrscheinlichkeit eines einzelnen oberen freien Eintrags. */
        private final double p;
        /** Mindestanzahl der Wrfel fr jeden einzelnen oberen freien Eintrags. */
        private final int teilung;

        /**
         * Berechnung grundlegender Zuwachsgren.
         *
         * @param gesamt
         *            Die Gesamtpunktzahl aus dem oberen Tabellenteil.
         * @param mittlererObererZuwachs
         *            Erwarteter Zuwachs, aufsummiert ber alle oberen Eintrge.
         * @param augensummeObereFreieEintraege
         *            Die Augensumme der Augen aller freien oberen Eintrge.
         * @param anzahlFreieObereEintraege
         *            Die Anzahl der oberen freien Eintrge.
         * @param mittlererUntererZuwachs
         *            Erwarteter Zuwachs, aufsummiert ber alle unteren Eintrge.
         */
        Zuwachs(int gesamt, double mittlererObererZuwachs, int augensummeObereFreieEintraege,
                int anzahlFreieObereEintraege, double mittlererUntererZuwachs) {
            if (gesamt < Spieler.GRENZE && augensummeObereFreieEintraege > 0) {
                teilung = (Spieler.GRENZE - gesamt + augensummeObereFreieEintraege - 1) / augensummeObereFreieEintraege;
                double q = Kombinatorik.mindestgleichenwahrscheinlichkeit(teilung);
                mittlererBonusZuwachs = Math.pow(q, anzahlFreieObereEintraege) * Spieler.BONUS;
                p = q != 0 ? q : 1;
            } else {
                teilung = 0;
                p = 1;
                mittlererBonusZuwachs = 0;
            }
            mittlererZuwachs = mittlererObererZuwachs + mittlererUntererZuwachs;
        }

        /** @return Der erwarte Zuwachs fr die gesamte Tabelle. */
        double zuwachs() {
            return mittlererZuwachs + mittlererBonusZuwachs;
        }

        /**
         * @param gesamt
         *            Die Gesamtpunktzahl aus dem oberen Tabellenteil.
         * @param guenstige
         *            Die Anzahl der gnstigen Wrfel fr einen Eintrag.
         * @param r
         *            Die Anzahl der verbleibenden Restwrfe.
         * @param augen
         *            Die Augenzahl fr den Eintrag, aus dem der Zuwachs resultiert.
         * @return Der erwarte Zuwachs fr die gesamte Tabelle.
         */
        double zuwachs(int gesamt, int guenstige, int r, int augen) {
            double q = 0;
            int punkteBisZumBonus = Spieler.GRENZE - gesamt;
            if (punkteBisZumBonus > 0) {
                punkteBisZumBonus -= guenstige * augen;
                q = 1;
                if (punkteBisZumBonus > 0) {
                    int n = Turnier.WUERFEL - guenstige;
                    int min = guenstige <= teilung ? teilung - guenstige : 0;
                    q = Kombinatorik.mindestgleichenwahrscheinlichkeit(n, min, r);
                }
            }
            return mittlererZuwachs + q / p * mittlererBonusZuwachs;
        }
    }

    /** Alle Standardeintrge als Tabelle zusammengefasst. */
    protected abstract class Basiseintragungen extends HashMap<Tabzeile, Eintrag> {
        /** Erlaubt Serialisierung. */
        private static final long serialVersionUID = 8130365096301351531L;

        /** Belegt die Standardeintrge, welche fr alle Varianten der Spielregeln gilt. */
        protected Basiseintragungen() {
            super(Tabzeile.ALLE.length, 1);
            put(Tabzeile.EINS, new Oben(Index.EINSER));
            put(Tabzeile.ZWEI, new Oben(Index.ZWEIER));
            put(Tabzeile.DREI, new Oben(Index.DREIER));
            put(Tabzeile.VIER, new Oben(Index.VIERER));
            put(Tabzeile.FUENF, new Oben(Index.FUENFER));
            put(Tabzeile.SECHS, new Oben(Index.SECHSER));
            put(Tabzeile.CHANCE, new Chance());
        }
    }

    /** Neutrale Wichtung, falls alle Wichtungen gleich sind. */
    private static final double NEUTRALWICHTUNG = 1;

    /** Zufallsgenerator zur Auswahl von Tipp-Strategien bei Mehrdeutigkeiten. */
    protected static final Random ZUFALL = new Random();

    /** Voraussichtlicher mittlerer Punktstand pro Feld, wobei alle drei Wrfe genutzt werden. */
    private final Basiseintragungen eintragungen;

    /** Bester Eintrag. */
    private Tabzeile bester;

    /** Anzahl der Wrfel pro Augenzahl. */
    protected final int[] augenhaeufigkeiten = new int[Wuerfel.AUGEN];
    /** Die Karte ordnet einem Pasch (2, 3, 4, 5 gleiche Wrfel) zu, ob genau ein solcher im Wurf vorhanden ist. */
    protected final Map<Paschaufzaehlung, Boolean> paschkarte = new HashMap<>(Paschaufzaehlung.values().length, 1);
    /** Im Turnier benutzte Wrfel. */
    protected final Wuerfel[] wuerfel;

    /** Summe aller Augen. */
    protected int augensumme;

    /**
     * Konstruktor.
     *
     * @param wuerfel
     *            Die im Turnier benutzten Wrfel.
     */
    protected Analyse(Wuerfel[] wuerfel) {
        this.wuerfel = wuerfel;
        eintragungen = eintragungen();
    }

    /**
     * @param eintrag
     *            Eintragsnummer.
     * @return Erwrfelter Punktstand.
     */
    public int wert(Tabzeile eintrag) {
        return eintrag(eintrag).wert();
    }

    /**
     * @param eintrag
     *            Die Zeile in der Tabelle, zu der ein Eintrag zurckgegeben werden soll.
     * @return Eintrag zur Tabellenzeile
     */
    private Eintrag eintrag(Tabzeile eintrag) {
        return eintragungen.get(eintrag);
    }

    /**
     * Ermittelt das Maximum ber alle Endstnde fr alle setzbaren Eintrge, und merkt sich einen Eintrag mit dem
     * voraussichtlich besten Endstand.
     *
     * @param spieler
     *            Der Spieler, der gerade an der Reihe ist.
     * @param spieler
     * @return Das Maximum aller Endstnde.
     */
    private double findeMaximum(Spieler spieler) {
        double max = 0;
        bester = null;
        for (Tabzeile z : Tabzeile.ALLE) {
            if (spieler.setzbar(z)) {
                double endstand = eintrag(z).endsumme();
                if (endstand > max) {
                    bester = z;
                    max = endstand;
                }
            }
        }
        return max;
    }

    /**
     * Ermittelt das Minimum ber alle Endstnde fr alle setzbaren Eintrge.
     *
     * @param spieler
     *            Der Spieler, der gerade an der Reihe ist.
     * @param max
     *            Der maximale erwartete Endstand fr alle setzbaren Eintrge.
     * @return Das Minimum aller Endstnde.
     */
    private double findeMinimum(Spieler spieler, double max) {
        double min = max;
        for (Tabzeile z : Tabzeile.ALLE) {
            if (spieler.setzbar(z)) {
                double endstand = eintrag(z).endsumme();
                if (endstand < min) {
                    min = endstand;
                }
            }
        }
        return min;
    }

    /**
     * Gewichtet die setzbaren Eintrge nach erwarteter Hhe des Endstandes. Das Gewicht je Eintrag entspricht der
     * linearen Abbildung des Intervals zwischen dem minimalen und maximalen Endstand, abgebildet auf das Interval [0,
     * 1].
     *
     * @param spieler
     *            Der Spieler, der gerade an der Reihe ist.
     * @param min
     *            Der minimale erwartete Endstand fr alle setzbaren Eintrge.
     * @param max
     *            Der maximale erwartete Endstand fr alle setzbaren Eintrge.
     */
    private void gewichte(Spieler spieler, double min, double max) {
        for (Tabzeile z : Tabzeile.ALLE) {
            if (spieler.setzbar(z)) {
                if (max - min > 0) {
                    double endstand = eintrag(z).endsumme();
                    eintrag(z).gewichte((endstand - min) / (max - min));
                } else {
                    eintrag(z).gewichte(NEUTRALWICHTUNG);
                }
            } else {
                eintrag(z).gewichte(0);
            }
        }
    }

    /**
     * Prognosen fr jeden setzbaren Eintrag berechnen, Zuwchse ermitteln, und Wrfelauswahlen fr Vorschlge treffen.
     *
     * @param spieler
     *            Der Spieler, der an der Reihe ist.
     */
    private void tippe(Spieler spieler) {
        double mittlererObererZuwachs = 0;
        int anzahlObereFreieEintraege = 0;
        int augensummeObereFreieEintraege = 0;
        for (Tabzeile z : Tabzeile.OBEN) {
            if (spieler.setzbar(z)) {
                Oben eintrag = (Oben) eintrag(z);
                mittlererObererZuwachs += eintrag.mittel();
                anzahlObereFreieEintraege++;
                augensummeObereFreieEintraege += eintrag.augen;
            }
        }
        double mittlererUntererZuwachs = 0;
        for (Tabzeile z : Tabzeile.UNTEN) {
            if (spieler.setzbar(z)) {
                mittlererUntererZuwachs += eintrag(z).mittel();
            }
        }
        Zuwachs zuwachs = new Zuwachs(spieler.gesamt(), mittlererObererZuwachs, augensummeObereFreieEintraege,
                anzahlObereFreieEintraege, mittlererUntererZuwachs);
        // Tipps berechnen
        for (Tabzeile z : Tabzeile.ALLE) {
            if (spieler.setzbar(z)) {
                eintrag(z).tippe(spieler, zuwachs);
            }
        }
    }

    /** @return Alle Eintragungen in der Tabelle. */
    protected abstract Basiseintragungen eintragungen();

    /** @return bester Eintrag (-1 heit keiner vorhanden) */
    Tabzeile bester() {
        return bester;
    }

    /**
     * @param eintrag
     *            Eintrag in der Tabelle.
     * @param w
     *            Referenz auf Wrfel.
     * @return Wrfel nochmals wrfeln fr Vorschlag?
     */
    boolean erneut(Tabzeile eintrag, Wuerfel w) {
        return eintrag(eintrag).erneut(w);
    }

    /** Tatschlich erwrfelte Punkte fr alle Eintrge errechnen. */
    void errechne() {
        // Zhle Wrfel pro Augenzahl, Summe bestimmen
        Arrays.fill(augenhaeufigkeiten, 0);
        augensumme = 0;
        for (Wuerfel w : wuerfel) {
            augensumme += w.augen();
            augenhaeufigkeiten[w.augen() - 1]++;
        }
        // Genau 2, 3, 4, 5 gleiche Wrfel?
        for (Paschaufzaehlung pasch : Paschaufzaehlung.values()) {
            boolean istPaschVorhanden = false;
            for (int augenhaeufigkeit : augenhaeufigkeiten) {
                istPaschVorhanden = augenhaeufigkeit == pasch.groesse() || istPaschVorhanden;
            }
            paschkarte.put(pasch, istPaschVorhanden);
        }
        // erwrfelte Punktzahl fr alle Eintrge
        for (Tabzeile z : Tabzeile.ALLE) {
            eintrag(z).bewerte();
        }
    }

    /**
     * Punktprognosen aus Wurf ableiten.
     *
     * @param spieler
     *            Spieler, der an Reihe ist.
     */
    void prognostiziere(Spieler spieler) {
        tippe(spieler);
        double max = findeMaximum(spieler);
        double min = findeMinimum(spieler, max);
        gewichte(spieler, min, max);

    }

    /**
     * @param eintrag
     *            Der zu untersuchende Eintrag.
     * @return Wrfelauswahl entspricht Vorschlag fr Eintrag.
     */
    boolean vorgeschlagen(Tabzeile eintrag) {
        int[] anzahlenJeAugenAusgewaehlte = new int[Wuerfel.AUGEN];
        int[] anzahlenJeAugenVorgeschlagene = new int[Wuerfel.AUGEN];
        for (Wuerfel w : wuerfel) {
            int a = w.augen() - 1;
            if (w.isSelected()) {
                anzahlenJeAugenAusgewaehlte[a]++;
            }
            if (eintrag(eintrag).erneut(w)) {
                anzahlenJeAugenVorgeschlagene[a]++;
            }
        }
        for (int a = 0; a < Wuerfel.AUGEN; a++) {
            if (anzahlenJeAugenAusgewaehlte[a] != anzahlenJeAugenVorgeschlagene[a]) {
                return false;
            }
        }
        return true;
    }

    /**
     * @param eintrag
     *            Eintragsnummer.
     * @return Wichtung fr erstellte Vorschlge.
     */
    double wichtung(Tabzeile eintrag) {
        return eintrag(eintrag).wichtung();
    }

}
