/******************************************************************************
 ** $Id: Auswahl.java 743 2015-05-31 00:46:40Z wmh $
 ** Diese Datei ist Bestandteil der Java-Quelltexte des Wrfelspiels JaFuffy.
 ** Lauffhig ab Java 6.
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software; you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation; either version 2 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program; if not, write to the Free Software
 ** Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 ******************************************************************************
 ** Die aktuellste Version von JaFuffy findet sich im Internet unter
 ** <http://jafuffy.3kelvin.de>.
 **
 ** Kommentare, Fehler oder Erweiterungswnsche bitte per E-Mail senden an
 ** <jafuffy@3kelvin.de>.
 ******************************************************************************/
package jafuffy.bedienung;

import jafuffy.Eigenschaften;
import jafuffy.logik.Auswertung;
import jafuffy.logik.Tabzeile;
import jafuffy.logik.Turnier;

import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.text.NumberFormat;
import java.util.Locale;

import javax.swing.BorderFactory;
import javax.swing.DefaultCellEditor;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JTable;

/**
 * Auswahl der Zelle, in die das Wrfelresultat gesetzt wird. Setzt auf Wunsch in die Tabelle
 * (nach 1mal bzw. 2mal Klick).
 */
@SuppressWarnings("serial")
public abstract class Auswahl extends DefaultCellEditor {

    /** Innenabstand der Zelle. */
    private static final int ABSTAND = 4;

    /** Akustisches Signal fr besonderes Wrfelergebnis. */
    static final Geraeusch HURRA = new Geraeusch("Hurra.wav");
    /** Zur Formatierung der Wichtung. */
    static final NumberFormat NF;
    static {
        NF = NumberFormat.getInstance(Locale.GERMAN);
        NF.setMinimumFractionDigits(2);
        NF.setMaximumFractionDigits(2);
    }

    /** Setzen. */
    private final JButton setzen;

    /** Turnier. */
    protected Turnier turnier;
    /** Auswertung. */
    protected Auswertung auswertung;
    /** Wie oft angeklickt? */
    protected int clicks;
    /** Aktuell angewhlter Eintrag des aktiven Spielers. */
    protected Tabzeile eintrag;
    /** Tabelle zum Neuzeichnen. */
    protected Tabelle tabelle;

    /**
     * Konstruktor.
     *
     * @param turnier
     *            Das Turnier, fr welches die Auswahl getroffen werden soll.
     */
    Auswahl(Turnier turnier) {
        super(new JCheckBox());
        this.turnier = turnier;
        this.auswertung = turnier.auswertung();
        setClickCountToStart(1);
        setzen = new JButton("setzen");
        setzen.setBorder(BorderFactory.createEmptyBorder(0, ABSTAND, 0, ABSTAND));
        setzen.setFocusPainted(false);
        setzen.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                geklickt();
            }
        });
    }

    /** Button klicken (beachtet Einfachklick oder nicht). */
    public void doClick() {
        setzen.doClick();
        if (!Eigenschaften.GLOBAL.aktiv("Einfachklick")) {
            setzen.doClick();
        }
    }

    /**
     * Aktiven Eintrag auswhlen.
     *
     * @param eintrag
     *            Tabellenzeile, die als aktiver Eintrag ausgewhlt wird.
     */
    void aktiviere(Tabzeile eintrag) {
        this.eintrag = eintrag;
        clicks = 0;
        editorComponent = setzen;
    }

    /**
     * @return Zelleneditor (JButton)
     */
    @Override
    public Component getTableCellEditorComponent(JTable table, Object value,
            boolean isSelected, int row, int col) {
        aktiviere(Tabzeile.eintrag(row));
        tabelle = (Tabelle) table;
        aktualisiere();
        return editorComponent;
    }

    /** Manahmen, falls Eintrag angeklickt wurde. */
    abstract void geklickt();

    /** Durchzufhrende Aktionen nach dem Wrfeln, z.B. Tipp aktualisieren. */
    void gewuerfelt() {
        aktualisiere();
    }

    /** Tipp aktualisieren. */
    private void aktualisiere() {
        String text;
        if (eintrag != null) {
            fireEditingCanceled();
            text = "1 Klick = " + auswertung.wert(eintrag) + " Punkte";
            if (auswertung.tipps()) {
                text =
                        "<html><p>" + text + "</p>" + "<p>Endstandwertung "
                                + NF.format(auswertung.wichtung(eintrag)) + "</p></html>";
            }
            editorComponent.setToolTipText(text);
        }
    }

}
