/******************************************************************************
 ** $Id: Spieler.java 739 2015-05-23 16:13:32Z wmh $
 ** Diese Datei ist Bestandteil der Java-Quelltexte des Wrfelspiels JaFuffy.
 ** Lauffhig ab Java 6.
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software; you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation; either version 2 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program; if not, write to the Free Software
 ** Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 ******************************************************************************
 ** Die aktuellste Version von JaFuffy findet sich im Internet unter
 ** <http://jafuffy.3kelvin.de>.
 **
 ** Kommentare, Fehler oder Erweiterungswnsche bitte per E-Mail senden an
 ** <jafuffy@3kelvin.de>.
 ******************************************************************************/
package jafuffy.logik;

import java.io.Serializable;
import java.util.HashMap;

/** Spieler; enthlt Punktzahlen und weitere Daten, die den Stand beschreiben. */
public class Spieler implements Comparable<Spieler>, Cloneable, Serializable {

    private static final long serialVersionUID = -5170511106942700515L;

    /** Maximale Anzahl der Spieler. */
    public static final int SPIELER = 4;
    /** Grenzpunktzahl fr Bonus. */
    static final int GRENZE = 63;
    /** Bonus. */
    static final int BONUS = 35;

    // Spielername
    private final String name;
    // Punktstnde (inkl. Zwischensummen, Endsumme)
    private final HashMap<Tabzeile, Integer> punktzahlen = new HashMap<Tabzeile, Integer>(
            Tabzeile.ANZAHL);
    // Wo sind die freien Eintrge?
    private final HashMap<Tabzeile, Boolean> setzbar = new HashMap<Tabzeile, Boolean>(
            Tabzeile.ANZAHL);
    // Wie oft darf der Spieler noch wrfeln?
    private int rest;
    // Schon gewrfelt? (Nur der aktive Spieler kann wrfeln!)
    private boolean gewuerfelt;
    // Spieler hat genau einmal gewrfelt
    private boolean einmalig;
    // Zurckmeldung ber nderungen von Daten an Punktemodell
    private transient Punkte.Meldung meldung;

    /**
     * Konstruktor.
     *
     * @param name
     *            Name des Spielers
     */
    public Spieler(String name) {
        this.name = name;
        vorbereiten();
    }

    /** Punktetabelle mchte Spielerdaten beobachten. */
    void beobachten(Punkte.Meldung meldung) {
        this.meldung = meldung;
    }

    /**
     * @return Spielername
     */
    @Override
    public String toString() {
        return name;
    }

    /**
     * @return Vorzeichen bei Vergleich
     */
    @Override
    public int compareTo(Spieler spieler) {
        if (endsumme() < spieler.endsumme()) {
            return -1;
        }
        if (endsumme() > spieler.endsumme()) {
            return +1;
        }
        return 0;
    }

    /**
     * Neue Spielrunde beginnen.
     */
    void vorbereiten() {
        for (Tabzeile z : Tabzeile.values()) {
            punktzahl(z, 0);
            switch (z) {
            case GESAMT:
            case BONUS:
            case OSUMME:
            case USUMME:
            case ESUMME:
                setzbarkeit(z, false);
                break;
            default:
                setzbarkeit(z, true);
            }
        }
        warten();
    }

    /**
     * Der Spieler wird aktiviert, um fr diese Runde seine Spielaktionen durchzufhren.
     */
    void aktivieren() {
        gewuerfelt = false;
        einmalig = false;
        rest = 3;
        melden();
    }

    /**
     * Der Spieler wird re-aktiviert, um seinen Setzvorgang rckgngig zu machen, oder um
     * Setzhilfen zu aktualisieren.
     */
    void reaktivieren() {
        melden();
    }

    /**
     * Spielerdaten unmittelbar vor dem Setzen kopieren fr eine Rcknahme.
     *
     * @return Objektkopie
     */
    Spieler kopie() {
        Spieler spieler = new Spieler(name);
        spieler.punktzahlen.putAll(punktzahlen);
        spieler.setzbar.putAll(setzbar);
        spieler.rest = 0;
        spieler.gewuerfelt = true;
        spieler.einmalig = einmalig;
        spieler.meldung = meldung;
        return spieler;
    }

    /**
     * Spieler hat gewrfelt.
     */
    void wuerfeln() {
        gewuerfelt = true;
        einmalig = rest == 3;
        rest--;
        melden();
    }

    /**
     * Erwrfelte Punktzahl nur setzen, aber Wrfel noch nicht abgeben zum nchsten Spieler.
     *
     * @param eintrag
     *            Eintragszeile auf dem Punktezettel
     * @param wert
     *            einzutragender Wert
     */
    public void eintragen(Tabzeile eintrag, int wert) {
        setzen(eintrag, wert, false);
    }

    /**
     * Erwrfelte Punktzahl setzen und auf nchste Spieleraktion warten.
     *
     * @param eintrag
     *            Eintragszeile auf dem Punktezettel
     * @param wert
     *            einzutragender Wert
     */
    void setzen(Tabzeile eintrag, int wert) {
        setzen(eintrag, wert, true);
    }

    /**
     * Der Spieler wartet, bis er an die Reihe kommt.
     */
    void warten() {
        gewuerfelt = false;
        einmalig = false;
        rest = 0;
        melden();
    }

    /**
     * @return Wie oft darf der Spieler noch wrfeln?
     */
    public int rest() {
        return rest;
    }

    /**
     * @return Hat der Spieler gewrfelt?
     */
    public boolean gewuerfelt() {
        return gewuerfelt;
    }

    /**
     * @return Hat der Spieler genau einmal gewrfelt?
     */
    boolean einmalig() {
        return einmalig;
    }

    /**
     * @param eintrag
     *            Nummer des Eintrags auf dem Punktezettel
     * @return Schon erwrfelte Punktzahl.
     */
    public int punkte(Tabzeile eintrag) {
        return punktzahl(eintrag);
    }

    /** @return Gesamtsumme */
    int gesamt() {
        return punktzahl(Tabzeile.GESAMT);
    }

    /**
     * @return Endsumme
     */
    public int endsumme() {
        return punktzahl(Tabzeile.ESUMME);
    }

    /**
     * Setzbarkeit fr einen Eintrag merken.
     *
     * @param eintrag
     *            Nummer des Eintrags auf dem Punktezettel
     * @param aktiviert
     *            Ist der Eintrag aktivert zum Setzen?
     */
    void setzbarkeit(Tabzeile eintrag, boolean aktiviert) {
        setzbar.put(eintrag, aktiviert);
    }

    /**
     * Setzbarkeit fr einen Eintrag zurckholen.
     *
     * @param eintrag
     *            Nummer des Eintrags auf dem Punktezettel
     * @return Ist der Eintrag noch setzbar?
     */
    boolean setzbar(Tabzeile eintrag) {
        return setzbar.get(eintrag);
    }

    /**
     * Erwrfelte Punktzahl setzen und auf nchste Spieleraktion warten.
     *
     * @param eintrag
     *            Eintragszeile auf dem Punktezettel
     * @param wert
     *            einzutragender Wert
     * @param weiter
     *            Gibt an, ob der Spieler die Wrfel zum nchsten Spieler weiter gibt.
     */
    private void setzen(Tabzeile eintrag, int wert, boolean weiter) {
        int oben = 0;
        int bonus = 0;
        int unten = 0;

        punktzahl(eintrag, wert);
        setzbarkeit(eintrag, false);

        for (Tabzeile z : Tabzeile.OBEN) {
            oben += punktzahl(z);
        }
        punktzahl(Tabzeile.GESAMT, oben);
        if (punktzahl(Tabzeile.GESAMT) >= GRENZE) {
            bonus = BONUS;
        }
        punktzahl(Tabzeile.BONUS, bonus);
        punktzahl(Tabzeile.OSUMME, oben + bonus);
        for (Tabzeile z : Tabzeile.UNTEN) {
            unten += punktzahl(z);
        }
        punktzahl(Tabzeile.USUMME, unten);
        punktzahl(Tabzeile.ESUMME, oben + bonus + unten);

        if (weiter) {
            warten();
        } else {
            reaktivieren();
        }
    }

    /** Punktzahl fr Eintrag merken */
    private void punktzahl(Tabzeile eintrag, int wert) {
        punktzahlen.put(eintrag, wert);
    }

    /** Gemerkte Punktzahl fr einen Eintrag zurckmelden */
    private int punktzahl(Tabzeile eintrag) {
        return punktzahlen.get(eintrag);
    }

    /** nderungen ber Daten an Punktemodell zurckmelden */
    private void melden() {
        if (meldung != null) {
            meldung.mitteilen();
        }
    }

}
