/******************************************************************************
 ** $Id: Tabzeile.java 739 2015-05-23 16:13:32Z wmh $
 ** Diese Datei ist Bestandteil der Java-Quelltexte des Wrfelspiels JaFuffy.
 ** Lauffhig ab Java 6.
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************
 ** Die aktuellste Version von JaFuffy findet sich im Internet unter
 ** <http://jafuffy.3kelvin.de>.
 **
 ** Kommentare, Fehler oder Erweiterungswnsche bitte per E-Mail senden an
 ** <jafuffy@3kelvin.de>.
 ******************************************************************************/
package jafuffy.logik;

/** Zeilenarten der Tabelle (Eintrge). */
public enum Tabzeile {

    /** Augenzahl Eins wird gezhlt. */
    EINS("1er", "nur Einser zhlen", Art.OBEN),
    /** Augenzahl Zwei wird gezhlt. */
    ZWEI("2er", "nur Zweier zhlen", Art.OBEN),
    /** Augenzahl Drei wird gezhlt. */
    DREI("3er", "nur Dreier zhlen", Art.OBEN),
    /** Augenzahl Vier wird gezhlt. */
    VIER("4er", "nur Vierer zhlen", Art.OBEN),
    /** Augenzahl Fnf wird gezhlt. */
    FUENF("5er", "nur Fnfer zhlen", Art.OBEN),
    /** Augenzahl Sechs wird gezhlt. */
    SECHS("6er", "nur Sechser zhlen", Art.OBEN),
    /**
     * Die Gesamtpunktzahl der oberen Eintrge, bei denen je eine Augenzahl zum Ergebnis
     * beitrgt.
     */
    GESAMT("gesamt", "Summe der oberen Eintrge"),
    /** Bonus, falls Gesamtpunktzahl der oberen Eintrge 63 berschreitet. */
    BONUS("Bonus ab 63", "ab 63 Punkten oben zustzlich 35 Bonuspunkte"),
    /** Gesamtpunktzahl der oberen Eintrge einschlielich Bonus. */
    OSUMME("obere Summe", "Summe der oberen Eintrge plus Bonuspunkte"),
    /** Dreierpasch. */
    DREIERPASCH("Dreierpasch", "alle Augen zhlen", Art.UNTEN),
    /** Viererpasch. */
    VIERERPASCH("Viererpasch", "alle Augen zhlen", Art.UNTEN),
    /** Full-House. */
    FULLHOUSE("Full-House", "25 Punkte", Art.UNTEN),
    /** Kleine Strae. */
    KLEINESTRASSE("Kleine Strae", "30 Punkte", Art.UNTEN),
    /** Groe Strae. */
    GROSSESTRASSE("Groe Strae", "40 Punkte", Art.UNTEN),
    /** JaFuffy (fnf mal gleiche Augenzahl). */
    JAFUFFY("JaFuffy", "50 Punkte", Art.UNTEN),
    /** Chance (Summe aller Augen). */
    CHANCE("Chance", "alle Augen zhlen", Art.UNTEN),
    /** Summer der unteren Eintrge. */
    USUMME("untere Summe", "Summe der unteren Eintrge"),
    /** Endsumme. */
    ESUMME("Endsumme", "obere und untere Summe zusammen");

    /** Arte des Tabelleneintrages. */
    private enum Art {
        /** Der Eintrag ist nicht setzbar und enthlt abgeleitete Ergebnisse. */
        DERIVAT,
        /** Der Eintrag ist setzbar und befindet sich im oberen Tabellenteil. */
        OBEN,
        /** Der Eintrag ist setzbar und befindet sich im unteren Tabellenteil. */
        UNTEN;
    }

    /** Alle Eintrge, fr die ein Wurf gesetzt werden kann. */
    static final Tabzeile[] ALLE = { EINS, ZWEI, DREI, VIER, FUENF, SECHS, DREIERPASCH,
            VIERERPASCH, FULLHOUSE, KLEINESTRASSE, GROSSESTRASSE, JAFUFFY, CHANCE };

    /** Die Anzahl aller Tabelleneintrge. */
    static final int ANZAHL = values().length;
    /** Alle oberen Eintrge. */
    public static final Tabzeile[] OBEN = { EINS, ZWEI, DREI, VIER, FUENF, SECHS };
    /** Alle unteren Eintrge. */
    public static final Tabzeile[] UNTEN = { DREIERPASCH, VIERERPASCH, FULLHOUSE,
            KLEINESTRASSE, GROSSESTRASSE, JAFUFFY, CHANCE };

    /**
     * Wandelt Nummerierung in Wert um.
     *
     * @param nummer
     *            Nummer des Aufzhlungswertes.
     * @return Aufzhlungswert.
     */
    public static Tabzeile eintrag(int nummer) {
        return values()[nummer];
    }

    /**
     * Wandelt Augenzahl in Aufzhlungswert um.
     *
     * @param augen
     *            Augenzahl.
     * @return Aufzhlungswert.
     */
    public static Tabzeile oben(int augen) {
        return eintrag(augen - 1);
    }

    /** Tabelleneintragsbezeichung. */
    private final String bezeichnung;
    /** Tabelleneintragsbeschreibung. */
    private final String beschreibung;
    /** Art des Tabelleneintrages. */
    private final Art art;

    /**
     * Erstellt Aufzhlungswert.
     *
     * @param bezeichnung
     *            Tabelleneintragsbezeichung.
     * @param beschreibung
     *            Tabelleneintragsbeschreibung.
     */
    Tabzeile(String bezeichnung, String beschreibung) {
        this(bezeichnung, beschreibung, Art.DERIVAT);
    }

    /**
     * Erstellt Aufzhlungswert.
     *
     * @param bezeichnung
     *            Tabelleneintragsbezeichung.
     * @param beschreibung
     *            Tabelleneintragsbeschreibung.
     * @param art
     *            Art des Aufzhlungswertes.
     */
    Tabzeile(String bezeichnung, String beschreibung, Art art) {
        this.bezeichnung = bezeichnung;
        this.beschreibung = beschreibung;
        this.art = art;
    }

    /** @return Erwrfelter Eintrag in Tabelle? */
    public boolean alle() {
        return oben() || unten();
    }

    /** @return Erwrfelter Eintrag im oberen Tabellenteil? */
    boolean oben() {
        return art == Art.OBEN;
    }

    /** @return Erwrfelter Eintrag im unteren Tabellenteil? */
    boolean unten() {
        return art == Art.UNTEN;
    }

    /**
     * @return Beschreibung.
     */
    public String beschreibung() {
        return beschreibung;
    }

    /**
     * @return Bezeichnung.
     */
    public String bezeichnung() {
        return bezeichnung;
    }

}
