/******************************************************************************
 ** $Id: Wuerfelbild.java 684 2015-01-27 21:56:05Z wmh $
 ** Diese Datei ist Bestandteil der Java-Quelltexte des Wrfelspiels JaFuffy.
 ** Lauffhig ab Java 6.
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************
 ** Die aktuellste Version von JaFuffy findet sich im Internet unter
 ** <http://jafuffy.3kelvin.de>.
 **
 ** Kommentare, Fehler oder Erweiterungswnsche bitte per E-Mail senden an
 ** <jafuffy@3kelvin.de>.
 ******************************************************************************/
package jafuffy;

import java.awt.BorderLayout;
import java.awt.event.MouseAdapter;
import java.util.Arrays;

import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JPanel;

/** Wrfelreihenteil des Wrfelfeldes. */
@SuppressWarnings("serial")
class Wuerfelreihe extends JPanel {
    /** Halber Mindestabstand zwischen zwei Wrfeln. */
    private static final int MINDESTWUERFELABSTAND = 2;
    /** Wrfel, die im laufenden Turnier verwendet werden. */
    private Wuerfel[] wuerfel;
    /** Alle Wrfeldarstellungen. */
    private final Wuerfelbild[] wuerfelbilder = new Wuerfelbild[Turnier.WUERFEL];

    /**
     * Konstruiert die Wrfelreihe.
     *
     * @param adapter
     *            Horcht auf Mausereignisse.
     * */
    Wuerfelreihe(MouseAdapter adapter) {
        super(new BorderLayout());
        for (int i = 0; i < Turnier.WUERFEL; i++) {
            wuerfelbilder[i] = new Wuerfelbild(i);
        }
        add(baue(adapter), BorderLayout.CENTER);
    }

    /**
     * Baut die Wrfelauswahl zusammen.
     *
     * @param adapter
     *            Horcht auf Mausereignisse.
     *
     * @return Wrfelreihe.
     */
    private JPanel baue(MouseAdapter adapter) {
        JPanel wuerfelauswahl = new JPanel();
        wuerfelauswahl.setLayout(new BoxLayout(wuerfelauswahl, BoxLayout.X_AXIS));
        for (Wuerfelbild bild : wuerfelbilder) {
            wuerfelauswahl.add(Box.createHorizontalGlue());
            wuerfelauswahl.add(Box.createHorizontalStrut(MINDESTWUERFELABSTAND));
            wuerfelauswahl.add(bild);
            wuerfelauswahl.add(Box.createHorizontalStrut(MINDESTWUERFELABSTAND));
        }
        wuerfelauswahl.add(Box.createHorizontalGlue());
        wuerfelauswahl.addMouseListener(adapter);
        return wuerfelauswahl;
    }

    /**
     * Aktiviert die Wrfelauswahl, so dass Wrfel ausgewhlt werden knnen.
     *
     * @param fertig
     *            Gibt an, ob der Spieler fertig ist.
     */
    void aktiviere(boolean fertig) {
        for (Wuerfelbild wuerfelbild : wuerfelbilder) {
            wuerfelbild.setEnabled(!fertig);
        }
    }

    /**
     * Vorbereitung auf nchsten Spieler, sofern Turnier noch nicht beendet.
     *
     * @param rest
     *            Anzahl der restlichen Wrfe
     */
    void akzeptiere(int rest) {
        for (Wuerfelbild bild : wuerfelbilder) {
            bild.setEnabled(true);
            bild.setSelected(rest == Turnier.RUNDEN);
        }
    }

    /** Beendet das Turnier. */
    void beende() {
        for (Wuerfelbild bild : wuerfelbilder) {
            bild.setModel(null);
        }

    }

    /**
     * Klickt alle Wrfel mit dem vorgegebenen Auswahlzustand an, um die Klick-Aktion
     * auszufhren.
     *
     * @param ausgewaehlt
     *            Gibt an, ob der die anzuklickenden Wrfel ausgewhlt sein sollen.
     */
    void klicke(boolean ausgewaehlt) {
        for (Wuerfelbild bild : wuerfelbilder) {
            if (bild.isSelected() == ausgewaehlt) {
                bild.doClick();
            }
        }
    }

    /** Pausiere, um jegliche Aktionen verhindern. */
    void pausiere() {
        for (Wuerfelbild bild : wuerfelbilder) {
            bild.setEnabled(false);
            bild.setSelected(false);
        }
    }

    /** @return Gibt ob irgendein Wrfel ausgewhlt ist. */
    boolean selektiert() {
        boolean selektiert = false;
        for (Wuerfelbild bild : wuerfelbilder) {
            selektiert = selektiert || bild.isSelected();
        }
        return selektiert;
    }

    /** Sortiere Wrfel fr die Anzeige in der Reihe. */
    void sortiere() {
        Arrays.sort(wuerfel);
        verwende(wuerfel);
    }

    /** @return Gibt an, ob die Wrfelreihe der Gre nach sortiert ist. */
    boolean sortiert() {
        boolean sortiert = true;
        for (int w = 0; w < Turnier.WUERFEL - 1; w++) {
            sortiert = sortiert && wuerfel[w].augen() <= wuerfel[w + 1].augen();
        }
        return sortiert;
    }

    /** Vorschlag zur Wrfelauswahl bernehmen. */
    void uebernehme() {
        for (Wuerfelbild bild : wuerfelbilder) {
            if (bild.isSelected() != bild.modell().vorgeschlagen()) {
                bild.doClick();
            }
        }
    }

    /**
     * Setzt Modelle, wie sie von den Wrfelbildern verwendet werden.
     *
     * @param wuerfel
     *            Die zu verwendenden Wrfelmodelle.
     */
    void verwende(Wuerfel[] wuerfel) {
        this.wuerfel = wuerfel;
        for (int w = 0; w < Turnier.WUERFEL; w++) {
            wuerfelbilder[w].setModel(wuerfel[w]);
        }
    }
}
