/******************************************************************************
 ** $Id: Spielplatz.java 687 2015-02-04 02:06:39Z wmh $
 ** Diese Datei ist Bestandteil der Java-Quelltexte des Wrfelspiels JaFuffy.
 ** Lauffhig ab Java 6.
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************
 ** Die aktuellste Version von JaFuffy findet sich im Internet unter
 ** <http://jafuffy.3kelvin.de>.
 **
 ** Kommentare, Fehler oder Erweiterungswnsche bitte per E-Mail senden an
 ** <jafuffy@3kelvin.de>.
 ******************************************************************************/package jafuffy;

import java.awt.BorderLayout;
import java.awt.CardLayout;
import java.awt.Font;
import java.awt.GridLayout;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.ArrayList;
import java.util.Arrays;

import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.ImageIcon;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.SwingConstants;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

/** Erstellung des Spielplatzes, der den Zettel und das Wrfelfeld enthlt. */
@SuppressWarnings("serial")
class Spielplatz extends JPanel implements PropertyChangeListener, ChangeListener {

    /** Intro-Grafik. */
    static final String INTRO = "Intro";
    /** Intro-Text . */
    static final String TEXT = "Text";
    /** Spielfeld. */
    static final String FELD = "Feld";

    /** Rand der Komponenten. */
    private static final int KOMPONENTENRAND = 2;
    /** Icon fr Gewinnerbekanntgabe. */
    private static final ImageIcon KRANZ = Grafik.icon("Kranz.png");
    /** Titel-Font. */
    private static final Font TITEL = new Font("SansSerif", Font.BOLD | Font.ITALIC, 32);
    /** Untertitel-Font. */
    private static final Font UNTERTITEL = new Font("SansSerif", Font.BOLD, 24);

    /** Zettel auf dem die Punkte notiert werden. */
    private final Zettel zettel = new Zettel();
    /** Das Wrfelfeld mit Wrfelbecher, allen fnf Wrfeln und ein Statusfeld. */
    private final Wuerfelfeld wuerfelfeld = new Wuerfelfeld();

    /** Konstruktor. */
    Spielplatz() {
        super(new CardLayout());
        UIManager.addPropertyChangeListener(this);
        /*
         * Mit dieser Hilfskonstruktion kann sich der Zettel vom Platz her gegen das Wrfelfeld
         * behaupten. Andernfalls wre die maximale Gre beschrnkt.
         */
        JPanel zettelfeld = new JPanel(new BorderLayout());
        zettelfeld.add(zettel, BorderLayout.CENTER);
        add(spielfeld(zettelfeld), FELD);
        add(new Intro(), INTRO);
        add(titelfeld(), TEXT);
        ((CardLayout) getLayout()).show(this, Eigenschaften.GLOBAL.aktiv("Intro") ? INTRO
                : TEXT);
    }

    /**
     * Realisiert Vernderungen des Look&Feel.
     *
     * @param evt
     */
    @Override
    public void propertyChange(PropertyChangeEvent evt) {
        if (evt.getPropertyName().equals("lookAndFeel")) {
            SwingUtilities.updateComponentTreeUI(this);
        }
    }

    /**
     * ChangeEvent eingetreten.
     *
     * @param ce
     */
    @Override
    public void stateChanged(ChangeEvent ce) {
        wuerfelfeld.stateChanged(ce);
        zettel.stateChanged(ce);
        if (CEJaFuffy.adressiert(ce, CEAblauf.class)) {
            switch (CEJaFuffy.<CEAblauf> ereignis(ce)) {
            case START:
                // Spielfeld zeigen
                ((CardLayout) getLayout()).show(this, Spielplatz.FELD);
                break;
            case RESULTAT:
                zeigeTurnierausgang((Turnier) ce.getSource());
                break;
            default:
            }
        }
    }

    /**
     * Erzeugt ein Spielfeld fr ein Turnier, in dem links der Zettel und rechts daneben das
     * Wrfelfeld liegen.
     *
     * @param zettelfeld
     *            Zettel
     * @return Spielfeld
     */
    private JPanel spielfeld(JPanel zettelfeld) {
        JPanel spielfeld = new JPanel();
        spielfeld.setLayout(new BoxLayout(spielfeld, BoxLayout.X_AXIS));
        zettelfeld.setBorder(BorderFactory.createEmptyBorder(KOMPONENTENRAND, KOMPONENTENRAND,
                KOMPONENTENRAND, KOMPONENTENRAND));
        spielfeld.add(zettelfeld);
        wuerfelfeld.setBorder(BorderFactory.createEmptyBorder(KOMPONENTENRAND, KOMPONENTENRAND,
                KOMPONENTENRAND, KOMPONENTENRAND));
        spielfeld.add(wuerfelfeld);
        return spielfeld;
    }

    /**
     * Erzeugt das Titelfeld, das als Begrung gezeigt wird.
     *
     * @return Titelfeld
     */
    private JPanel titelfeld() {
        JPanel titelfeld = new JPanel();
        titelfeld.setLayout(new BoxLayout(titelfeld, BoxLayout.PAGE_AXIS));
        JLabel titel = new JLabel("JaFuffy!", SwingConstants.CENTER);
        titel.setEnabled(false);
        titel.setFont(TITEL);
        titel.setAlignmentX(CENTER_ALIGNMENT);
        titelfeld.add(titel, BorderLayout.CENTER);
        JLabel untertitel = new JLabel("- Das Wrfelspiel -", SwingConstants.CENTER);
        untertitel.setEnabled(false);
        untertitel.setFont(UNTERTITEL);
        untertitel.setAlignmentX(CENTER_ALIGNMENT);
        titelfeld.add(Box.createVerticalGlue());
        titelfeld.add(Box.createVerticalGlue());
        titelfeld.add(titel);
        titelfeld.add(Box.createVerticalGlue());
        titelfeld.add(untertitel);
        titelfeld.add(Box.createVerticalGlue());
        titelfeld.add(Box.createVerticalGlue());
        return titelfeld;
    }

    /**
     * Mitteilung ber Spielende und Platzbelegung der Spieler aus dem Turnier.
     *
     * @param turnier
     *            Das Turnier, fr das eine Mitteilung erfolgt.
     */
    private void zeigeTurnierausgang(Turnier turnier) {

        JPanel rangliste = new JPanel();
        ArrayList<Spieler> spieler = turnier.spieler();

        if (spieler.size() == 1) {
            rangliste.setLayout(new GridLayout(1, 2));
            rangliste.add(new JLabel(spieler.get(0).toString(), SwingConstants.CENTER));
            rangliste.add(new JLabel(spieler.get(0).endsumme() + " Punkte",
                    SwingConstants.CENTER));
        } else {
            Object[] plaetze = spieler.toArray();
            Arrays.sort(plaetze);
            int lowscore = ((Spieler) plaetze[spieler.size() - 1]).endsumme();
            rangliste.setLayout(new GridLayout(0, 3));
            for (int i = spieler.size() - 1, score, rang = 1; i >= 0; i--) {
                score = ((Spieler) plaetze[i]).endsumme();
                if (score < lowscore) {
                    rang++;
                    lowscore = score;
                }
                rangliste.add(new JLabel(rang + ". Platz: "));
                rangliste.add(new JLabel(((Spieler) plaetze[i]).toString()));
                rangliste.add(new JLabel(score + " Punkte", SwingConstants.RIGHT));
            }
        }

        JOptionPane.showMessageDialog(this, rangliste, turnier.nummer() < turnier.anzahl()
                || turnier.anzahl() == 0 ? "JaFuffy (Resultat " + turnier.nummer() + ". Spiel)"
                : "JaFuffy (Resultat letztes Spiel)", JOptionPane.INFORMATION_MESSAGE, KRANZ);

    }

}
