/******************************************************************************
 ** $Id: Ende.java 697 2015-02-08 15:47:26Z wmh $
 ** Diese Datei ist Bestandteil der Java-Quelltexte des Wrfelspiels JaFuffy.
 ** Lauffhig ab Java 6.
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software; you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation; either version 2 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program; if not, write to the Free Software
 ** Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 ******************************************************************************
 ** Die aktuellste Version von JaFuffy findet sich im Internet unter
 ** <http://jafuffy.3kelvin.de>.
 **
 ** Kommentare, Fehler oder Erweiterungswnsche bitte per E-Mail senden an
 ** <jafuffy@3kelvin.de>.
 ******************************************************************************/
package jafuffy;

import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectOutputStream;

import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

/**
 * Nachfrage (nur bei Applikation), ob das Spiel beendet werden soll. Sichern des Turnierstands
 * und der Statistik bei Programmende (Turnierende/abbruch).
 */
class Ende extends WindowAdapter implements ChangeListener {

    /** Abzuspeichernde Statistik. */
    private final Statistik statistik;
    /** Umgebungsfenster. */
    private final Fenster fenster;
    /** Laufendes Turnier. */
    private Turnier turnier;

    /**
     * Konstruktor.
     *
     * @param statistik
     *            Geladene Statistik
     * @param fenster
     *            Umgebendes Fenster
     */
    Ende(Statistik statistik, Fenster fenster) {
        this.statistik = statistik;
        this.fenster = fenster;
    }

    /**
     * ChangeEvent eingetreten.
     *
     * @param ce
     *            Ereignis im Turnierablauf.
     */
    @Override
    public void stateChanged(ChangeEvent ce) {
        if (CEJaFuffy.adressiert(ce, CEAblauf.class)) {
            switch (CEJaFuffy.<CEAblauf> ereignis(ce)) {
            case START:
                this.turnier = (Turnier) ce.getSource();
                break;
            case ABBRUCH:
            case ENDE:
                turnier = null;
                break;
            default:
            }
        }
    }

    /**
     * Programm auf Anfrage beenden, und damit Fenster schlieen.
     *
     * @param we
     *            Fensterereignis.
     */
    @Override
    public void windowClosing(WindowEvent we) {
        if (Eigenschaften.GLOBAL.neustart()
                || JOptionPane.showConfirmDialog(fenster, "JaFuffy beenden?",
                        "JaFuffy (Frage)", JOptionPane.YES_NO_OPTION) == JOptionPane.YES_OPTION) {
            sichereTurnierEigenschaften();
            sichereStatistik();
            beendeApplikation();
        }
    }

    /** Beenden der Applikation bedeutet beenden des Turniers und schlieen des Fensters. */
    private void beendeApplikation() {
        // Turnier abgebrochen oder beendet
        if (turnier != null) {
            turnier.beende();
        }
        // Schluss
        fenster.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
    }

    /** Statistik sichern. */
    private void sichereStatistik() {
        try {
            ObjectOutputStream out =
                    new ObjectOutputStream(new FileOutputStream(Pfad.datei(Statistik.STAT)));
            out.writeObject(statistik);
            out.close();
        } catch (IOException e) {
            e.printStackTrace();
            Signal.ALARM.play();
            JOptionPane.showMessageDialog(fenster, "Statistik konnte nicht gesichert werden!\n"
                    + "Java-Exception:\n" + e, "JaFuffy (Fehler)", JOptionPane.ERROR_MESSAGE);
        }
    }

    /** Turniereigenschaften sichern. */
    private void sichereTurnierEigenschaften() {
        try {
            Eigenschaften.GLOBAL.speichere(fenster.getX(), fenster.getY(), fenster.getWidth(),
                    fenster.getHeight());
        } catch (IOException e) {
            e.printStackTrace();
            Signal.ALARM.play();
            JOptionPane.showMessageDialog(fenster,
                    "JaFuffy-Einstellungen konnten nicht gesichert werden!\n"
                            + "Java-Exception:\n" + e, "JaFuffy (Fehler)",
                    JOptionPane.ERROR_MESSAGE);
        }
    }

}
