/******************************************************************************
 ** $Id: AuswertungErweitert.java 591 2014-08-02 12:29:04Z wmh $
 ** Diese Datei ist Bestandteil der Java-Quelltexte des Wrfelspiels JaFuffy.
 ** Lauffhig ab Java 6.
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software; you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation; either version 2 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program; if not, write to the Free Software
 ** Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 ******************************************************************************
 ** Die aktuellste Version von JaFuffy findet sich im Internet unter
 ** <http://jafuffy.3kelvin.de>.
 **
 ** Kommentare, Fehler oder Erweiterungswnsche bitte per E-Mail senden an
 ** <jafuffy@3kelvin.de>.
 ******************************************************************************/
package jafuffy;

/** Erweiterte Auswertung. */
class AuswertungErweitert extends AuswertungKlassisch {

    private static final long serialVersionUID = -5463902030609191338L;

    /** Joker-Bonus. */
    static final int JOKER = 100;
    /** Erwartete Punktzahl pro Spiel (Schtzung). */
    private static final int MITTELWERT = 240;
    /** Standardabweichung (Schtzung). */
    private static final int ABWEICHUNG = 60;

    // Soll "Joker!" bzw. "Bonus"-Meldung erscheinen?
    private boolean joker = false;
    private boolean bonus = false;
    private boolean setzzwang = false;

    /** Regelname. */
    static final String REGEL = "Erweiterte Regeln";

    /**
     * Konstruktor.
     * 
     * @param variante
     *            Spielregelvariante
     * @param wuerfel
     *            Die im Turnier benutzten Wrfel.
     */
    public AuswertungErweitert(Variante variante, Wuerfel[] wuerfel) {
        super(variante, wuerfel);
    }

    /**
     * @return Regelname
     */
    @Override
    public String toString() {
        return REGEL;
    }

    /**
     * Textnachricht "Joker!" setzen?
     * 
     * @param setzzwang
     */
    void setzzwangnachricht(boolean setzzwang) {
        if (setzzwang) {
            // Spieler darf keinen Wrfel mehr in den Becher legen
            for (int w = 0; w < 5; w++) {
                wuerfel[w].setEnabled(false);
            }
        }
        this.setzzwang = setzzwang;
    }

    /**
     * Textnachricht "Joker!" setzen?
     * 
     * @param joker
     */
    void jokernachricht(boolean joker) {
        if (joker) {
            // Spieler darf keinen Wrfel mehr in den Becher legen
            for (int w = 0; w < 5; w++) {
                wuerfel[w].setEnabled(false);
            }
        }
        this.joker = joker;
    }

    /**
     * Textnachricht "Bonus!" setzen?
     * 
     * @param bonus
     */
    void bonusnachricht(boolean bonus) {
        this.bonus = bonus;
    }

    /**
     * Hat der Spieler einen Joker erwrfelt?
     * 
     * @return
     */
    boolean joker() {
        return super.wert(Tabzeile.JAFUFFY) > 0 && spieler.punkte(Tabzeile.JAFUFFY) > 0
                && !spieler.setzbar(Tabzeile.oben(wuerfel[0].augen()));
    }

    /**
     * Setzzwang (JaFuffy gewrfelt, JaFuffy-Feld belegt, oben frei)?
     * 
     * @return
     */
    boolean setzzwang() {
        return super.wert(Tabzeile.JAFUFFY) > 0 && spieler.punkte(Tabzeile.JAFUFFY) > 0
                && spieler.setzbar(Tabzeile.oben(wuerfel[0].augen()));
    }

    /**
     * @return geschtzte mittlere Punktzahl pro Spiel
     */
    @Override
    int mittelwert() {
        return MITTELWERT;
    }

    /**
     * @return geschtzte Abweichung von mittlerer Punktzahl pro Spiel
     */
    @Override
    int abweichung() {
        return ABWEICHUNG;
    }

    /**
     * Erwrfelte Punkte fr das Eintragsfeld. Bercksichtigt JaFuffy-Bonus.
     * 
     * @param eintrag
     *            Nummer des Eintrags auf dem Punktezettel
     * @return Punktzahl
     */
    @Override
    int wert(Tabzeile eintrag) {
        switch (eintrag) {
        case KLEINESTRASSE:
            return joker() || super.wert(eintrag) > 0 ? Analyse.KleineStrasse.PUNKTE : 0;
        case GROSSESTRASSE:
            return joker() || super.wert(eintrag) > 0 ? Analyse.GrosseStrasse.PUNKTE : 0;
        case JAFUFFY:
            if (super.wert(eintrag) > 0 && spieler.punkte(eintrag) > 0) {
                return spieler.punkte(eintrag) + JOKER;
            }
        default:
            return super.wert(eintrag);
        }
    }

    /**
     * @param eintrag
     *            Zeilennummer des Eintrags
     * @return Texteintrag fr Tabelle
     */
    @Override
    String text(Tabzeile eintrag) {
        if (eintrag == Tabzeile.JAFUFFY && joker) {
            return "Joker!";
        } else if (eintrag == Tabzeile.JAFUFFY && bonus) {
            return "Bonus!";
        } else if (setzzwang && eintrag == Tabzeile.oben(wuerfel[0].augen())) {
            return "Gesetzt!";
        } else {
            return null;
        }
    }

    /**
     * @param eintrag
     *            noch nicht gesetzter Eintrag
     * @return Tipp, wieviele Punkte erreichbar inklusive Joker-Meldung
     */
    @Override
    String spicker(Tabzeile eintrag) {
        if (Eigenschaften.GLOBAL.aktiv("Spicker") && spieler.gewuerfelt() && joker()
                && eintrag == Tabzeile.JAFUFFY) {
            String text;
            text =
                    "<p align=center><em>Joker (+ 100 Bonuspunkte)!</em></p>"
                            + "<p>Bitte Wurf in freies Feld eintragen.</p>";
            if (tipps()) {
                return "<html>" + text + "<p>Kontextmen!</p></html>";
            } else {
                return "<html>" + text + "</html>";
            }
        } else {
            return super.spicker(eintrag);
        }
    }

}
