/******************************************************************************
 ** $Id: Verfolgung.java 3520 2024-08-07 22:22:58Z wmh $
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package ebflmaennle.oberflaeche;

import java.awt.Component;
import java.awt.Point;
import java.awt.event.ComponentEvent;
import java.awt.event.ComponentListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;

import javax.swing.SwingUtilities;

import ebflmaennle.berechnung.Ausschnitt;
import ebflmaennle.berechnung.Kalkulator;
import ebflmaennle.berechnung.Kalkulator.Parameter;
import ebflmaennle.oberflaeche.Leinwand.Zeiger;

public class Verfolgung extends PropertyChangeSupport implements PropertyChangeListener {
    private static final long serialVersionUID = -2343637526422897305L;

    private final Ausschnitt ausschnitt;
    private final Kalkulator kalkulator;
    private final Component leinwand;

    private boolean beobachtbar;
    private Point position;
    private Zeiger zeiger;
    private Parameter parameter;

    public Verfolgung(final Ausschnitt ausschnitt, final Kalkulator kalkulator, final Component leinwand) {
        super(ausschnitt);
        this.ausschnitt = ausschnitt;
        this.kalkulator = kalkulator;
        this.leinwand = leinwand;
        leinwand.addComponentListener(new ComponentListener() {

            @Override
            public void componentShown(final ComponentEvent ereignis) {
            }

            @Override
            public void componentResized(final ComponentEvent ereignis) {
                ausschnitt.setze(ereignis.getComponent().getSize());
                kalkulator.beantrage();
            }

            @Override
            public void componentMoved(final ComponentEvent ereignis) {
            }

            @Override
            public void componentHidden(final ComponentEvent ereignis) {
            }
        });
        leinwand.addMouseListener(new MouseListener() {

            @Override
            public void mouseReleased(final MouseEvent ereignis) {
                melde(beobachtbar ? ereignis.getPoint() : null);
            }

            @Override
            public void mousePressed(final MouseEvent ereignis) {
                position = ereignis.getPoint();
            }

            @Override
            public void mouseExited(final MouseEvent ereignis) {
                beobachtbar = false;
                melde(null);
                zeiger = null;
                parameter = null;
            }

            @Override
            public void mouseEntered(final MouseEvent ereignis) {
                beobachtbar = true;
                melde(ereignis.getPoint());
            }

            @Override
            public void mouseClicked(final MouseEvent ereignis) {
                final var groesse = ausschnitt.leinwandgroesse();
                final var position = ereignis.getPoint();
                final var delta = new Point(groesse.width / 2 - position.x, groesse.height / 2 - position.y);
                final var vergroessert = ereignis.isControlDown();
                final var verkleinert = ereignis.isAltDown();
                final var verstaerkt = ereignis.isShiftDown();
                melde(position, delta, vergroessert, verkleinert, verstaerkt);
            }
        });
        leinwand.addMouseMotionListener(new MouseMotionListener() {

            @Override
            public void mouseMoved(final MouseEvent ereignis) {
                melde(ereignis.getPoint());
            }

            @Override
            public void mouseDragged(final MouseEvent ereignis) {
                if (!beobachtbar) {
                    return;
                }
                final var position = ereignis.getPoint();
                assert Verfolgung.this.position != null;
                final var delta = new Point(position.x - Verfolgung.this.position.x,
                        position.y - Verfolgung.this.position.y);
                assert delta.x != 0 || delta.y != 0;
                if (delta.x != 0 || delta.y != 0) {
                    melde(position, delta, false, false, false);
                }
            }
        });
        leinwand.addMouseWheelListener(ereignis -> {
            final var position = ereignis.getPoint();
            final var verstaerkt = ereignis.isShiftDown();
            final var radstufe = ereignis.getWheelRotation();
            melde(position, null, false, false, verstaerkt, radstufe);
        });
    }

    @Override
    public void propertyChange(final PropertyChangeEvent ereignis) {
        if ("SwingSelectedKey".equals(ereignis.getPropertyName()) && ereignis.getNewValue().equals(true)
                || "Berechnung".equals(ereignis.getPropertyName()) && ereignis.getNewValue().equals(false)) {
            zeiger = null;
            parameter = null;
            SwingUtilities.invokeLater(() -> melde(leinwand.getMousePosition()));
        }
    }

    private void melde(final Point position, final Point delta, final boolean vergroessert, final boolean verkleinert,
            final boolean verstaerkt, final int radstufe) {
        var veraendert = false;
        var fokusstaerke = 1;
        if (verstaerkt) {
            fokusstaerke = 2;
        }
        var zoomschritt = 0;
        if (vergroessert) {
            zoomschritt = Ausschnitt.zoomschritt(fokusstaerke);
            veraendert = ausschnitt.fokussiere(position, zoomschritt);
        } else if (verkleinert) {
            zoomschritt = Ausschnitt.zoomschritt(-fokusstaerke);
            veraendert = ausschnitt.fokussiere(position, zoomschritt);
        } else if (radstufe != 0) {
            zoomschritt = radstufe;
            if (verstaerkt) {
                zoomschritt *= 2;
            }
            veraendert = ausschnitt.skaliere(position, zoomschritt);
        }
        if (!veraendert) {
            zoomschritt = 0;
        }
        if (delta != null) {
            ausschnitt.verschiebe(delta);
            this.position = position;
            veraendert = true;
        }
        final var zeiger = new Zeiger(position, delta, zoomschritt);
        final var parameter = kalkulator.parameter(zeiger);
        firePropertyChange("Zeiger", this.zeiger, zeiger);
        this.zeiger = zeiger;
        firePropertyChange("Parameter", this.parameter, parameter);
        this.parameter = parameter;
        if (veraendert) {
            kalkulator.beantrage();
        }
    }

    private void melde(final Point position, final Point delta, final boolean vergroessert, final boolean verkleinert,
            final boolean verstaerkt) {
        melde(position, delta, vergroessert, verkleinert, verstaerkt, 0);
    }

    private void melde(final Point position) {
        melde(position, null, false, false, false);
    }

}