/******************************************************************************
 ** $Id: Tafel.java 3595 2024-09-07 18:39:14Z wmh $
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package ebflmaennle.oberflaeche;

import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.geom.Rectangle2D;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

import javax.swing.JDialog;
import javax.swing.JPanel;

import ebflmaennle.berechnung.Ausschnitt;
import ebflmaennle.berechnung.Option;

public class Tafel extends JDialog implements PropertyChangeListener {

    private static final long serialVersionUID = 1926712717463360602L;

    private static final String PLATZHALTER_DAUER = "#####0";
    private static final String PLATZHALTER_DARSTELLUNGSDAUER = "##0";
    private static final String PLATZHALTER_GESAMTDARSTELLUNGSDAUER = "###0";
    private static final String PLATZHALTER_AUSDEHNUNG = "0.000000E00";
    private static final String PLATZHALTER_AUFLOESUNG = "0.000E00";
    private static final String PLATZHALTER_GEOMETRIE = "###0";
    private static final String PLATZHALTER_ANTIALIASING = "nein";
    private static final String PLATZHALTER_ZOOM_STUFE = "#0.000";
    private static final String PLATZHALTER_ZOOM_EXP = "0.0E0";
    private static final String PLATZHALTER_DIMENSION = "0.0000";
    private static final Formatierer FORMAT_ZOOM_STUFE = new Formatierer(PLATZHALTER_ZOOM_STUFE);
    private static final Formatierer FORMAT_ZOOM_EXP = new Formatierer(PLATZHALTER_ZOOM_EXP);
    private static final Formatierer FORMAT_AUSDEHNUNG = new Formatierer(PLATZHALTER_AUSDEHNUNG);
    private static final Formatierer FORMAT_AUFLOESUNG = new Formatierer(PLATZHALTER_AUFLOESUNG);
    private static final Formatierer FORMAT_GEOMETRIE = new Formatierer(PLATZHALTER_GEOMETRIE);
    private static final Formatierer FORMAT_DIMENSION = new Formatierer(PLATZHALTER_DIMENSION);

    private static String indexzeile(final String text) {
        return "<html>" + text + "<sub> </sub></html>";
    }

    private final Messwert a1 = new Messwert(Formatierer.PLATZHALTER_KOORDINATE_POS + Formatierer.PLATZHALTER_ESIGN);
    private final Messwert b1 = new Messwert(Formatierer.PLATZHALTER_KOORDINATE_POS + Formatierer.PLATZHALTER_ESIGN);
    private final Messwert a2 = new Messwert(Formatierer.PLATZHALTER_KOORDINATE_POS + Formatierer.PLATZHALTER_ESIGN);
    private final Messwert b2 = new Messwert(Formatierer.PLATZHALTER_KOORDINATE_POS + Formatierer.PLATZHALTER_ESIGN);
    private final Messwert g = new Messwert(PLATZHALTER_AUSDEHNUNG + Formatierer.PLATZHALTER_ESIGN);
    private final Messwert h = new Messwert(PLATZHALTER_AUSDEHNUNG + Formatierer.PLATZHALTER_ESIGN);
    private final Messwert pxs = new Messwert(PLATZHALTER_AUSDEHNUNG + Formatierer.PLATZHALTER_ESIGN);
    private final Messwert zoom = new Messwert(
            "2^" + PLATZHALTER_ZOOM_STUFE + "=" + PLATZHALTER_ZOOM_EXP + "#" + Formatierer.PLATZHALTER_ESIGN);
    private final Messwert m = new Messwert(PLATZHALTER_GEOMETRIE);
    private final Messwert n = new Messwert(PLATZHALTER_GEOMETRIE);
    private final Messwert iterationslimit = new Messwert("2^" + Formatierer.PLATZHALTER_LIMIT_STUFE + "="
            + Formatierer.PLATZHALTER_LIMIT_EXP + Formatierer.PLATZHALTER_ESIGN);
    private final Messwert rechendauer = new Messwert(PLATZHALTER_DAUER);
    private final Messwert gesamtfaerbedauer = new Messwert(PLATZHALTER_GESAMTDARSTELLUNGSDAUER + "="
            + PLATZHALTER_DARSTELLUNGSDAUER + "+" + PLATZHALTER_DARSTELLUNGSDAUER);
    private final Messwert gesamtmaldauer = new Messwert(PLATZHALTER_GESAMTDARSTELLUNGSDAUER + "="
            + PLATZHALTER_DARSTELLUNGSDAUER + "+" + PLATZHALTER_DARSTELLUNGSDAUER);
    private final Messwert iterationsmaximum = new Messwert(
            Formatierer.PLATZHALTER_LIMIT_EXP + Formatierer.PLATZHALTER_ESIGN);
    private final Messwert dimension = new Messwert(PLATZHALTER_DIMENSION);
    private final Messwert antialiasing = new Messwert(PLATZHALTER_ANTIALIASING);
    private final Messwert bevorzugung = new Messwert(Option.BEVORZUGUNGSSTANDARD.toString());
    private final Messwert qualitaet = new Messwert(Option.QUALITAETSSTANDARD.toString());

    private int faerbedauerwert;
    private int restfaerbedauerwert;
    private int maldauerwert;
    private int restmaldauerwert;

    public Tafel(final Fenster fenster) {
        super(fenster, "Ebflmännle: Instrumententafel");
        setFocusableWindowState(false);
        setContentPane(tafel());
        pack();
        setLocationRelativeTo(fenster);
    }

    @Override
    public void propertyChange(final PropertyChangeEvent event) {
        final var name = event.getPropertyName();
        final var wert = event.getNewValue().toString();
        if (name == "Rechendauer") {
            final var rechendauerwert = (int) (((long) event.getNewValue()) / 1000000);
            rechendauer.setText(Integer.toString(rechendauerwert));
        } else if (name == "Färbedauer") {
            faerbedauerwert = (int) (((long) event.getNewValue()) / 1000000);
            aktualisiereGesamtfaerbedauer();
        } else if (name == "Restfärbedauer") {
            restfaerbedauerwert = (int) (((long) event.getNewValue()) / 1000000);
            aktualisiereGesamtfaerbedauer();
        } else if (name == "Maldauer") {
            maldauerwert = (int) (((long) event.getNewValue()) / 1000000);
            aktualisiereGesamtmaldauer();
        } else if (name == "Restmaldauer") {
            restmaldauerwert = (int) (((long) event.getNewValue()) / 1000000);
            aktualisiereGesamtmaldauer();
        } else if (name == "Auflösung") {
            pxs.setText(FORMAT_AUFLOESUNG.format(event.getNewValue()));
            final double zoomwert = Ausschnitt.BASISPXS / (float) event.getNewValue();
            zoom.setText("2^" + FORMAT_ZOOM_STUFE.formatiere(Math.log(zoomwert) / Math.log(2)) + "="
                    + FORMAT_ZOOM_EXP.formatiere(zoomwert));
        } else if (name == "Berechnung") {
            final boolean rechnend = (Boolean) event.getNewValue();
            rechendauer.setEnabled(!rechnend);
            gesamtfaerbedauer.setEnabled(!rechnend);
            gesamtmaldauer.setEnabled(!rechnend);
            iterationsmaximum.setEnabled(!rechnend);
        } else if (name == "Iterationslimit") {
            final var limit = (int) event.getNewValue();
            iterationslimit.setText("2^" + Formatierer.FORMAT_LIMIT_STUFE.formatiere(Math.log(limit) / Math.log(2))
                    + "=" + Formatierer.FORMAT_LIMIT_EXP.formatiere(limit));
        } else if (name == "Iterationsmaximum") {
            final var limit = (int) event.getNewValue();
            iterationsmaximum.setText(Formatierer.FORMAT_LIMIT_EXP.formatiere(limit));
        } else if (name == "Qualitaet") {
            qualitaet.setText(wert);
        } else if (name == "Dimension") {
            final var zahl = (double) event.getNewValue();
            if (Double.isNaN(zahl) || zahl == 0) {
                dimension.setText("n/a");
            } else {
                dimension.setText(FORMAT_DIMENSION.format(zahl));
            }
        } else if (name == "Bildgröße") {
            final var ausschnitt = (Dimension) event.getNewValue();
            m.setText(FORMAT_GEOMETRIE.formatiere(ausschnitt.width));
            n.setText(FORMAT_GEOMETRIE.formatiere(ausschnitt.height));
        } else if (name == "Antialiasing") {
            antialiasing.setText(((int) event.getNewValue()) == 1 ? "nein" : "<html>&times;" + wert + "</html>");
        } else if (name == "Sichtfenster") {
            final var fenster = (Rectangle2D) event.getNewValue();
            a1.setText(indexzeile(Formatierer.FORMAT_KOORDINATE.formatiere(fenster.getX())));
            a2.setText(indexzeile(Formatierer.FORMAT_KOORDINATE.formatiere(fenster.getX() + fenster.getWidth())));
            b1.setText(indexzeile(Formatierer.FORMAT_KOORDINATE.formatiere(fenster.getY())));
            b2.setText(indexzeile(Formatierer.FORMAT_KOORDINATE.formatiere(fenster.getY() + fenster.getHeight())));
            g.setText(FORMAT_AUSDEHNUNG.formatiere(fenster.getWidth()));
            h.setText(FORMAT_AUSDEHNUNG.formatiere(fenster.getHeight()));
        } else if (name == "Bevorzugung") {
            bevorzugung.setText(wert);
        }
    }

    private void aktualisiereGesamtfaerbedauer() {
        gesamtfaerbedauer.setText(Integer.toString(faerbedauerwert + restfaerbedauerwert) + "=" + faerbedauerwert + "+"
                + restfaerbedauerwert);
    }

    private void aktualisiereGesamtmaldauer() {
        gesamtmaldauer.setText(
                Integer.toString(maldauerwert + restmaldauerwert) + "=" + maldauerwert + "+" + restmaldauerwert);
    }

    private JPanel ausdehnung() {
        final var ausdehnung = new Instrument("Ausdehnung");
        ausdehnung.erstelle("b = ", g, "Ausschnittsbreite");
        ausdehnung.erstelle("h = ", h, "Ausschnittshöhe");
        ausdehnung.erstelle("d = ", pxs, "Länge der elementaren Zellen um einen Punkt");
        ausdehnung.erstelle("* : ", zoom, "Vergrößerungsfaktor");
        return ausdehnung;
    }

    private JPanel ausschnitt() {
        final var ausschnitt = new Instrument("Parameterraum");
        ausschnitt.setToolTipText("""
                <html>\
                {c&thinsp;=&thinsp;a+bi | \
                a<sub>1</sub>&thinsp;&le;&thinsp;a&thinsp;&le;&thinsp;a<sub>2</sub>\
                 , \
                b<sub>1</sub>&thinsp;&le;&thinsp;b&thinsp;&le;&thinsp;b<sub>2</sub>\
                }\
                </html>""");
        ausschnitt.erstelle("<html>a<sub>1</sub>&thinsp;=&thinsp;</html>", a1);
        ausschnitt.erstelle("<html>a<sub>2</sub>&thinsp;=&thinsp;</html>", a2);
        ausschnitt.erstelle("<html>b<sub>1</sub>&thinsp;=&thinsp;</html>", b1);
        ausschnitt.erstelle("<html>b<sub>2</sub>&thinsp;=&thinsp;</html>", b2);
        return ausschnitt;
    }

    private JPanel gitteriteration() {
        final var gitteriteration = new Instrument("Gitter / Iterationen");
        gitteriteration.erstelle("m = ", m, "Horizontale Anzahl der Gitterpunkte");
        gitteriteration.erstelle("n = ", n, "Vertikale Anzahl der Gitterpunkte");
        gitteriteration.erstelle("#!: ", iterationslimit,
                "Mindestgrenze für die erlaubte maximale Anzahl der Iterationsschritte");
        return gitteriteration;
    }

    private JPanel messungen() {
        final var messungen = new Instrument("Zeiten(ms) / Iterationen");
        messungen.erstelle("Berechnung: ", rechendauer);
        messungen.erstelle("Färbung: ", gesamtfaerbedauer, "Während Berechnung plus zusätzlicher Finalisierung");
        messungen.erstelle("Malen: ", gesamtmaldauer, "Während Berechnung plus zusätzlicher Finalisierung");
        messungen.erstelle("# = ", iterationsmaximum,
                "Tatsächlich auftretendes Maximum der Anzahl der Iterationsschritte pro Punkt");
        return messungen;
    }

    private JPanel modi() {
        final var modi = new Instrument("Modi");
        modi.erstelle("A : ", antialiasing, "Antialiasing");
        modi.erstelle("B : ", bevorzugung, "Bevorzugung Geschwindigkeit versus Präzision");
        modi.erstelle("#?: ", qualitaet, "Anzuwendende Methode der Qualitaet");
        return modi;
    }

    private JPanel tafel() {
        final var tafel = new JPanel();
        tafel.setLayout(new FlowLayout(FlowLayout.LEADING));
        tafel.add(modi());
        tafel.add(gitteriteration());
        tafel.add(ausschnitt());
        tafel.add(ausdehnung());
        tafel.add(messungen());
        return tafel;
    }

}
