/******************************************************************************
 ** $Id: Menue.java 3577 2024-09-01 21:39:19Z wmh $
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package ebflmaennle.oberflaeche;

import java.awt.Dimension;
import java.awt.GraphicsEnvironment;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.InputEvent;
import java.awt.event.ItemEvent;
import java.awt.event.KeyEvent;

import javax.swing.Action;
import javax.swing.ButtonGroup;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JFrame;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JRadioButtonMenuItem;
import javax.swing.KeyStroke;

import ebflmaennle.Main.Betrieb;
import ebflmaennle.Main.Definition;
import ebflmaennle.farbe.Modell;
import ebflmaennle.oberflaeche.Delegation.Beantragung;

public class Menue extends JMenuBar {

    private static final long serialVersionUID = 2741905874813086748L;

    private static int modifizierer(final boolean wenig, final int modifizierer) {
        return wenig ? modifizierer : modifizierer | InputEvent.SHIFT_DOWN_MASK;
    }

    private static String staerke(final boolean wenig) {
        return "(" + (wenig ? "wenig" : "mehr") + ")";
    }

    private static String kommando(final boolean wenig, final String kuerzel) {
        return wenig ? kuerzel : kuerzel + kuerzel;
    }

    private static JMenuItem eintrag(final String name, final String kommando, final int taste, final int modifizierer,
            final ActionListener beobachter, final String hilfe) {
        final var eintrag = new JMenuItem(name);
        eintrag.setActionCommand(kommando);
        eintrag.setAccelerator(KeyStroke.getKeyStroke(taste, modifizierer));
        eintrag.addActionListener(beobachter);
        eintrag.setToolTipText(hilfe);
        return eintrag;
    }

    private static JMenuItem eintrag(final String name, final String kommando, final int taste, final int modifizierer,
            final ActionListener beobachter) {
        return eintrag(name, kommando, taste, modifizierer, beobachter, null);
    }

    private static JMenuItem eintrag(final String basis, final String kuerzel, final boolean wenig, final int taste,
            final int modifizierer, final ActionListener beobachter, final String hilfe) {
        final var name = basis + " " + staerke(wenig);
        final var kommando = kommando(wenig, kuerzel);
        return eintrag(name, kommando, taste, modifizierer(wenig, modifizierer), beobachter, hilfe);
    }

    private static JMenuItem eintrag(final String basis, final String kuerzel, final boolean wenig, final int taste,
            final int modifizerer, final ActionListener beobachter) {
        final var name = basis + " " + staerke(wenig);
        final var kommando = kommando(wenig, kuerzel);
        return eintrag(name, kommando, taste, modifizierer(wenig, modifizerer), beobachter);
    }

    private static JMenuItem eintrag(final String basis, final String kuerzel, final boolean wenig, final int taste,
            final ActionListener beobachter) {
        return eintrag(basis, kuerzel, wenig, taste, 0, beobachter);
    }

    private static JMenu menue(final String name, final String hilfe, final JMenuItem... eintraege) {
        final var menu = new JMenu(name);
        menu.setToolTipText(hilfe);
        for (final var eintrag : eintraege) {
            if (eintrag != null) {
                menu.add(eintrag);
            } else {
                menu.addSeparator();
            }
        }
        return menu;
    }

    private static JMenu menue(final String name, final JMenuItem... eintraege) {
        return menue(name, null, eintraege);
    }

    private static JMenu zoom(final ActionListener ausschnittsbeobachter) {
        return menue("Zoom", "Mausrad drehen, Umschalttaste erhöht Wirkung.",
                eintrag("Verkleinern", "/", true, KeyEvent.VK_DIVIDE, ausschnittsbeobachter),
                eintrag("Vergrößern", "*", true, KeyEvent.VK_MULTIPLY, ausschnittsbeobachter), null,
                eintrag("Verkleinern", "/", false, KeyEvent.VK_DIVIDE, ausschnittsbeobachter),
                eintrag("Vergrößern", "*", false, KeyEvent.VK_MULTIPLY, ausschnittsbeobachter));
    }

    private static JMenu verschiebung(final ActionListener ausschnittsbeobachter) {
        return menue("Verschiebung", "Maustaste gedrückt halten und Maus bewegen.",
                eintrag("Links", "<", true, KeyEvent.VK_LEFT, ausschnittsbeobachter),
                eintrag("Rechts", ">", true, KeyEvent.VK_RIGHT, ausschnittsbeobachter),
                eintrag("Runter", "R", true, KeyEvent.VK_DOWN, ausschnittsbeobachter),
                eintrag("Hoch", "H", true, KeyEvent.VK_UP, ausschnittsbeobachter), null,
                eintrag("Links", "<", false, KeyEvent.VK_LEFT, ausschnittsbeobachter),
                eintrag("Rechts", ">", false, KeyEvent.VK_RIGHT, ausschnittsbeobachter),
                eintrag("Runter", "R", false, KeyEvent.VK_DOWN, ausschnittsbeobachter),
                eintrag("Hoch", "H", false, KeyEvent.VK_UP, ausschnittsbeobachter));
    }

    private static JMenu fokussierung(final ActionListener ausschnittsbeobachter) {
        return menue("Fokussierung",
                eintrag("Zentrierung", "O", KeyEvent.VK_NUMPAD5, 0, ausschnittsbeobachter,
                        "Mausklick zentriert anvisierten Punkt."),
                null,
                eintrag("Zentrierte Vergrößerung", "+", true, KeyEvent.VK_MULTIPLY, InputEvent.CTRL_DOWN_MASK,
                        ausschnittsbeobachter,
                        "Mausklick bei gedrückter Steuerungstaste zentriert anvisierten Punkt und vergrößert."),
                eintrag("Zentrierte Verkleinerung", "-", true, KeyEvent.VK_DIVIDE, InputEvent.CTRL_DOWN_MASK,
                        ausschnittsbeobachter,
                        "Mausklick bei gedrückter Alternativtaste zentriert anvisierten Punkt und verkleinert."),
                null,
                eintrag("Zentrierte Vergrößerung", "+", false, KeyEvent.VK_MULTIPLY, InputEvent.CTRL_DOWN_MASK,
                        ausschnittsbeobachter,
                        "Mausklick bei gedrückter Steuerungs- und Umschalttaste zentriert anvisierten Punkt und vergrößert mehr."),
                eintrag("Zentrierte Verkleinerung", "-", false, KeyEvent.VK_DIVIDE, InputEvent.CTRL_DOWN_MASK,
                        ausschnittsbeobachter,
                        "Mausklick bei gedrückter Alternativ- und Umschalttaste zentriert anvisierten Punkt und verkleinert mehr."));
    }

    private static JMenu ausschnitt(final ActionListener ausschnittsbeobachter) {
        return menue("Ausschnitt", zoom(ausschnittsbeobachter), verschiebung(ausschnittsbeobachter),
                fokussierung(ausschnittsbeobachter));
    }

    private static JMenu datei(final Action eingabeaktion, final Action exportaktion, final Action importaktion,
            final Action fotoaktion, final Action ende) {
        return menue("Datei", new JMenuItem(eingabeaktion), new JMenuItem(exportaktion), new JMenuItem(importaktion),
                null, new JMenuItem(fotoaktion), null, new JMenuItem(ende));
    }

    private JCheckBoxMenuItem antialiasing;
    private JCheckBoxMenuItem vollbild;
    private JCheckBoxMenuItem vorschau;
    private JCheckBoxMenuItem prinzipien;
    private JCheckBoxMenuItem innenblockschaetzung;
    private JCheckBoxMenuItem aussenblockschaetzung;
    private JCheckBoxMenuItem zentrumdetektion;
    private JCheckBoxMenuItem innendetektion;
    private JCheckBoxMenuItem distanzschaetzung;
    private JCheckBoxMenuItem multiplikatoren;
    private JCheckBoxMenuItem abstandschaetzung;
    private JCheckBoxMenuItem schrittschaetzung;
    private JCheckBoxMenuItem bildfrequzenz;
    private JCheckBoxMenuItem scheibe;
    private JCheckBoxMenuItem andreaskreuz;
    private JCheckBoxMenuItem koerperkopferkennung;
    private final JMenuItem qualitaetsanpassung = new JMenuItem("Anpassung");

    private JRadioButtonMenuItem geschwindigkeit;
    private JRadioButtonMenuItem standard;
    private JRadioButtonMenuItem praezision;
    private JRadioButtonMenuItem qualitaetswahl;
    private JRadioButtonMenuItem qualitaetsautomatik;
    private JRadioButtonMenuItem qualitaetsoptimum;

    private boolean istVollbildMoeglich;

    public Menue(final Betrieb betrieb, final Beantragung beantragung, final Modell farbmodellbeobachter,
            final ActionListener gestaltung, final Delegation delegation, final ActionListener ausschnittsbeobachter,
            final Aktionen aktionen) {
        errichteVollbild(betrieb);
        erstelleEintraege(betrieb, aktionen, delegation, farbmodellbeobachter, gestaltung, ausschnittsbeobachter);
        add(datei(aktionen.get("Eingabe"), aktionen.get("Export"), aktionen.get("Import"), aktionen.get("Foto"),
                aktionen.get("Ende")));
        add(darstellung(farbmodellbeobachter, aktionen.get("Farbgestaltung"), aktionen.get("Tafel"),
                aktionen.get("Pipette"), aktionen.get("Zentrum")));
        add(methode());
        add(qualitaet(delegation, aktionen.get("Abbruch")));
        add(assistenz(beantragung, delegation, farbmodellbeobachter, gestaltung));
        add(ausschnitt(ausschnittsbeobachter));
        add(hilfe(aktionen.get("Info")));
    }

    @Override
    public Dimension getPreferredSize() {
        final var dim = super.getPreferredSize();
        if (istVollbildMoeglich) {
            dim.height = 0;
        }
        return dim;
    }

    private void erstelleEintraege(final Betrieb betrieb, final Aktionen aktionen, final Delegation delegation,
            final Modell farbmodellbeobachter, final ActionListener gestaltung,
            final ActionListener ausschnittsbeobachter) {
        if (betrieb == Betrieb.VOLLSTAENDIG) {
            antialiasing = new JCheckBoxMenuItem(delegation.antialiasingsaktion());
        } else {
            antialiasing = new JCheckBoxMenuItem("Antialiasing");
            antialiasing.setToolTipText("Aufgrund von Speicherknappheit außer Betrieb.");
            antialiasing.setEnabled(false);
        }
        vorschau = new JCheckBoxMenuItem(delegation.vorschauaktion());
        scheibe = new JCheckBoxMenuItem(aktionen.get("Scheibe"));
        andreaskreuz = new JCheckBoxMenuItem(aktionen.get("Andreaskreuz"));
        bildfrequzenz = new JCheckBoxMenuItem(delegation.bildfrequenzaktion());
        prinzipien = new JCheckBoxMenuItem(aktionen.get("Prinzipien"));
        innenblockschaetzung = new JCheckBoxMenuItem(aktionen.get("Innenblockschätzung"));
        aussenblockschaetzung = new JCheckBoxMenuItem(delegation.aussenblockschaetzungsaktion());
        zentrumdetektion = new JCheckBoxMenuItem(aktionen.get("Zentrumdetektion"));
        innendetektion = new JCheckBoxMenuItem(delegation.innendetektionsaktion());
        distanzschaetzung = new JCheckBoxMenuItem(delegation.distanzschaetzungsaktion());
        multiplikatoren = new JCheckBoxMenuItem(aktionen.get("Multiplikatoren"));
        abstandschaetzung = new JCheckBoxMenuItem(aktionen.get("Abstandschätzung"));
        schrittschaetzung = new JCheckBoxMenuItem(delegation.schrittschaetzungsaktion());
        koerperkopferkennung = new JCheckBoxMenuItem(delegation.koerperkopferkennungsaktion());
        final var definitionsimportaktion = aktionen.get("Import");
        definitionsimportaktion.addPropertyChangeListener(ereignis -> {
            uebernehme((Definition) ereignis.getNewValue());
        });
    }

    private void errichteVollbild(final Betrieb betrieb) {
        if (betrieb != Betrieb.MINIMAL) {
            vollbild = new JCheckBoxMenuItem(new Aktion("Vollbild", KeyEvent.VK_F11, false) {
                private static final long serialVersionUID = -5696881638145123537L;

                @Override
                public void actionPerformed(final ActionEvent event) {
                    final var device = GraphicsEnvironment.getLocalGraphicsEnvironment().getDefaultScreenDevice();
                    final var normal = device.getFullScreenWindow() == null;
                    final var rahmen = (JFrame) getTopLevelAncestor();
                    istVollbildMoeglich = device.isFullScreenSupported() && normal;
                    rahmen.dispose();
                    rahmen.setUndecorated(istVollbildMoeglich);
                    rahmen.setResizable(!istVollbildMoeglich);
                    if (istVollbildMoeglich) {
                        device.setFullScreenWindow(rahmen);
                    } else {
                        device.setFullScreenWindow(null);
                        rahmen.setVisible(true);
                    }
                }
            });
        } else {
            vollbild = new JCheckBoxMenuItem("Vollbild");
            vollbild.setToolTipText("Aufgrund von Speicherknappheit außer Betrieb.");
            vollbild.setEnabled(false);
        }
    }

    private JMenu darstellung(final Modell farbmodellbeobachter, final Action farbgestaltungsaktion,
            final Action tafelaktion, final Action pipettenaktion, final Action zentrumaktion) {
        final var darstellung = new JMenu("Darstellung");
        darstellung.add(new JCheckBoxMenuItem(farbgestaltungsaktion));
        darstellung.addSeparator();
        darstellung.add(vollbild);
        darstellung.add(antialiasing);
        darstellung.add(vorschau);
        darstellung.addSeparator();
        darstellung.add(new JCheckBoxMenuItem(tafelaktion));
        darstellung.add(new JCheckBoxMenuItem(pipettenaktion));
        darstellung.add(scheibe);
        darstellung.add(new JCheckBoxMenuItem(zentrumaktion));
        darstellung.add(andreaskreuz);
        darstellung.addSeparator();
        prinzipien.addActionListener(farbmodellbeobachter.prinzipienwaechter());
        darstellung.add(prinzipien);
        darstellung.add(bildfrequzenz);
        return darstellung;
    }

    private JMenu methode() {
        final var methode = new JMenu("Methode");
        methode.add(zentrumdetektion);
        methode.add(innendetektion);
        methode.add(distanzschaetzung);
        methode.add(multiplikatoren);
        methode.add(abstandschaetzung);
        methode.add(schrittschaetzung);
        methode.addSeparator();
        methode.add(innenblockschaetzung);
        methode.add(aussenblockschaetzung);
        methode.addSeparator();
        methode.add(koerperkopferkennung);
        return methode;
    }

    private JMenu qualitaet(final Delegation delegation, final Action abbruchaktion) {
        qualitaetsautomatik = new JRadioButtonMenuItem(delegation.qualitaetsautomatikaktion());
        qualitaetsanpassung.addActionListener(delegation.qualitaetsbeobachter());
        qualitaetsanpassung.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, 0));
        qualitaetswahl = new JRadioButtonMenuItem(delegation.qualitaetswahlaktion());
        qualitaetsoptimum = new JRadioButtonMenuItem(delegation.qualitaetsoptimumkaktion());
        final var qualitaetsgruppe = new ButtonGroup();
        qualitaetsgruppe.add(qualitaetsautomatik);
        qualitaetsgruppe.add(qualitaetswahl);
        qualitaetsgruppe.add(qualitaetsoptimum);
        final var groeber = new JMenuItem("Reduktion");
        groeber.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_SUBTRACT, 0));
        groeber.addActionListener(delegation.qualitaetsbeobachter());
        final var feiner = new JMenuItem("Erhöhung");
        feiner.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_ADD, 0));
        feiner.addActionListener(delegation.qualitaetsbeobachter());
        final var abbruch = new JMenuItem(abbruchaktion);
        final var qualitaet = new JMenu("Qualität");
        qualitaet.add(qualitaetsautomatik);
        qualitaet.add(qualitaetswahl);
        qualitaet.add(qualitaetsoptimum);
        qualitaet.addSeparator();
        qualitaet.add(groeber);
        qualitaet.add(feiner);
        qualitaet.addSeparator();
        qualitaet.add(qualitaetsanpassung);
        qualitaet.add(abbruch);
        qualitaetsoptimum.addItemListener(ereignis -> {
            final var aktiviert = ereignis.getStateChange() != ItemEvent.SELECTED;
            groeber.setEnabled(aktiviert);
            feiner.setEnabled(aktiviert);
            qualitaetsanpassung.setEnabled(aktiviert);
        });
        return qualitaet;
    }

    private JMenu bevorzugung(final Action geschwindigkeitsaktion, final ActionListener qualitaetsbeobachter) {
        geschwindigkeit = new JRadioButtonMenuItem(geschwindigkeitsaktion);
        standard = new JRadioButtonMenuItem("Standard");
        standard.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_F3, 0));
        standard.addActionListener(qualitaetsbeobachter);
        standard.setSelected(true);
        praezision = new JRadioButtonMenuItem(new Aktion("Präzision", KeyEvent.VK_F4, false) {

            private static final long serialVersionUID = 8417934270646044636L;

            @Override
            public void actionPerformed(final ActionEvent event) {
            }

        });
        praezision.setEnabled(false);
        final var gruppe = new ButtonGroup();
        gruppe.add(geschwindigkeit);
        gruppe.add(standard);
        gruppe.add(praezision);
        final var bevorzugung = new JMenu("Bevorzugung");
        bevorzugung.add(geschwindigkeit);
        bevorzugung.add(standard);
        bevorzugung.add(praezision);
        return bevorzugung;
    }

    private JMenu parametrisierung(final Beantragung beantragung, final ActionListener farbmodellbeobachter,
            final ActionListener gestaltung) {
        final var retro = new JMenuItem(new Aktion("Retro", KeyEvent.VK_F5, false) {
            private static final long serialVersionUID = -7616497738343702859L;

            @Override
            public void actionPerformed(final ActionEvent event) {
                antialiasing.setSelected(false);
                vorschau.setSelected(true);
                zentrumdetektion.setSelected(true);
                innendetektion.setSelected(false);
                distanzschaetzung.setSelected(false);
                multiplikatoren.setSelected(true);
                abstandschaetzung.setSelected(false);
                schrittschaetzung.setSelected(false);
                innenblockschaetzung.setSelected(true);
                aussenblockschaetzung.setSelected(true);
                koerperkopferkennung.setSelected(true);
                qualitaetsautomatik.setSelected(true);
                beantragung.beantrage();
            }
        });
        retro.addActionListener(farbmodellbeobachter);
        retro.addActionListener(gestaltung);
        final var schnelligkeit = new JMenuItem(new Aktion("Schnelligkeit", KeyEvent.VK_F6, false) {
            private static final long serialVersionUID = -5353897971183191979L;

            @Override
            public void actionPerformed(final ActionEvent event) {
                antialiasing.setSelected(false);
                vorschau.setSelected(false);
                zentrumdetektion.setSelected(true);
                innendetektion.setSelected(false);
                distanzschaetzung.setSelected(true);
                multiplikatoren.setSelected(true);
                abstandschaetzung.setSelected(true);
                schrittschaetzung.setSelected(true);
                innenblockschaetzung.setSelected(true);
                aussenblockschaetzung.setSelected(true);
                koerperkopferkennung.setSelected(true);
                qualitaetsautomatik.setSelected(true);
                beantragung.beantrage();
            }
        });
        schnelligkeit.addActionListener(farbmodellbeobachter);
        schnelligkeit.addActionListener(gestaltung);
        final var kompromiss = new JMenuItem(new Aktion("Kompromiss", KeyEvent.VK_F7, true) {
            private static final long serialVersionUID = 4402918868613075822L;

            @Override
            public void actionPerformed(final ActionEvent event) {
                antialiasing.setSelected(false);
                vorschau.setSelected(false);
                zentrumdetektion.setSelected(true);
                innendetektion.setSelected(false);
                distanzschaetzung.setSelected(true);
                multiplikatoren.setSelected(true);
                abstandschaetzung.setSelected(true);
                schrittschaetzung.setSelected(true);
                innenblockschaetzung.setSelected(true);
                schrittschaetzung.setSelected(true);
                aussenblockschaetzung.setSelected(false);
                koerperkopferkennung.setSelected(true);
                qualitaetsautomatik.setSelected(true);
                beantragung.beantrage();
            }
        });
        kompromiss.addActionListener(farbmodellbeobachter);
        kompromiss.addActionListener(gestaltung);
        final var qualitaet = new JMenuItem(new Aktion("Qualität", KeyEvent.VK_F8, false) {
            private static final long serialVersionUID = 7708542545480085728L;

            @Override
            public void actionPerformed(final ActionEvent event) {
                antialiasing.setSelected(true);
                vorschau.setSelected(false);
                zentrumdetektion.setSelected(false);
                innendetektion.setSelected(true);
                distanzschaetzung.setSelected(true);
                multiplikatoren.setSelected(true);
                abstandschaetzung.setSelected(true);
                schrittschaetzung.setSelected(false);
                innenblockschaetzung.setSelected(true);
                aussenblockschaetzung.setSelected(false);
                koerperkopferkennung.setSelected(true);
                qualitaetsoptimum.setSelected(true);
                beantragung.beantrage();
            }
        });
        qualitaet.addActionListener(farbmodellbeobachter);
        qualitaet.addActionListener(gestaltung);

        final var parametrisierung = new JMenu("Parametrisierung");
        parametrisierung.add(retro);
        parametrisierung.add(schnelligkeit);
        parametrisierung.add(kompromiss);
        parametrisierung.add(qualitaet);

        return parametrisierung;
    }

    private JMenu assistenz(final Beantragung beantragung, final Delegation delegation,
            final ActionListener farbmodellbeobachter, final ActionListener gestaltung) {
        final var assistenz = new JMenu("Assistenz");
        assistenz.add(bevorzugung(delegation.geschwindigkeitsaktion(), delegation.qualitaetsbeobachter()));
        assistenz.add(parametrisierung(beantragung, farbmodellbeobachter, gestaltung));
        return assistenz;
    }

    private JMenu hilfe(final Action infoaktion) {
        final var hilfe = new JMenu("Hilfe");
        final var ueber = new JMenuItem(infoaktion);
        hilfe.add(ueber);
        return hilfe;
    }

    private void uebernehme(final Definition definition) {
        final var option = definition.option();
        final var ausschnitt = definition.ausschnitt();
        aussenblockschaetzung.setSelected(option.aussenblockschaetzung);
        innenblockschaetzung.setSelected(option.innenblockschaetzung);
        multiplikatoren.setSelected(option.multiplikatoren);
        zentrumdetektion.setSelected(option.zentrumdetektion);
        innendetektion.setSelected(option.innendetektion);
        abstandschaetzung.setSelected(option.abstandschaetzung);
        schrittschaetzung.setSelected(option.schrittschaetzung);
        distanzschaetzung.setSelected(option.distanzschaetzung);
        koerperkopferkennung.setSelected(option.koerperkopferkennung);
        bildfrequzenz.setSelected(option.bildfrequenz);
        prinzipien.setSelected(option.prinzipien);
        switch (option.qualitaet) {
            case AUTOMATIK:
                qualitaetsautomatik.setSelected(true);
                break;
            case BENUTZER:
                qualitaetswahl.setSelected(true);
                break;
            case OPTIMUM:
                qualitaetsoptimum.setSelected(true);
                break;
        }
        switch (option.bevorzugung) {
            case ZEIT:
                geschwindigkeit.setSelected(true);
                break;
            case STANDARD:
                standard.setSelected(true);
                break;
            case PRAEZISION:
                praezision.setSelected(true);
                break;
        }
        antialiasing.setSelected(ausschnitt.antialiasing() == 2);
        vorschau.setSelected(ausschnitt.vorschau() == 2);
    }

}
