/******************************************************************************
 ** $Id: Leinwand.java 3547 2024-08-14 22:51:11Z wmh $
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package ebflmaennle.oberflaeche;

import java.awt.AlphaComposite;
import java.awt.BasicStroke;
import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.RenderingHints;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseWheelEvent;
import java.awt.geom.Ellipse2D;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.awt.image.BufferedImage;
import java.awt.image.DataBufferInt;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.lang.management.ManagementFactory;
import java.lang.management.ThreadMXBean;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

import javax.swing.Action;
import javax.swing.BorderFactory;
import javax.swing.JFrame;
import javax.swing.JPanel;
import javax.swing.SwingUtilities;
import javax.swing.Timer;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import ebflmaennle.Main;
import ebflmaennle.Main.Definition;
import ebflmaennle.berechnung.Algorithmus.Block;
import ebflmaennle.berechnung.Ausschnitt;
import ebflmaennle.berechnung.Charakteristik;
import ebflmaennle.berechnung.Kalkulator;
import ebflmaennle.farbe.Modell;
import ebflmaennle.farbe.Parameter;

public class Leinwand extends JPanel implements ChangeListener, PropertyChangeListener, Delegation.Position {

    private static final long serialVersionUID = -8601507407213791842L;
    private static final int ANDREASKREUZRADIUS = 9;

    public record Zeiger(Point position, Point delta, int radstufe) {
        @Override
        public boolean equals(final Object objekt) {
            if (this == objekt) {
                return true;
            }
            if ((objekt == null) || !(objekt instanceof final Zeiger zeiger)
                    || ((position == null) == (zeiger.position != null)) || ((delta == null) == (zeiger.delta != null))
                    || radstufe != 0 || zeiger.radstufe != 0) {
                return false;
            }
            return (position == null || position.equals(zeiger.position))
                    && (delta == null || delta.equals(zeiger.delta));
        }
    }

    /**
     * @param bild
     *            Bilddatenhaltung
     * @param grafik
     *            Grafikumgebung zur Darstellung
     */
    private record Blatt(BufferedImage bild, Graphics2D grafik) {
    }

    private class Parallelisierer implements Runnable {
        private final boolean aktualisierung;

        Parallelisierer(final boolean aktualisierung) {
            this.aktualisierung = aktualisierung;
        }

        @Override
        public void run() {
            while (!warteschlange.isEmpty()) {
                final var block = warteschlange.poll();
                if (block != null) {
                    modell.faerbe(charakteristiken, farben(), pixeldistanzquadrat, block.ecke(), bildgroesse.width,
                            block.dk(), block.dl());
                    if (aktualisierung) {
                        ausmale(new Rectangle(block.ecke() % bildgroesse.width, block.ecke() / bildgroesse.width,
                                block.dk() + 1, block.dl() + 1));
                    }
                }
            }
        }
    }

    private static final float ALPHA = 0.5f;
    private static final int ANIMATIONSRESERVENDIVISOR = 8;
    private static final int AUFLAUFDAUER = 1000; // ms
    private static final int INTERAKTIVZEIT = 125; // ms
    private static final Point NULLPUNKT = new Point(0, 0);
    private static final ThreadMXBean MESSUNG = ManagementFactory.getThreadMXBean();

    public static long speicherdatenbedarf(final int maximum) {
        final long shaubtbild = Integer.BYTES * maximum;
        final long szweitbild = Integer.BYTES * maximum;
        final long sanzeige = Integer.BYTES * maximum / 4;
        return shaubtbild + szweitbild + sanzeige;
    }

    private final Charakteristik[] charakteristiken;
    private final ExecutorService faerber = Executors.newFixedThreadPool(Main.KERNANZAHL);
    private final Modell modell;
    private final MouseAdapter mausbeobachter = new MouseAdapter() {

        @Override
        public void mouseDragged(final MouseEvent ereignis) {
            if (!ziehend) {
                ziehend = true;
                aendereZeigerbild();
            }
        }

        @Override
        public void mouseReleased(final MouseEvent ereignis) {
            ziehend = false;
            aendereZeigerbild();
        }

        @Override
        public void mouseWheelMoved(final MouseWheelEvent ereignis) {
            if (!zoomend) {
                zoomend = true;
                aendereZeigerbild();
            }
        }

    };

    private final Timer auflauftimer = new Timer(AUFLAUFDAUER, ereignis -> faerbe(true));
    private final Timer interaktivtimer = new Timer(INTERAKTIVZEIT, ereignis -> {
        faerbe(false);
        repaint();
    });
    private final ConcurrentLinkedQueue<Block> warteschlange = new ConcurrentLinkedQueue<>();
    private int antialiasing = 1;
    private int vorschau = 1;
    /** Pixel pro Punkt. */
    private float vorgaengerppp;
    /** Pixel pro Punkt. */
    private float nachfolgerppp = 1;
    private float zelllaengenquadrat;
    private float pixeldistanzquadrat;
    private Object antialiasinghinweis = RenderingHints.VALUE_ANTIALIAS_OFF;
    private Object interpolationshinweis = RenderingHints.VALUE_INTERPOLATION_BILINEAR;
    private long vorgaengermaldauer = -1;
    private long nachfolgermaldauer = 0;
    private long vorgaengerrestmaldauer = -1;
    private long nachfolgerrestmaldauer = 0;
    private long vorgaengerfaerbedauer = -1;
    private long nachfolgerfaerbedauer = 0;
    private long vorgaengerrestfaerbedauer = -1;
    private long nachfolgerrestfaerbedauer = 0;
    private boolean istScheibenanzeigeGewuenscht;
    private boolean istScheibenanzeigeVerfuegbar;
    private final Point scheibenzeiger = new Point();
    private double scheibenradius;
    private Ellipse2D scheibe;
    private boolean istAndreaskreuzanzeigeGewuenscht;
    private final Point andreaskreuzzeiger = new Point();
    private Rectangle2D andreaskreuz;
    private boolean istAndreaskreuzVerfuegbar;
    /** Das gerade in Arbeit befindliche Blatt, das sich in Nachfolge befindet und angezeigt wird. */
    private Blatt arbeitsblatt = new Blatt(null, null);
    /** Schattenblatt zum alten Arbeitsblatt, zur Vorbereitung in das transformierte neue Arbeitsblatt einfügt. */
    private Blatt schattenblatt;
    private Dimension bildgroesse = new Dimension(0, 0);
    private double vergroesserung = 1;
    private final Point fixpunkt = new Point(0, 0);
    private Point verschiebung = new Point(0, 0);
    private boolean ziehend;
    private boolean zoomend;
    private boolean berechnend;
    private boolean schwebend;

    public Leinwand(final Dimension standardgroesse, final Dimension maximalgroesse, final Modell modell,
            final Charakteristik[] charakteristiken) {
        this.modell = modell;
        this.charakteristiken = charakteristiken;
        setLayout(new BorderLayout());
        setPreferredSize(standardgroesse);
        if (maximalgroesse != null) {
            setMaximumSize(maximalgroesse);
        }
        setBackground(Color.GRAY);
        setBorder(BorderFactory.createEmptyBorder());
        addMouseListener(mausbeobachter);
        addMouseMotionListener(mausbeobachter);
        addMouseWheelListener(mausbeobachter);
    }

    @Override
    public void paintComponent(final Graphics g) {
        super.paintComponent(g);
        final var start = MESSUNG.getCurrentThreadCpuTime();
        final var g2 = (Graphics2D) g;
        g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING, antialiasinghinweis);
        g2.setRenderingHint(RenderingHints.KEY_INTERPOLATION, interpolationshinweis);
        g2.scale(nachfolgerppp, nachfolgerppp);
        g2.translate((1 - vergroesserung) * fixpunkt.x, (1 - vergroesserung) * fixpunkt.y);
        g2.scale(vergroesserung, vergroesserung);
        g2.translate(verschiebung.x, verschiebung.y);
        g2.drawImage(arbeitsblatt.bild, 0, 0, null);
        if (!berechnend && schwebend) {
            if (istAndreaskreuzanzeigeGewuenscht && istAndreaskreuzVerfuegbar) {
                zeichneAndreaskreuz(g2, (int) andreaskreuz.getX(), (int) andreaskreuz.getY(),
                        (int) andreaskreuz.getWidth());
            }
            if (istScheibenanzeigeGewuenscht && istScheibenanzeigeVerfuegbar) {
                g2.setComposite(AlphaComposite.getInstance(AlphaComposite.SRC_OVER, ALPHA));
                g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
                zeichneScheibe(g2, (int) scheibe.getX(), (int) scheibe.getY(), (int) scheibe.getWidth());
            }
        }
        final var dauer = MESSUNG.getCurrentThreadCpuTime() - start;
        if (berechnend) {
            nachfolgermaldauer += dauer;
        } else {
            nachfolgerrestmaldauer = dauer;
        }
        interaktivtimer.stop();
    }

    @Override

    public void propertyChange(final PropertyChangeEvent ereignis) {
        assert SwingUtilities.isEventDispatchThread();
        switch (ereignis.getPropertyName()) {
            case "Bildgröße":
                behandleBildgroesse(ereignis);
                break;
            case "Antialiasing":
            case "Vorschau":
                behandleAntialiasingVorschau(ereignis);
                break;
            case "Auflösung":
                behandleAufloesung(ereignis);
                break;
            case "Stopp":
                behandleStopp(ereignis);
                break;
            case "Berechnung":
                behandleBerechnung(ereignis);
                break;
            case Action.SELECTED_KEY:
                behandleSelectedKey(ereignis);
                break;
            case "Zeiger":
                behandleZeiger(ereignis);
                break;
            case "Zentrum":
                behandleZentrum(ereignis);
                break;
            case "Import":
                behandleImport(ereignis);
                break;
        }
    }

    @Override
    public void stateChanged(final ChangeEvent ereignis) {
        if (ereignis.getSource().getClass() == Kalkulator.class) {
            zoomend = false;
            faerbe(false);
            modell.faerbe(charakteristiken, farben(), pixeldistanzquadrat, bildgroesse.width, bildgroesse.height);
            setzeScheibenradius();
            versetzeScheibe();
            if (antialiasing > 1) {
                bevorzugeBildqualitaet(true);
                repaint();
            } else {
                ausmale(new Rectangle(bildgroesse.width - 1, 0, 1, bildgroesse.height));
                ausmale(new Rectangle(0, bildgroesse.height - 1, bildgroesse.width - 1, 1));
            }
        } else if (ereignis.getSource().getClass() == Parameter.class) {
            if (!berechnend) {
                nachfolgerfaerbedauer = 0;
                nachfolgerrestfaerbedauer = modell.faerbe(charakteristiken, farben(), pixeldistanzquadrat);
                repaint();
            }
        } else {
            throw new RuntimeException(ereignis.getSource().toString());
        }
        SwingUtilities.invokeLater(() -> {
            firePropertyChange("Färbedauer", vorgaengerfaerbedauer, nachfolgerfaerbedauer);
            vorgaengerfaerbedauer = nachfolgerfaerbedauer;
            firePropertyChange("Restfärbedauer", vorgaengerrestfaerbedauer, nachfolgerrestfaerbedauer);
            vorgaengerrestfaerbedauer = nachfolgerrestfaerbedauer;
            firePropertyChange("Maldauer", vorgaengermaldauer, nachfolgermaldauer);
            vorgaengermaldauer = nachfolgermaldauer;
            firePropertyChange("Restmaldauer", vorgaengerrestmaldauer, nachfolgerrestmaldauer);
            vorgaengerrestmaldauer = nachfolgerrestmaldauer;
        });
    }

    public BufferedImage bild() {
        return arbeitsblatt.bild;
    }

    public void vormerke(final Block block) {
        warteschlange.add(block);
    }

    @Override
    public Point2D.Double fokus() {
        final var position = getMousePosition();
        if (position == null) {
            return null;
        }
        return new Point2D.Double(position.getX() / getWidth(), position.getY() / getHeight());
    }

    @Override
    public void beachte(final Zeiger zeiger) {
        verfolge(zeiger);
    }

    private void behandleBildgroesse(final PropertyChangeEvent ereignis) {
        schwebend = false;
        skaliereScheibe();
        versetzeScheibe();
        skaliereAndreaskreuz();
        versetzeAndreaskeuz();
        erzeugeBlaetter(ereignis);
        repaint();
    }

    private void behandleAntialiasingVorschau(final PropertyChangeEvent ereignis) {
        final var wert = (int) ereignis.getNewValue();
        if ("Antialiasing".equals(ereignis.getPropertyName())) {
            antialiasing = wert;
        } else {
            vorschau = wert;
        }
        nachfolgerppp = (float) vorschau / (float) antialiasing;
    }

    private void behandleAufloesung(final PropertyChangeEvent ereignis) {
        final var zelllaenge = (float) ereignis.getNewValue();
        zelllaengenquadrat = zelllaenge * zelllaenge;
        pixeldistanzquadrat = zelllaengenquadrat / 2;
    }

    private void behandleStopp(final PropertyChangeEvent ereignis) {
        if ((boolean) ereignis.getNewValue()) {
            bevorzugeBildqualitaet(false);
        }
    }

    private void behandleBerechnung(final PropertyChangeEvent ereignis) {
        berechnend = (boolean) ereignis.getNewValue();
        aendereZeigerbild();
        verfolgeBerechnung();
    }

    private void behandleSelectedKey(final PropertyChangeEvent ereignis) {
        final var kommando = ((Action) ereignis.getSource()).getValue(Action.ACTION_COMMAND_KEY);
        final var selektiert = (boolean) ereignis.getNewValue();
        if ("Scheibe".equals(kommando)) {
            istScheibenanzeigeGewuenscht = selektiert;
            setzeScheibenradius();
            versetzeScheibe();
            repaint();
        } else if ("Andreaskreuz".equals(kommando)) {
            istAndreaskreuzanzeigeGewuenscht = selektiert;
            versetzeAndreaskeuz();
            repaint();
        }
    }

    private void behandleZeiger(final PropertyChangeEvent ereignis) {
        verfolge((Zeiger) ereignis.getNewValue());
    }

    private void behandleZentrum(final PropertyChangeEvent ereignis) {
        verschiebeZentrum(ereignis);
        versetzeAndreaskeuz();
    }

    private void behandleImport(final PropertyChangeEvent ereignis) {
        final var definition = (Definition) ereignis.getNewValue();
        setPreferredSize(definition.ausschnitt().leinwandgroesse());
        arbeitsblatt.grafik.clearRect(0, 0, arbeitsblatt.bild.getWidth(), arbeitsblatt.bild.getHeight());
        final var fenster = (JFrame) SwingUtilities.getWindowAncestor(this);
        fenster.pack();
    }

    private void ausmale(final Rectangle realrahmen) {
        if (realrahmen == null) {
            repaint();
        } else {
            repaint(darstellungsrahmen(realrahmen));
        }
    }

    private void ausmale(final Rectangle vorgaengerrealrahmen, final Rectangle nachfolgerrealrahmen) {
        if (schwebend) {
            ausmale(vorgaengerrealrahmen);
            nachfolgerrealrahmen.grow(nachfolgerrealrahmen.width / ANIMATIONSRESERVENDIVISOR,
                    nachfolgerrealrahmen.height / ANIMATIONSRESERVENDIVISOR);
            ausmale(nachfolgerrealrahmen);
        }
    }

    private void aendereZeigerbild() {
        final Cursor cursor;
        if (ziehend) {
            cursor = Cursor.getPredefinedCursor(Cursor.MOVE_CURSOR);
        } else if (zoomend) {
            cursor = Cursor.getPredefinedCursor(Cursor.HAND_CURSOR);
        } else if (berechnend) {
            cursor = Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR);
        } else {
            cursor = Cursor.getPredefinedCursor(Cursor.CROSSHAIR_CURSOR);
        }
        setCursor(cursor);
    }

    private long bearbeiteFaerbewarteschlange(final boolean aktualisierung) {
        assert SwingUtilities.isEventDispatchThread();
        final var start = System.nanoTime();
        final List<Parallelisierer> parallelisiererliste = new ArrayList<>(Main.KERNANZAHL);
        for (var i = 0; i < Main.KERNANZAHL; i++) {
            parallelisiererliste.add(new Parallelisierer(aktualisierung));
        }
        final List<Future<?>> resultate = new ArrayList<>(parallelisiererliste.size());
        for (final Parallelisierer parallelisierer : parallelisiererliste) {
            resultate.add(faerber.submit(parallelisierer));
        }
        for (final Future<?> resultat : resultate) {
            try {
                resultat.get();
            } catch (InterruptedException | ExecutionException exception) {
                exception.printStackTrace();
            }
        }
        return System.nanoTime() - start;
    }

    private void bevorzugeBildqualitaet(final boolean qualitaet) {
        antialiasinghinweis = qualitaet ? RenderingHints.VALUE_ANTIALIAS_ON : RenderingHints.VALUE_ANTIALIAS_OFF;
        interpolationshinweis = qualitaet ? RenderingHints.VALUE_INTERPOLATION_BILINEAR
                : RenderingHints.VALUE_INTERPOLATION_NEAREST_NEIGHBOR;
    }

    private Rectangle darstellungsrahmen(final Rectangle realrahmen) {
        final var x = realrahmen.x * vorschau / antialiasing - 1;
        final var y = realrahmen.y * vorschau / antialiasing - 1;
        final var w = ((realrahmen.width + antialiasing - 1) * vorschau) / antialiasing + 2;
        final var h = ((realrahmen.height + antialiasing - 1) * vorschau) / antialiasing + 2;
        return new Rectangle(x, y, w, h);
    }

    private int realgroesse(final int darstellungsgroesse) {
        return darstellungsgroesse * antialiasing / vorschau;
    }

    private Blatt erzeugeBlatt(final int breite, final int hoehe, Graphics2D grafik) {
        final var bild = new BufferedImage(breite, hoehe, BufferedImage.TYPE_INT_RGB);
        bild.setAccelerationPriority(1);
        if (grafik != null) {
            grafik.dispose();
        }
        grafik = bild.createGraphics();
        grafik.setBackground(Color.GRAY);
        return new Blatt(bild, grafik);
    }

    private void erzeugeBlaetter(final PropertyChangeEvent ereignis) {
        final var vorgaengerbreite = bildgroesse.width;
        final var vorgaengerhoehe = bildgroesse.height;
        bildgroesse = (Dimension) ereignis.getNewValue();
        final var nachfolgerbreite = bildgroesse.width;
        final var nachfolgerhoehe = bildgroesse.height;
        schattenblatt = arbeitsblatt;
        arbeitsblatt = erzeugeBlatt(nachfolgerbreite, nachfolgerhoehe, arbeitsblatt.grafik);
        arbeitsblatt.grafik.clearRect(0, 0, nachfolgerbreite, nachfolgerhoehe);
        final var transformation = arbeitsblatt.grafik.getTransform();
        final double skalierung = vorgaengerppp / nachfolgerppp;
        assert vorgaengerbreite != nachfolgerbreite || vorgaengerhoehe != nachfolgerhoehe;
        arbeitsblatt.grafik.scale(skalierung, skalierung);
        final var tx = (nachfolgerbreite - vorgaengerbreite * skalierung) / 2;
        final var ty = (nachfolgerhoehe - vorgaengerhoehe * skalierung) / 2;
        arbeitsblatt.grafik.translate(tx, ty);
        arbeitsblatt.grafik.drawImage(schattenblatt.bild, 0, 0, null);
        arbeitsblatt.grafik.setTransform(transformation);
        schattenblatt = erzeugeBlatt(nachfolgerbreite, nachfolgerhoehe, schattenblatt.grafik);
        vorgaengerppp = nachfolgerppp;
    }

    private void faerbe(final boolean aktualisierung) {
        final var dauer = bearbeiteFaerbewarteschlange(aktualisierung);
        if (berechnend) {
            nachfolgerfaerbedauer += dauer;
        } else {
            nachfolgerrestfaerbedauer += dauer;
        }
    }

    private int[] farben() {
        final var datenpuffer = arbeitsblatt.bild.getRaster().getDataBuffer();
        return ((DataBufferInt) datenpuffer).getData();
    }

    private void verfolgeBerechnung() {
        if (berechnend) {
            nachfolgerfaerbedauer = 0;
            nachfolgerrestfaerbedauer = 0;
            nachfolgermaldauer = 0;
            nachfolgerrestmaldauer = 0;
            auflauftimer.restart();
            bevorzugeBildqualitaet(false);
            if (!verschiebung.equals(NULLPUNKT) || vergroesserung != 1) {
                final var transformation = schattenblatt.grafik.getTransform();
                schattenblatt.grafik.setBackground(Color.GRAY);
                schattenblatt.grafik.clearRect(0, 0, schattenblatt.bild.getWidth(), schattenblatt.bild.getHeight());
                schattenblatt.grafik.translate(fixpunkt.x * (1 - vergroesserung) + verschiebung.x,
                        fixpunkt.y * (1 - vergroesserung) + verschiebung.y);
                schattenblatt.grafik.scale(vergroesserung, vergroesserung);
                schattenblatt.grafik.drawImage(arbeitsblatt.bild, 0, 0, null);
                schattenblatt.grafik.setTransform(transformation);
                final var blatt = schattenblatt;
                schattenblatt = arbeitsblatt;
                arbeitsblatt = blatt;
                interaktivtimer.start();
            }
            verschiebung = new Point(0, 0);
            vergroesserung = 1;
        } else {
            auflauftimer.stop();
            interaktivtimer.stop();
        }
    }

    private void verfolgeZeiger(final Zeiger zeiger) {
        istScheibenanzeigeVerfuegbar = zeiger.position != null;
        if (zeiger.position != null) {
            schwebend = true;
            final var x = realgroesse(zeiger.position.x);
            final var y = realgroesse(zeiger.position.y);
            scheibenzeiger.x = x;
            scheibenzeiger.y = y;
        }
        if (zeiger.delta != null) {
            assert !zeiger.delta.equals(NULLPUNKT);
            schwebend = false;
            final var dx = realgroesse(zeiger.delta.x);
            final var dy = realgroesse(zeiger.delta.y);
            verschiebung.x += dx;
            verschiebung.y += dy;
            scheibenzeiger.x += dx;
            scheibenzeiger.y += dy;
            andreaskreuzzeiger.x += dx;
            andreaskreuzzeiger.y += dy;
        }
        if (zeiger.radstufe != 0) {
            schwebend = false;
            vergroesserung *= Ausschnitt.skalierung(zeiger.radstufe);
            final var x = realgroesse(zeiger.position.x);
            final var y = realgroesse(zeiger.position.y);
            fixpunkt.setLocation(x, y);
        }
        if (schwebend && !berechnend) {
            setzeScheibenradius();
        } else {
            scheibenradius = 0;
        }
    }

    private void skaliereScheibe() {
        final var skalierung = vorgaengerppp / nachfolgerppp;
        scheibenzeiger.x *= skalierung;
        scheibenzeiger.y *= skalierung;
        scheibenradius *= skalierung;
    }

    private void skaliereAndreaskreuz() {
        final var skalierung = vorgaengerppp / nachfolgerppp;
        andreaskreuzzeiger.x *= skalierung;
        andreaskreuzzeiger.y *= skalierung;
    }

    private void setzeScheibenradius() {
        if (!istScheibenanzeigeGewuenscht) {
            return;
        }
        final var p = scheibenzeiger.x + bildgroesse.width * scheibenzeiger.y;
        final var lq = charakteristiken[p].lq;
        scheibenradius = Math.ceil(Math.sqrt(Math.abs(lq / zelllaengenquadrat)));
    }

    private void verfolge(final Zeiger zeiger) {
        if (zeiger != null) {
            verfolgeZeiger(zeiger);
            versetzeScheibe();
        }
    }

    private void verschiebeZentrum(final PropertyChangeEvent ereignis) {
        final var zeiger = (Zeiger) ereignis.getNewValue();
        istAndreaskreuzVerfuegbar = zeiger != null;
        if (!istAndreaskreuzVerfuegbar) {
            return;
        }
        andreaskreuzzeiger.x = realgroesse(zeiger.position.x);
        andreaskreuzzeiger.y = realgroesse(zeiger.position.y);
    }

    private void versetzeScheibe() {
        if (!istScheibenanzeigeGewuenscht) {
            return;
        }
        final var x = (int) (scheibenzeiger.x - scheibenradius) + 1;
        final var y = (int) (scheibenzeiger.y - scheibenradius) + 1;
        final var s = (int) Math.ceil(2 * scheibenradius) - 1;
        final var vorgaengerrealrahmen = scheibe != null ? scheibe.getBounds() : null;
        scheibe = new Ellipse2D.Double(x, y, s, s);
        ausmale(vorgaengerrealrahmen, scheibe.getBounds());
    }

    private void versetzeAndreaskeuz() {
        if (!istAndreaskreuzanzeigeGewuenscht) {
            return;
        }
        final var s = realgroesse(2 * ANDREASKREUZRADIUS - 1);
        final var x = (int) (andreaskreuzzeiger.getX() - s / 2) + 1;
        final var y = (int) (andreaskreuzzeiger.getY() - s / 2) + 1;
        final var vorgaengerandreaskreuz = andreaskreuz != null ? andreaskreuz.getBounds() : null;
        if (vorgaengerandreaskreuz != null) {
            vorgaengerandreaskreuz.grow(2, 2);
        }
        andreaskreuz = new Rectangle2D.Double(x, y, s, s);
        ausmale(vorgaengerandreaskreuz, andreaskreuz.getBounds());
    }

    private void zeichneScheibe(final Graphics2D g2, final int x, final int y, final int s) {
        g2.setColor(Color.GRAY);
        g2.fillOval(x + 3 * s / 8, y + 3 * s / 8, s / 4, s / 4);
        g2.fillOval(x + s / 4, y + s / 4, s / 2, s / 2);
        g2.fillOval(x, y, s, s);
        g2.setColor(Color.LIGHT_GRAY);
        g2.drawOval(x + 3 * s / 8, y + 3 * s / 8, s / 4 - 1, s / 4 - 1);
        g2.drawOval(x + s / 4, y + s / 4, s / 2 - 1, s / 2 - 1);
        g2.drawOval(x, y, s - 1, s - 1);
    }

    private void zeichneAndreaskreuz(final Graphics2D g2, final int x, final int y, final int s) {
        g2.setColor(Color.DARK_GRAY);
        g2.setStroke(new BasicStroke(3, BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND));
        g2.drawLine(x - 1, y, x - 1 + s + 2, y + s + 1);
        g2.drawLine(x - 1 + s + 2, y, x - 1, y + s + 1);
        g2.setColor(Color.WHITE);
        g2.setStroke(new BasicStroke(1));
        g2.drawLine(x, y, x + s, y + s);
        g2.drawLine(x + s, y, x, y + s);
    }

}
