/******************************************************************************
 ** $Id: Inspektor.java 3631 2024-12-24 16:34:32Z wmh $
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package ebflmaennle.oberflaeche;

import java.awt.FlowLayout;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;

import javax.swing.Action;
import javax.swing.JDialog;
import javax.swing.JPanel;

import ebflmaennle.Verfolgung;
import ebflmaennle.berechnung.Charakteristik;
import ebflmaennle.berechnung.Kalkulator.Parameter;

public class Inspektor extends JDialog implements PropertyChangeListener {

    public interface Gegenstand {
        Parameter parameter(Parameter parameter);
    }

    private static String platzhalter(final String platzhalter) {
        return platzhalter.substring(0, platzhalter.indexOf(';'));
    }

    private static final long serialVersionUID = -2765520550155796249L;

    private static final String FLUCHTZEITPLATZHALTER = "##0.###E0/~;~/0";
    private static final Formatierer FLUCHTZEITFORMATIERER = new Formatierer(FLUCHTZEITPLATZHALTER);
    private static final String MULTIPLIKATORPLATZHALTER = "0.000000E0;D0";
    private static final Formatierer MULTIPLIKATORFORMATIERER = new Formatierer(MULTIPLIKATORPLATZHALTER);
    private static final String ENTFERNUNGSPLATZHALTER = "+0.000000E0;-0";
    private static final Formatierer ENTFERNUNGSFORMATIERER = new Formatierer(ENTFERNUNGSPLATZHALTER);

    private final Verfolgung<Inspektor> verfolgung = new Verfolgung<>(this);

    private final Messwert a = new Messwert(Formatierer.PLATZHALTER_KOORDINATE_POS + Formatierer.PLATZHALTER_ESIGN);
    private final Messwert b = new Messwert(Formatierer.PLATZHALTER_KOORDINATE_POS + Formatierer.PLATZHALTER_ESIGN);
    private final Messwert nuper = new Messwert(platzhalter(FLUCHTZEITPLATZHALTER) + "+0");
    private final Messwert mul = new Messwert(platzhalter(MULTIPLIKATORPLATZHALTER) + "+0");
    private final Messwert s = new Messwert(platzhalter(ENTFERNUNGSPLATZHALTER) + "+0");

    public Inspektor(final Fenster fenster, final String titel) {
        super(fenster, titel);
        setFocusableWindowState(false);
        setContentPane(pipette());
        pack();
        setLocationRelativeTo(fenster);
        aktualisiere();
    }

    @Override
    public void propertyChange(final PropertyChangeEvent event) {
        final var parameter = (Parameter) event.getNewValue();
        if (parameter == null) {
            aktualisiere();
        } else {
            aktualisiere(parameter);
        }
    }

    public void registriere(final Action aktivierung, final PropertyChangeSupport aenderungsverursacher) {
        verfolgung.registriere(aktivierung, aenderungsverursacher);
    }

    private void aktualisiere(final Parameter parameter) {
        final var x = parameter.ort().getX();
        final var y = parameter.ort().getY();
        final var nu = parameter.charakteristik().nu;
        final var mulq = parameter.charakteristik().mulq;
        final var lq = parameter.charakteristik().lq;
        a.setText(Formatierer.FORMAT_KOORDINATE.formatiere(x));
        b.setText(Formatierer.FORMAT_KOORDINATE.formatiere(y));
        nuper.setText(fluchtzeit(nu));
        mul.setText(multiplikator(mulq, nu));
        s.setText(entfernung(lq, mulq));
    }

    private void aktualisiere() {
        a.leere();
        b.leere();
        nuper.leere();
        mul.leere();
        s.leere();
    }

    private JPanel charakteristiken() {
        final var charakteristiken = new Instrument("Charakteristiken");
        charakteristiken.erstelle("  nu/per = ", nuper, "Fluchtzeit/(Domänen)Periode");
        charakteristiken.erstelle("[D]|mul| = ", mul, "[Detektor]Betrag des Multiplikators");
        charakteristiken.erstelle("     s   = ", s,
                "Schätzung für äußere Distanz (positiv) oder inneren Abstand (negativ)");
        return charakteristiken;
    }

    private String entfernung(final float lq, final float mulq) {
        if (Charakteristik.offen(lq)) {
            return "unbetrachtet";
        }
        if (Charakteristik.unbestimmt(lq)) {
            return "unbestimmt";
        }
        if (Charakteristik.zentral(lq) || Charakteristik.zentral(mulq)) {
            return "zentral";
        }
        return ENTFERNUNGSFORMATIERER.format(Math.signum(lq) * Math.sqrt(Math.abs(lq)));
    }

    private String fluchtzeit(final float nu) {
        if (Charakteristik.offen(nu)) {
            return "unbetrachtet";
        }
        if (Charakteristik.unbestimmt(nu)) {
            return "unbestimmt";
        }
        if (Charakteristik.zentral(nu)) {
            return "zentral";
        }
        return FLUCHTZEITFORMATIERER.format(nu);
    }

    private String multiplikator(final float mulq, final float nu) {
        if (Charakteristik.offen(mulq)) {
            return "unbetrachtet";
        }
        if (Charakteristik.unbestimmt(mulq)) {
            return "unbestimmt";
        }
        if (Charakteristik.zentral(mulq)) {
            return "zentral";
        }
        if (Charakteristik.aussen(nu)) {
            return "1 + positiv";
        }
        if (mulq == 0) {
            return "0";
        }
        return MULTIPLIKATORFORMATIERER.format(Math.signum(mulq) * Math.sqrt(Math.abs(mulq)));
    }

    private JPanel parameter() {
        final var koordinate = new Instrument("Parameter c = a + b*i");
        koordinate.erstelle("a = ", a);
        koordinate.erstelle("b = ", b);
        return koordinate;
    }

    private JPanel pipette() {
        final var pipette = new JPanel();
        pipette.setLayout(new FlowLayout(FlowLayout.LEADING));
        pipette.add(parameter());
        pipette.add(charakteristiken());
        return pipette;
    }
}
