/******************************************************************************
 ** $Id: Gestaltung.java 3595 2024-09-07 18:39:14Z wmh $
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package ebflmaennle.oberflaeche;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.text.DecimalFormat;
import java.util.Hashtable;

import javax.swing.AbstractButton;
import javax.swing.Action;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JSlider;
import javax.swing.JTextField;
import javax.swing.border.EmptyBorder;
import javax.swing.event.ChangeListener;

import ebflmaennle.Main.Definition;
import ebflmaennle.berechnung.Option;
import ebflmaennle.farbe.Modell;
import ebflmaennle.farbe.Modell.Modus;
import ebflmaennle.farbe.Parameter;

public class Gestaltung extends JDialog implements PropertyChangeListener, ActionListener {

    private class Aktivierung implements ActionListener {
        @Override
        public void actionPerformed(final ActionEvent event) {
            aktualisiere(Modus.valueOf(event.getActionCommand()));
        }

    }

    private static class Beschriftungstabelle extends Hashtable<Integer, JLabel> {

        private static final long serialVersionUID = -6518345900597675170L;

        private static JLabel marke(final int groessenordnung, final double schritt,
                final float schriftgroessenfaktor) {
            final var formatierer = new DecimalFormat("##0.##");
            final var marke = new JLabel(formatierer.format(farbfrequenz(groessenordnung, schritt)));
            marke.setFont(marke.getFont().deriveFont(schriftgroessenfaktor * marke.getFont().getSize()));
            return marke;
        }

        private static int position(final int groessenordnung, final double schritt) {
            return (int) ((groessenordnung + Math.log10(schritt)) * FARBFREQUENZSCHIEBERGROESSE
                    / Modell.FREQUENZORDNUNGEN);
        }

        private static double farbfrequenz(final int groessenordnung, final double schritt) {
            return Math.pow(FARBFREQUENZSTUFE, groessenordnung) * schritt;
        }

        Beschriftungstabelle() {
            int groessenordnung;
            for (groessenordnung = 0; groessenordnung < Modell.FREQUENZORDNUNGEN; groessenordnung++) {
                erzeuge(groessenordnung, 1, NORMALSCHRIFTFAKTOR);
                erzeuge(groessenordnung, 1.4, KLEINSCHRIFTFAKTOR);
                erzeuge(groessenordnung, 2, NORMALSCHRIFTFAKTOR);
                erzeuge(groessenordnung, 3, KLEINSCHRIFTFAKTOR);
                erzeuge(groessenordnung, 4, KLEINSCHRIFTFAKTOR);
                erzeuge(groessenordnung, 5, NORMALSCHRIFTFAKTOR);
                erzeuge(groessenordnung, 7, KLEINSCHRIFTFAKTOR);
            }
            erzeuge(groessenordnung, 1, NORMALSCHRIFTFAKTOR);
        }

        private void erzeuge(final int groessenordnung, final double schritt, final float schriftfaktor) {
            put(position(groessenordnung, schritt), marke(groessenordnung, schritt, schriftfaktor));
        }

    }

    private static final long serialVersionUID = -1046408919243272543L;
    private static final int ABSTAND = 10;
    private static final int PLATZBESCHAFFUNG = 175;
    private static final float HINWEISGROESSE = 10f;
    private static final int RAHMEN = 5;
    private static final int FARBFREQUENZSTUFE = 10;
    private static final int FARBFREQUENZSCHIEBERGROESSE = 400;
    private static final int FREQUENZFAKTORMINIMUM = 0;
    private static final int FREQUENZFAKTORMAXIMUM = 25;
    private static final int FREQUENZFAKTORSTANDARD = 5;

    private static final float NORMALSCHRIFTFAKTOR = 1f;
    private static final float KLEINSCHRIFTFAKTOR = 0.8f;

    private static JLabel legende(final String beschreibung) {
        final var label = new JLabel(beschreibung);
        label.setBorder(new EmptyBorder(RAHMEN, RAHMEN, RAHMEN, RAHMEN));
        label.setAlignmentX(Component.CENTER_ALIGNMENT);
        label.setForeground(Color.ORANGE.darker());
        return label;
    }

    private final Modell modell;
    private final Aktivierung aktivierung = new Aktivierung();
    private JPanel reglerfeld;
    private JSlider farbfrequenzregler;
    private JSlider frequenzfaktorregler;
    private JSlider spektrumregler;
    private JSlider verschiebungsregler;
    private JSlider korrekturregler;
    private JRadioButton hsbmodusknopf;
    private JRadioButton lchmodusknopf;
    private JRadioButton retromodusknopf;
    private JRadioButton zweifarbenmodusknopf;
    private JRadioButton graumodusknopf;
    private JRadioButton standardfuellungsknopf;
    private JRadioButton abstandkonturenknopf;
    private JRadioButton multiplikatorkonturenknopf;
    private JButton prinzipminusknopf;
    private JButton prinzipplusknopf;
    private JCheckBoxMenuItem zentrumhervorhebungsknopf;
    private boolean multiplikatorkonturen = Option.MULTIPLIKATOREN;
    private boolean abstandkonturen = Option.ABSTANDSCHAETZUNG;
    private boolean zentrumhervorhebung = Option.ZENTRUMDETEKTION;
    private boolean prinzipdarstellung = Option.PRINZIPIEN;

    public Gestaltung(final Fenster fenster, final Modell modell) {
        super(fenster, "Ebflmännle: Farbgestaltung");
        this.modell = modell;
        setFocusableWindowState(false);
        erzeugeModenknoepfe();
        erzeugeFuellungswaehler();
        erzeugeReglerfeld();
        setContentPane(hauptfeld());
        pack();
        setLocationRelativeTo(fenster);
    }

    @Override
    public void actionPerformed(final ActionEvent ereignis) {
        hsbmodusknopf.setSelected(true);
    }

    @Override
    public void propertyChange(final PropertyChangeEvent ereignis) {
        final var eigenschaft = ereignis.getPropertyName();
        final var wert = ereignis.getNewValue();
        if (Action.SELECTED_KEY.equals(eigenschaft)) {
            final var aktion = (Action) ereignis.getSource();
            final var kommando = aktion.getValue(Action.ACTION_COMMAND_KEY);
            behandleAktion(kommando, aktion);
        } else if ("Import".equals(eigenschaft)) {
            final var parameter = ((Definition) wert).parameter();
            aktualisiere(parameter);
            aktualisiere(parameter.modus);
            modell.synchronisiere();
        } else if ("Berechnung".equals(eigenschaft)) {
            behandleBerechnung(!wert.equals(true));
        }
    }

    private void behandleWahl(final boolean wert, final AbstractButton wahl, final AbstractButton standard) {
        wahl.setEnabled(wert);
        if (!wahl.isEnabled() && wahl.isSelected()) {
            standard.setSelected(true);
        }
    }

    private void behandleAktion(final Object kommando, final Action aktion) {
        final var wert = (boolean) aktion.getValue(Action.SELECTED_KEY);
        if ("Abstandschätzung".equals(kommando)) {
            abstandkonturen = wert;
            behandleWahl(wert, abstandkonturenknopf, standardfuellungsknopf);
        } else if ("Multiplikatoren".equals(kommando)) {
            multiplikatorkonturen = wert;
            behandleWahl(wert, multiplikatorkonturenknopf, standardfuellungsknopf);
        } else if ("Zentrumdetektion".equals(kommando)) {
            zentrumhervorhebung = wert;
            zentrumhervorhebungsknopf.setEnabled(wert);
        } else if ("Prinzipien".equals(kommando)) {
            prinzipdarstellung = wert;
            behandlePrinzipien(wert);
        }
    }

    private void behandleBerechnung(final boolean aktiviert) {
        farbfrequenzregler.setEnabled(aktiviert);
        verschiebungsregler.setEnabled(aktiviert);
        spektrumregler.setEnabled(aktiviert);
        korrekturregler.setEnabled(aktiviert);
        frequenzfaktorregler.setEnabled(aktiviert);
        hsbmodusknopf.setEnabled(aktiviert);
        lchmodusknopf.setEnabled(aktiviert);
        retromodusknopf.setEnabled(aktiviert);
        zweifarbenmodusknopf.setEnabled(aktiviert);
        graumodusknopf.setEnabled(aktiviert);
        standardfuellungsknopf.setEnabled(aktiviert);
        abstandkonturenknopf.setEnabled(aktiviert && abstandkonturen);
        multiplikatorkonturenknopf.setEnabled(aktiviert && multiplikatorkonturen);
        prinzipminusknopf.setEnabled(aktiviert && prinzipdarstellung);
        prinzipplusknopf.setEnabled(aktiviert && prinzipdarstellung);
        zentrumhervorhebungsknopf.setEnabled(aktiviert && zentrumhervorhebung);
    }

    private void behandlePrinzipien(final boolean prinzipien) {
        prinzipminusknopf.setEnabled(prinzipien);
        prinzipplusknopf.setEnabled(prinzipien);
    }

    private void aktualisiere(final Modus modus) {
        (switch (modus) {
            case HSB -> hsbmodusknopf;
            case LCH -> lchmodusknopf;
            case RETRO -> retromodusknopf;
            case ZWEIFARBIG -> zweifarbenmodusknopf;
            case GRAU -> graumodusknopf;
        }).setSelected(true);
        final var bunt = modus == Modus.HSB || modus == Modus.LCH;
        final var grau = modus == Modus.GRAU;
        final var zweifarbig = modus == Modus.ZWEIFARBIG;
        korrekturregler.setEnabled(bunt || grau || zweifarbig);
        farbfrequenzregler.setEnabled(bunt || grau || zweifarbig);
        spektrumregler.setEnabled(bunt);
    }

    private void aktualisiere(final Parameter parameter) {
        farbfrequenzregler.setValue((int) (-Math.log10(parameter.farbperiode) / Modell.FREQUENZORDNUNGEN
                * farbfrequenzregler.getMaximum()));
        korrekturregler.setValue((int) Math.round(50 * (1 - Math.log(parameter.korrektur))));
        spektrumregler.setValue((int) (parameter.spektrum * spektrumregler.getMaximum()));
        verschiebungsregler.setValue((int) (parameter.verschiebung * verschiebungsregler.getMaximum()));
        frequenzfaktorregler.setValue((int) (parameter.frequenzfaktor));
        standardfuellungsknopf.setSelected(true);
        multiplikatorkonturen = parameter.multiplikatorkonturen;
        multiplikatorkonturenknopf.setEnabled(multiplikatorkonturen);
        multiplikatorkonturenknopf.setSelected(multiplikatorkonturen);
        abstandkonturen = parameter.abstandkonturen;
        abstandkonturenknopf.setEnabled(abstandkonturen);
        abstandkonturenknopf.setSelected(abstandkonturen);
        zentrumhervorhebung = parameter.zentrumhervorhebung;
        zentrumhervorhebungsknopf.setEnabled(zentrumhervorhebung);
        zentrumhervorhebungsknopf.setSelected(zentrumhervorhebung);
        prinzipdarstellung = parameter.prinzipdarstellung;
        prinzipminusknopf.setEnabled(prinzipdarstellung);
        prinzipplusknopf.setEnabled(prinzipdarstellung);
    }

    private JRadioButton modenknopf(final Modus modus, final Modell modell, final String toolTip, final int mnemonic) {
        final var radioButton = new JRadioButton(modus.toString());
        radioButton.setActionCommand(modus.name());
        radioButton.setMnemonic(mnemonic);
        radioButton.addActionListener(modell.moduswaechter());
        radioButton.addActionListener(aktivierung);
        radioButton.setToolTipText(toolTip);
        return radioButton;
    }

    private void erzeugeModenknoepfe() {
        hsbmodusknopf = modenknopf(Modus.HSB, modell, "Neonbunter intensiver Regenbogen. Schnell.", KeyEvent.VK_H);
        hsbmodusknopf.setSelected(true);
        lchmodusknopf = modenknopf(Modus.LCH, modell, "Pastellfarbener Verlauf mit gleichmäßiger Helligkeit. Langsam.",
                KeyEvent.VK_L);
        retromodusknopf = modenknopf(Modus.RETRO, modell,
                "Pallete mit Retrostandardfarben von hohem Kontrast. Sehr schnell.", KeyEvent.VK_T);
        zweifarbenmodusknopf = modenknopf(Modus.ZWEIFARBIG, modell, "Zweifarbig, abgeleitet vom HSB Farbraum. Schnell.",
                KeyEvent.VK_Z);
        graumodusknopf = modenknopf(Modus.GRAU, modell, "Graustufen, abgeleitet vom HSB Farbraum. Schnell.",
                KeyEvent.VK_G);
    }

    private JSlider regler(final int min, final int max, final int wert, final int hauptabstand,
            final int zwischenabstand, final boolean einrastend, final ChangeListener beobachter) {
        final var regler = new JSlider();
        regler.setMinimum(min);
        regler.setMaximum(max);
        regler.setValue(wert);
        regler.setPaintLabels(true);
        regler.setPaintTicks(true);
        regler.setSnapToTicks(einrastend);
        regler.setMajorTickSpacing(hauptabstand);
        regler.setMinorTickSpacing(zwischenabstand);
        regler.addChangeListener(beobachter);
        return regler;
    }

    private void erzeugeReglerfeld() {
        reglerfeld = new JPanel();
        reglerfeld.setBorder(BorderFactory.createCompoundBorder(BorderFactory.createEtchedBorder(),
                BorderFactory.createEmptyBorder(ABSTAND, ABSTAND, ABSTAND, ABSTAND)));
        reglerfeld.setLayout(new BoxLayout(reglerfeld, BoxLayout.Y_AXIS));
        farbfrequenzregler = regler(0, FARBFREQUENZSCHIEBERGROESSE, FARBFREQUENZSCHIEBERGROESSE / 10,
                FARBFREQUENZSCHIEBERGROESSE / Modell.FREQUENZORDNUNGEN, 0, false, modell.farbfrequenzwaechter());
        farbfrequenzregler.setLabelTable(new Beschriftungstabelle());
        reglerfeld.add(Box.createVerticalStrut(ABSTAND));
        reglerfeld.add(legende("Farbfrequenz (\"Rauschunterdrückung\")"));
        reglerfeld.add(farbfrequenzregler);
        reglerfeld.add(Box.createVerticalGlue());
        verschiebungsregler = regler(0, 360, 180, 45, 5, true, modell.verschiebungswaechter());
        reglerfeld.add(legende("Farbrotation in ° (\"Farbverschiebung \")"));
        reglerfeld.add(verschiebungsregler);
        reglerfeld.add(Box.createVerticalGlue());
        spektrumregler = regler(0, 100, 100, 10, 2, true, modell.spektrumwaechter());
        reglerfeld.add(legende("Farbspektrumausschnitt (in %)"));
        reglerfeld.add(spektrumregler);
        reglerfeld.add(Box.createVerticalGlue());
        korrekturregler = regler(0, 100, 50, 25, 5, true, modell.korrekturwaechter());
        reglerfeld.add(legende("Korrektur (in %)"));
        reglerfeld.add(korrekturregler);
        reglerfeld.add(Box.createVerticalGlue());
        frequenzfaktorregler = regler(FREQUENZFAKTORMINIMUM, FREQUENZFAKTORMAXIMUM, FREQUENZFAKTORSTANDARD, 5, 1, true,
                modell.frequenzfaktorwaechter());
        reglerfeld.add(legende("Frequenzfaktor (Entfernungen und Konturen)"));
        reglerfeld.add(frequenzfaktorregler);
        reglerfeld.add(Box.createVerticalStrut(ABSTAND));
    }

    private void erzeugeFuellungswaehler() {
        standardfuellungsknopf = new JRadioButton("Standardfüllung");
        standardfuellungsknopf.setMnemonic(KeyEvent.VK_S);
        standardfuellungsknopf.setSelected(true);
        abstandkonturenknopf = new JRadioButton("Abstandkonturen");
        abstandkonturenknopf.setToolTipText("Nur verfügbar bei Abstandschätzung");
        abstandkonturenknopf.setEnabled(Option.ABSTANDSCHAETZUNG);
        abstandkonturenknopf.addItemListener(modell.abstandkonturenwaechter());
        abstandkonturenknopf.setMnemonic(KeyEvent.VK_A);
        multiplikatorkonturenknopf = new JRadioButton("Multiplikatorkonturen");
        multiplikatorkonturenknopf.setToolTipText("Nur verfügbar bei Multiplikatoren");
        multiplikatorkonturenknopf.setEnabled(Option.MULTIPLIKATOREN);
        multiplikatorkonturenknopf.addItemListener(modell.multiplikatorkonturenwaechter());
        multiplikatorkonturenknopf.setMnemonic(KeyEvent.VK_M);
    }

    private JPanel auswahlfeld(final Modell modell) {
        final var auswahlfeld = new JPanel();
        auswahlfeld.setLayout(new BoxLayout(auswahlfeld, BoxLayout.PAGE_AXIS));
        auswahlfeld.add(faerbungsfeld());
        auswahlfeld.add(Box.createVerticalStrut(ABSTAND));
        auswahlfeld.add(fuellfeld());
        auswahlfeld.add(Box.createVerticalStrut(ABSTAND));
        auswahlfeld.add(zentrumfeld(modell));
        auswahlfeld.add(Box.createVerticalStrut(ABSTAND));
        auswahlfeld.add(palettenfeld(modell));
        return auswahlfeld;
    }

    private JPanel faerbungsfeld() {
        final var faerbungsgruppe = new ButtonGroup();
        faerbungsgruppe.add(hsbmodusknopf);
        faerbungsgruppe.add(lchmodusknopf);
        faerbungsgruppe.add(retromodusknopf);
        faerbungsgruppe.add(zweifarbenmodusknopf);
        faerbungsgruppe.add(graumodusknopf);
        final var faerbungsfeld = new JPanel();
        faerbungsfeld.setLayout(new BoxLayout(faerbungsfeld, BoxLayout.PAGE_AXIS));
        faerbungsfeld.setBorder(BorderFactory.createCompoundBorder(
                BorderFactory.createTitledBorder(BorderFactory.createEtchedBorder(), ("Farbmodelle")),
                BorderFactory.createEmptyBorder(ABSTAND, ABSTAND, ABSTAND, ABSTAND)));
        faerbungsfeld.add(Box.createVerticalGlue());
        faerbungsfeld.add(hsbmodusknopf);
        faerbungsfeld.add(Box.createVerticalStrut(ABSTAND));
        faerbungsfeld.add(lchmodusknopf);
        faerbungsfeld.add(Box.createVerticalStrut(ABSTAND));
        faerbungsfeld.add(retromodusknopf);
        faerbungsfeld.add(Box.createVerticalStrut(ABSTAND));
        faerbungsfeld.add(zweifarbenmodusknopf);
        faerbungsfeld.add(Box.createVerticalStrut(ABSTAND));
        faerbungsfeld.add(graumodusknopf);
        faerbungsfeld.add(Box.createVerticalGlue());
        return faerbungsfeld;
    }

    private JPanel fuellfeld() {
        final var fuellgruppe = new ButtonGroup();
        fuellgruppe.add(standardfuellungsknopf);
        fuellgruppe.add(abstandkonturenknopf);
        fuellgruppe.add(multiplikatorkonturenknopf);
        final var fuellfeld = new JPanel();
        fuellfeld.setLayout(new BoxLayout(fuellfeld, BoxLayout.PAGE_AXIS));
        fuellfeld.setBorder(BorderFactory.createCompoundBorder(
                BorderFactory.createTitledBorder(BorderFactory.createEtchedBorder(), "Füllungen"),
                BorderFactory.createEmptyBorder(ABSTAND, ABSTAND, ABSTAND, ABSTAND)));
        fuellfeld.add(Box.createVerticalGlue());
        fuellfeld.add(standardfuellungsknopf);
        fuellfeld.add(Box.createVerticalStrut(ABSTAND));
        fuellfeld.add(multiplikatorkonturenknopf);
        fuellfeld.add(Box.createVerticalStrut(ABSTAND));
        fuellfeld.add(abstandkonturenknopf);
        fuellfeld.add(Box.createVerticalGlue());
        return fuellfeld;
    }

    private JPanel hauptfeld() {
        final var hauptfeld = new JPanel(new BorderLayout(ABSTAND, ABSTAND));
        hauptfeld.setBorder(new EmptyBorder(ABSTAND, ABSTAND, ABSTAND, ABSTAND));
        hauptfeld.add(auswahlfeld(modell), BorderLayout.LINE_START);
        hauptfeld.add(reglerfeld, BorderLayout.CENTER);
        hauptfeld.add(hinweisfeld(), BorderLayout.PAGE_END);
        return hauptfeld;
    }

    private JPanel hinweisfeld() {
        final var hinweis = new JTextField();
        hinweis.setBorder(BorderFactory.createCompoundBorder(BorderFactory.createEtchedBorder(),
                BorderFactory.createEmptyBorder(ABSTAND / 2, PLATZBESCHAFFUNG, ABSTAND / 2, PLATZBESCHAFFUNG)));
        hinweis.setAlignmentX(Component.LEFT_ALIGNMENT);
        hinweis.setEditable(false);
        hinweis.setFont(hinweis.getFont().deriveFont(HINWEISGROESSE));
        hinweis.setText("""
                Farbgestaltungen \
                "Zweifarbig" & "Graustufen"\
                 kommen mit Distanzschätzung besser zur Geltung.""");
        final var hinweisfeld = new JPanel();
        hinweisfeld.add(hinweis);
        return hinweisfeld;
    }

    private JButton palettenknopf(final String text, final String hilfe, final int mnemonic,
            final ActionListener beobachter, final boolean enabled) {
        final var button = new JButton(text);
        button.setToolTipText(hilfe);
        button.setMnemonic(mnemonic);
        button.setAlignmentX(Component.LEFT_ALIGNMENT);
        button.addActionListener(beobachter);
        button.setEnabled(enabled);
        return button;
    }

    private JPanel palettenfeld(final Modell modell) {
        prinzipminusknopf = palettenknopf("Ab", "Nur verfügbar bei der Darstellung von Prinzipien", KeyEvent.VK_B,
                modell, Option.PRINZIPIEN);
        prinzipplusknopf = palettenknopf("Auf", "Nur verfügbar bei der Darstellung von Prinzipien", KeyEvent.VK_F,
                modell, Option.PRINZIPIEN);
        final var dehnfeld = new JPanel();
        dehnfeld.add(prinzipminusknopf);
        dehnfeld.add(Box.createHorizontalGlue());
        dehnfeld.add(prinzipplusknopf);
        final var palettenfeld = new JPanel();
        palettenfeld.setLayout(new BoxLayout(palettenfeld, BoxLayout.PAGE_AXIS));
        palettenfeld.setAlignmentX(Component.LEFT_ALIGNMENT);
        palettenfeld.setBorder(BorderFactory.createCompoundBorder(
                BorderFactory.createTitledBorder(BorderFactory.createEtchedBorder(), "Prinzippalette"),
                BorderFactory.createEmptyBorder(ABSTAND, ABSTAND, ABSTAND, ABSTAND)));
        palettenfeld.add(Box.createVerticalGlue());
        palettenfeld.add(dehnfeld, BorderLayout.LINE_START);
        palettenfeld.add(Box.createVerticalGlue());
        return palettenfeld;
    }

    private JPanel zentrumfeld(final Modell modell) {
        zentrumhervorhebungsknopf = new JCheckBoxMenuItem("Hervorhebung");
        zentrumhervorhebungsknopf.setToolTipText("Nur verfügbar bei Zentrumdetektion");
        zentrumhervorhebungsknopf.setMnemonic(KeyEvent.VK_V);
        zentrumhervorhebungsknopf.setActionCommand("Zentrumhervorhebung");
        zentrumhervorhebungsknopf.addActionListener(modell);
        zentrumhervorhebungsknopf.setEnabled(Option.INNENBLOCKSCHAETZUNG);
        final var zentrumfeld = new JPanel();
        zentrumfeld.setLayout(new BoxLayout(zentrumfeld, BoxLayout.PAGE_AXIS));
        zentrumfeld.setAlignmentX(Component.LEFT_ALIGNMENT);
        zentrumfeld.setBorder(BorderFactory.createCompoundBorder(
                BorderFactory.createTitledBorder(BorderFactory.createEtchedBorder(), "Zentrum"),
                BorderFactory.createEmptyBorder(ABSTAND, ABSTAND, ABSTAND, ABSTAND)));
        zentrumfeld.add(zentrumhervorhebungsknopf);
        return zentrumfeld;
    }

}
