/******************************************************************************
 ** $Id: Delegation.java 3572 2024-08-31 21:35:21Z wmh $
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package ebflmaennle.oberflaeche;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.geom.Point2D;

import ebflmaennle.berechnung.Ausschnitt;
import ebflmaennle.berechnung.Option;
import ebflmaennle.berechnung.Option.Qualitaet;
import ebflmaennle.oberflaeche.Leinwand.Zeiger;

public class Delegation {

    public interface Beantragung {
        void beantrage();
    }

    public interface Position {
        /** Gibt relative Mauskoordinaten zurück. */
        Point2D.Double fokus();

        void beachte(Zeiger zeiger);
    }

    private abstract class Bearbeitungsaktion extends Aktion {
        private static final long serialVersionUID = -6983195702679313373L;

        Bearbeitungsaktion(final String name, final int taste, final boolean aktiv, final String hilfe) {
            super(name, taste, aktiv, hilfe);
            addPropertyChangeListener(event -> {
                if (SELECTED_KEY.equals(event.getPropertyName())) {
                    praepariere((boolean) event.getNewValue());
                }
            });
        }

        Bearbeitungsaktion(final String name, final int taste, final boolean aktiv) {
            this(name, taste, aktiv, null);
        }

        @Override
        public void actionPerformed(final ActionEvent event) {
            beantragung.beantrage();
        }

        protected abstract void praepariere(final boolean aktiviert);
    }

    private class Abstandschaetzungsaktion extends Bearbeitungsaktion {
        private static final long serialVersionUID = -8366692353523901121L;

        Abstandschaetzungsaktion(final String name, final int taste) {
            super(name, taste, option.abstandschaetzung);
        }

        @Override
        protected void praepariere(final boolean aktiviert) {
            option.abstandschaetzung = aktiviert;
            scheibenaktion.setEnabled(aktiviert || option.distanzschaetzung);
        }
    }

    private class Antialiasingaktion extends Bearbeitungsaktion {
        private static final boolean ANFANGSZUSTAND = false;
        private static final long serialVersionUID = 3636745918591798624L;

        Antialiasingaktion(final String name, final int taste) {
            super(name, taste, ANFANGSZUSTAND);
        }

        @Override
        protected void praepariere(final boolean aktiviert) {
            ausschnitt.aktiviereAntialiasing(aktiviert);
        }
    }

    private class Aussenblockschaetzungsaktion extends Bearbeitungsaktion {
        private static final long serialVersionUID = 7332274787429054094L;

        Aussenblockschaetzungsaktion(final String name, final int taste) {
            super(name, taste, option.aussenblockschaetzung);
        }

        @Override
        protected void praepariere(final boolean aktiviert) {
            option.aussenblockschaetzung = aktiviert;
        }
    }

    private class Bildfrequenzaktion extends Bearbeitungsaktion {
        private static final long serialVersionUID = 1842223539667089506L;

        Bildfrequenzaktion(final String name, final int taste) {
            super(name, taste, option.bildfrequenz);
        }

        @Override
        protected void praepariere(final boolean aktiviert) {
            option.bildfrequenz = aktiviert;
        }
    }

    private class Qualitaetsautomatikaktion extends Bearbeitungsaktion {
        private static final long serialVersionUID = 6503685209983942624L;

        Qualitaetsautomatikaktion(final String name, final int taste) {
            super(name, taste, option.qualitaet == Qualitaet.AUTOMATIK);
        }

        @Override
        protected void praepariere(final boolean aktiviert) {
            option.qualitaet = Option.Qualitaet.AUTOMATIK;
        }
    }

    private class Qualitaetswahlkaktion extends Bearbeitungsaktion {
        private static final long serialVersionUID = 6594663825713938523L;

        Qualitaetswahlkaktion(final String name, final int taste) {
            super(name, taste, option.qualitaet == Qualitaet.BENUTZER);
        }

        @Override
        protected void praepariere(final boolean aktiviert) {
            option.qualitaet = Option.Qualitaet.BENUTZER;
        }
    }

    private class Qualitaetsoptimumkaktion extends Bearbeitungsaktion {
        private static final long serialVersionUID = -1598754339708493020L;

        private boolean innendetektion;

        Qualitaetsoptimumkaktion(final String name, final int taste) {
            super(name, taste, option.qualitaet == Qualitaet.OPTIMUM);
        }

        @Override
        protected void praepariere(final boolean aktiviert) {
            if (aktiviert) {
                innendetektion = option.innendetektion;
                option.innendetektion = true;
                innendetektionsaktion.setEnabled(false);
                innendetektionsaktion.setSelected(true);
            } else {
                innendetektionsaktion.setEnabled(true);
                innendetektionsaktion.setSelected(innendetektion);
            }
            option.qualitaet = Option.Qualitaet.OPTIMUM;
        }
    }

    private class Distanzschaetzungsaktion extends Bearbeitungsaktion {
        private static final long serialVersionUID = 978333563173210533L;

        Distanzschaetzungsaktion(final String name, final int taste) {
            super(name, taste, option.distanzschaetzung);
        }

        @Override
        protected void praepariere(final boolean aktiviert) {
            option.distanzschaetzung = aktiviert;
            scheibenaktion.setEnabled(aktiviert || option.abstandschaetzung);
        }
    }

    private class Geschwindigkeitsaktion extends Bearbeitungsaktion {
        private static final long serialVersionUID = 7433432121124899400L;

        Geschwindigkeitsaktion(final String name, final int taste) {
            super(name, taste, option.bevorzugung == Option.Bevorzugung.ZEIT);
        }

        @Override
        protected void praepariere(final boolean aktiviert) {
            option.bevorzugung = Option.Bevorzugung.ZEIT;
            vorschauaktion.setSelected(aktiviert);
        }
    }

    private class Innenblockschaetzungsaktion extends Bearbeitungsaktion {
        private static final long serialVersionUID = -1729177162813723452L;

        Innenblockschaetzungsaktion(final String name, final int taste) {
            super(name, taste, option.innenblockschaetzung);
        }

        @Override
        protected void praepariere(final boolean aktiviert) {
            option.innenblockschaetzung = aktiviert;
        }
    }

    private class Innendetektionsaktion extends Bearbeitungsaktion {
        private static final long serialVersionUID = -6117340362425364110L;

        Innendetektionsaktion(final String name, final int taste) {
            super(name, taste, option.innendetektion, "Gesperrt, falls Qualitaet auf Optimum eingestellt");
        }

        @Override
        protected void praepariere(final boolean aktiviert) {
            option.innendetektion = aktiviert;
        }
    }

    private class Koerperkopferkennungsaktion extends Bearbeitungsaktion {
        private static final long serialVersionUID = -2554165174875866049L;

        Koerperkopferkennungsaktion(final String name, final int taste) {
            super(name, taste, option.koerperkopferkennung);
        }

        @Override
        protected void praepariere(final boolean aktiviert) {
            option.koerperkopferkennung = aktiviert;
        }
    }

    private class Multiplikatorenaktion extends Bearbeitungsaktion {
        private static final long serialVersionUID = 6094563871756560584L;

        Multiplikatorenaktion(final String name, final int taste) {
            super(name, taste, option.multiplikatoren);
        }

        @Override
        protected void praepariere(final boolean aktiviert) {
            option.multiplikatoren = aktiviert;
            abstandschaetzungsaktion.setEnabled(aktiviert);
        }
    }

    private class Prinzipienaktion extends Bearbeitungsaktion {
        private static final long serialVersionUID = -5145762843933073569L;

        Prinzipienaktion(final String name, final int taste) {
            super(name, taste, option.prinzipien);
        }

        @Override
        protected void praepariere(final boolean aktiviert) {
            option.prinzipien = aktiviert;
        }
    }

    private static class Scheibenaktion extends Aktion {
        private static final long serialVersionUID = 4198203775883482121L;

        Scheibenaktion(final String name, final int taste) {
            super(name, taste, """
                    <html>
                    <p>Überlagert eine transparente Scheibe zur Darstellung der Distanz- oder Abstandschätzungen.</p>
                    <p>Dazu muss eine dieser Schätzungen aktiv sein.</p>
                    </html>""");
            setEnabled(Option.ABSTANDSCHAETZUNG || Option.DISTANZSCHAETZUNG);
        }

        @Override
        public void actionPerformed(final ActionEvent ereignis) {
        }
    }

    private static class Andreaskreuzaktion extends Aktion {
        private static final long serialVersionUID = -6276894456741140773L;

        Andreaskreuzaktion(final String name, final int taste) {
            super(name, taste,
                    "Berechnet ein zum Mauszeiger nahegelegenes Zentrum. Kennzeichnung durch ein Andreaskreuz.");
        }

        @Override
        public void actionPerformed(final ActionEvent ereignis) {
        }
    }

    private class Schrittschaetzungsaktion extends Bearbeitungsaktion {
        private static final long serialVersionUID = 8860346759964044640L;

        Schrittschaetzungsaktion(final String name, final int taste) {
            super(name, taste, option.schrittschaetzung);
            setEnabled(Option.SCHRITTSCHAETZUNG && Option.AUSSENBLOCKSCHAETZUNG
                    || Option.DISTANZSCHAETZUNG && Option.INNENBLOCKSCHAETZUNG);
        }

        @Override
        protected void praepariere(final boolean aktiviert) {
            option.schrittschaetzung = aktiviert;
        }
    }

    private class Vorschauaktion extends Bearbeitungsaktion {
        private static final boolean ANFANGSZUSTAND = false;
        private static final long serialVersionUID = -676031783857352614L;

        Vorschauaktion(final String name, final int taste) {
            super(name, taste, ANFANGSZUSTAND);
        }

        @Override
        protected void praepariere(final boolean aktiviert) {
            ausschnitt.aktiviereVorschau(aktiviert);
        }
    }

    private class Zentrumdetektionsaktion extends Bearbeitungsaktion {
        private static final long serialVersionUID = 5501045476341341583L;

        Zentrumdetektionsaktion(final String name, final int taste) {
            super(name, taste, option.zentrumdetektion);
        }

        @Override
        protected void praepariere(final boolean aktiviert) {
            option.zentrumdetektion = aktiviert;
        }
    }

    private static class Parameteraktion extends Aktion {
        private static final long serialVersionUID = -8404698992978753738L;

        Parameteraktion(final String name, final int taste, final String hilfe) {
            super(name, taste, hilfe);
        }

        @Override
        public void actionPerformed(final ActionEvent ereignis) {
        }
    }

    private class Ausschnittsbeobachter implements ActionListener {
        private static final double REL = 0.0625;

        private final Position position;

        Ausschnittsbeobachter(final Position position) {
            this.position = position;
        }

        @Override
        public void actionPerformed(final ActionEvent event) {
            final var kommando = event.getActionCommand();
            var veraltet = true;
            switch (kommando) {
                case "*" -> veraltet = skaliere(-1);
                case "**" -> veraltet = skaliere(-2);
                case "/" -> veraltet = skaliere(+1);
                case "//" -> veraltet = skaliere(+2);
                case "<" -> verschiebe(+1, 0);
                case ">" -> verschiebe(-1, 0);
                case "R" -> verschiebe(0, -1);
                case "H" -> verschiebe(0, +1);
                case "<<" -> verschiebe(+2, 0);
                case ">>" -> verschiebe(-2, 0);
                case "RR" -> verschiebe(0, -2);
                case "HH" -> verschiebe(0, +2);
                case "O" -> zentriere();
                case "+" -> fokussiere(+1);
                case "-" -> fokussiere(-1);
                case "++" -> fokussiere(+2);
                case "--" -> fokussiere(-2);
            }
            if (veraltet) {
                beantragung.beantrage();
            }
        }

        private boolean skaliere(final int stufe) {
            final var veraendert = ausschnitt.skaliere(stufe);
            if (veraendert) {
                position.beachte(new Zeiger(ausschnitt.darstellungsmittelpunkt(), null, stufe));
            }
            return veraendert;
        }

        private void verschiebe(final int relx, final int rely) {
            ausschnitt.verschiebe(REL * relx, REL * rely);
            position.beachte(new Zeiger(null, ausschnitt.darstellungspunkt(REL * relx, REL * rely), 0));
        }

        private void zentriere() {
            final var fokus = position.fokus();
            if (fokus == null) {
                return;
            }
            final var relx = 0.5 - fokus.x;
            final var rely = 0.5 - fokus.y;
            position.beachte(new Zeiger(null, ausschnitt.darstellungspunkt(relx, rely), 0));
            ausschnitt.verschiebe(relx, rely);
        }

        private void fokussiere(final int staerke) {
            var fokus = position.fokus();
            if (fokus == null) {
                fokus = new Point2D.Double(0.5, 0.5);
            }
            final var relx = 0.5 - fokus.x;
            final var rely = 0.5 - fokus.y;
            final var fixpunkt = ausschnitt.darstellungspunkt(fokus.x, fokus.y);
            final var delta = ausschnitt.darstellungspunkt(relx, rely);
            final var zoomschritt = Ausschnitt.zoomschritt(staerke);
            position.beachte(new Zeiger(fixpunkt, delta, zoomschritt));
            ausschnitt.fokussiere(fokus.x, fokus.y, zoomschritt);
            ausschnitt.verschiebe(relx, rely);
        }

    }

    class Qualitaetsbeobachter implements ActionListener {
        @Override
        public void actionPerformed(final ActionEvent event) {
            switch (event.getActionCommand()) {
                case "Standard":
                    option.bevorzugung = Option.Bevorzugung.STANDARD;
                    beantragung.beantrage();
                    break;
                case "Anpassung":
                    beantragung.beantrage();
                    break;
                case "Reduktion":
                    ausschnitt.skizziere();
                    beantragung.beantrage();
                    break;
                case "Erhöhung":
                    ausschnitt.detailliere();
                    beantragung.beantrage();
                    break;
                default:
                    break;
            }
        }
    }

    private final Ausschnitt ausschnitt;
    private final Beantragung beantragung;
    private final Option option;

    private Abstandschaetzungsaktion abstandschaetzungsaktion;
    private Aussenblockschaetzungsaktion aussenblockschaetzungsaktion;
    private Distanzschaetzungsaktion distanzschaetzungsaktion;
    private Innenblockschaetzungsaktion innenblockschaetzungsaktion;
    private Innendetektionsaktion innendetektionsaktion;
    private Scheibenaktion scheibenaktion;
    private Schrittschaetzungsaktion schrittschaetzungsaktion;
    private Vorschauaktion vorschauaktion;
    private Qualitaetswahlkaktion qualitaetswahlkaktion;

    private Qualitaetsbeobachter qualitaetsbeobachter;

    public Delegation(final Beantragung beantragung, final Ausschnitt ausschnitt, final Option option) {
        this.beantragung = beantragung;
        this.ausschnitt = ausschnitt;
        this.option = option;
    }

    public ActionListener ausschnittsbeobachter(final Position position) {
        return new Ausschnittsbeobachter(position);
    }

    public Aktion abstandschaetzungsaktion() {
        if (abstandschaetzungsaktion == null) {
            abstandschaetzungsaktion = new Abstandschaetzungsaktion("Abstandschätzung", KeyEvent.VK_A);
        }
        return abstandschaetzungsaktion;
    }

    public Aktion innenblockschaetzungsaktion() {
        if (innenblockschaetzungsaktion == null) {
            innenblockschaetzungsaktion = new Innenblockschaetzungsaktion("Innenblockschätzung", KeyEvent.VK_I);
        }
        return innenblockschaetzungsaktion;
    }

    public Aktion multiplikatorensaktion() {
        return new Multiplikatorenaktion("Multiplikatoren", KeyEvent.VK_M);
    }

    public Aktion prinzipienaktion() {
        return new Prinzipienaktion("Prinzipien", KeyEvent.VK_P);
    }

    public Aktion scheibenaktion() {
        if (scheibenaktion == null) {
            scheibenaktion = new Scheibenaktion("Scheibe", KeyEvent.VK_O);
        }
        return scheibenaktion;
    }

    public Aktion andreaskreuzaktion() {
        return new Andreaskreuzaktion("Andreaskreuz", KeyEvent.VK_X);
    }

    public Aktion pipettenaktion(final Inspektor pipette) {
        return new Parameteraktion("Pipette", KeyEvent.VK_NUMBER_SIGN,
                "Zeigt Ort sowie Charakteristiken zum vom Mauszeiger anvisierten Parameter.") {
            private static final long serialVersionUID = -3416881481703848871L;

            @Override
            public void actionPerformed(final ActionEvent ereignis) {
                pipette.setVisible((boolean) getValue(SELECTED_KEY));
            }
        };
    }

    public Aktion zentrumaktion(final Inspektor zentrum) {
        return new Parameteraktion("Zentrum", KeyEvent.VK_T,
                "Berechnet ein zum Mauszeiger nahegelegenes Zentrum. Zeigt Ort samt Charakteristiken.") {
            private static final long serialVersionUID = -3416881481703848871L;

            @Override
            public void actionPerformed(final ActionEvent ereignis) {
                zentrum.setVisible((boolean) getValue(SELECTED_KEY));
            }
        };
    }

    public Aktion zentrumdetektionsaktion() {
        return new Zentrumdetektionsaktion("Zentrumdetektion", KeyEvent.VK_Z);
    }

    Aktion antialiasingsaktion() {
        return new Antialiasingaktion("Antialiasing", KeyEvent.VK_2);
    }

    Aktion aussenblockschaetzungsaktion() {
        if (aussenblockschaetzungsaktion == null) {
            aussenblockschaetzungsaktion = new Aussenblockschaetzungsaktion("Außenblockschätzung", KeyEvent.VK_E);
        }
        return aussenblockschaetzungsaktion;
    }

    Aktion bildfrequenzaktion() {
        return new Bildfrequenzaktion("Bildfrequenzoptimierung", KeyEvent.VK_Q);
    }

    Aktion qualitaetsautomatikaktion() {
        return new Qualitaetsautomatikaktion("Automatik", KeyEvent.VK_SPACE);
    }

    Aktion qualitaetswahlaktion() {
        if (qualitaetswahlkaktion == null) {
            qualitaetswahlkaktion = new Qualitaetswahlkaktion("Benutzer", KeyEvent.VK_HOME);
        }
        return qualitaetswahlkaktion;
    }

    ActionListener qualitaetsbeobachter() {
        if (qualitaetsbeobachter == null) {
            qualitaetsbeobachter = new Qualitaetsbeobachter();
        }
        return qualitaetsbeobachter;
    }

    Aktion qualitaetsoptimumkaktion() {
        return new Qualitaetsoptimumkaktion("Optimum", KeyEvent.VK_END);
    }

    Aktion distanzschaetzungsaktion() {
        if (distanzschaetzungsaktion == null) {
            distanzschaetzungsaktion = new Distanzschaetzungsaktion("Distanzschätzung", KeyEvent.VK_D);
        }
        return distanzschaetzungsaktion;
    }

    Aktion geschwindigkeitsaktion() {
        return new Geschwindigkeitsaktion("Geschwindigkeit", KeyEvent.VK_F2);
    }

    Aktion innendetektionsaktion() {
        if (innendetektionsaktion == null) {
            innendetektionsaktion = new Innendetektionsaktion("Innendetektion", KeyEvent.VK_0);
        }
        return innendetektionsaktion;
    }

    Aktion koerperkopferkennungsaktion() {
        return new Koerperkopferkennungsaktion("Körper-Kopf-Erkennung", KeyEvent.VK_K);
    }

    Aktion schrittschaetzungsaktion() {
        if (schrittschaetzungsaktion == null) {
            schrittschaetzungsaktion = new Schrittschaetzungsaktion("Schrittschätzung", KeyEvent.VK_S);
        }
        return schrittschaetzungsaktion;
    }

    Aktion vorschauaktion() {
        if (vorschauaktion == null) {
            vorschauaktion = new Vorschauaktion("Vorschau", KeyEvent.VK_V);
        }
        return vorschauaktion;
    }

}
