/******************************************************************************
 ** $Id: Definitionsoperation.java 3554 2024-08-17 21:49:05Z wmh $
 ** Diese Datei ist Bestandteil der Java-Quelltexte des Programms Ebflmännle.
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package ebflmaennle.oberflaeche;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.lang.reflect.Modifier;
import java.text.ParseException;

import ebflmaennle.Main.Definition;
import ebflmaennle.berechnung.Option.Bevorzugung;
import ebflmaennle.berechnung.Option.Qualitaet;
import ebflmaennle.farbe.Modell.Modus;

public class Definitionsoperation {

    static final String FORMAT = "YAML";
    static final String SUFFIX = "." + FORMAT.toLowerCase();

    private final Definition definition;
    private File datei;

    public Definitionsoperation(final Definition definition) {
        this.definition = definition;
    }

    private String lies(final BufferedReader leser, final Object objekt)
            throws IOException, ParseException, NoSuchFieldException, IllegalArgumentException, IllegalAccessException {
        String zeile;
        for (zeile = leser.readLine(); zeile != null && zeile.startsWith(" "); zeile = leser.readLine()) {
            zeile = zeile.stripTrailing();
            final var teile = zeile.split(":");
            if (teile.length != 2) {
                throw new ParseException("Keine Zerlegung möglich von: " + zeile, 0);
            }
            final var name = teile[0].trim();
            final var wert = teile[1].trim();
            final var feld = objekt.getClass().getDeclaredField(name);
            feld.setAccessible(true);
            final var typ = feld.getType();
            if (typ == boolean.class) {
                feld.setBoolean(objekt, Boolean.parseBoolean(wert));
            } else if (typ == Modus.class) {
                feld.set(objekt, Modus.valueOf(wert));
            } else if (typ == Qualitaet.class) {
                feld.set(objekt, Qualitaet.valueOf(wert));
            } else if (typ == Bevorzugung.class) {
                feld.set(objekt, Bevorzugung.valueOf(wert));
            } else if (typ == int.class) {
                feld.setInt(objekt, Integer.parseInt(wert));
            } else if (typ == float.class) {
                feld.setFloat(objekt, Float.parseFloat(wert));
            } else if (typ == double.class) {
                feld.setDouble(objekt, Double.parseDouble(wert));
            }
        }
        return zeile;
    }

    private void schreibe(final BufferedWriter schreiber, final Object objekt)
            throws IllegalArgumentException, IllegalAccessException, IOException {
        schreiber.write(objekt.getClass().getSimpleName() + ":");
        schreiber.newLine();
        final var felder = objekt.getClass().getDeclaredFields();
        for (final var feld : felder) {
            if ((feld.getModifiers() & (Modifier.STATIC | Modifier.FINAL)) != (Modifier.STATIC | Modifier.FINAL)) {
                feld.setAccessible(true);
                final var name = feld.getName();
                final var wert = feld.get(objekt);
                final var inhalt = wert.getClass().isEnum() ? ((Enum<?>) wert).name() : wert;
                schreiber.write(" " + name + ": " + inhalt);
                schreiber.newLine();
            }
        }
    }

    private String stamm(final File datei) {
        return datei.getName().substring(0, datei.getName().toLowerCase().lastIndexOf(SUFFIX));
    }

    File datei() {
        return datei;
    }

    String exportiere(final File datei) {
        try (final var schreiber = new BufferedWriter(new FileWriter(datei))) {
            schreibe(schreiber, definition.option());
            schreibe(schreiber, definition.parameter());
            schreibe(schreiber, definition.ausschnitt());
        } catch (final IOException | IllegalArgumentException | IllegalAccessException ausnahme) {
            ausnahme.printStackTrace();
        }
        this.datei = datei;
        return stamm(datei);
    }

    String importiere(final File datei) throws ParseException, NoSuchFieldException, IllegalAccessException {
        try (final var leser = new BufferedReader(new FileReader(datei))) {
            var zeile = leser.readLine().stripTrailing();
            if (!zeile.endsWith(":")) {
                throw new ParseException("Trennzeichen \":\" erwartet in: " + zeile, zeile.length() - 1);
            }
            while (zeile != null) {
                if (zeile.equals(definition.option().getClass().getSimpleName() + ":")) {
                    zeile = lies(leser, definition.option());
                } else if (zeile.equals(definition.parameter().getClass().getSimpleName() + ":")) {
                    zeile = lies(leser, definition.parameter());
                } else if (zeile.equals(definition.ausschnitt().getClass().getSimpleName() + ":")) {
                    zeile = lies(leser, definition.ausschnitt());
                } else {
                    zeile = leser.readLine();
                }
            }
        } catch (final IOException | IllegalArgumentException ausnahme) {
            ausnahme.printStackTrace();
        }
        this.datei = datei;
        return stamm(datei);
    }

    public Definition definition() {
        return definition;
    }

}
