/******************************************************************************
 ** $Id: Modell.java 3547 2024-08-14 22:51:11Z wmh $
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package ebflmaennle.farbe;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;

import javax.swing.JCheckBoxMenuItem;
import javax.swing.JRadioButton;
import javax.swing.JSlider;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import ebflmaennle.Rundschreiben;
import ebflmaennle.berechnung.Charakteristik;

public class Modell implements ActionListener {

    public enum Modus {
        HSB("HSB Farbraum"), LCH("LCH Farbraum"), RETRO("Retro"), ZWEIFARBIG("Zweifarbig"), GRAU("Graustufen");

        private final String beschreibung;

        Modus(final String beschreibung) {
            this.beschreibung = beschreibung;
        }

        @Override
        public String toString() {
            return beschreibung;
        }
    }

    private abstract class Waechter<T> {
        protected abstract void extrahiere(final T event);

        void bearbeite(final T event) {
            extrahiere(event);
            synchronisiere();
            rundschreiben.fireStateChanged();
        }
    }

    private abstract class Aenderungswaechter extends Waechter<ChangeEvent> implements ChangeListener {
        @Override
        public void stateChanged(final ChangeEvent event) {
            bearbeite(event);
        }
    }

    private abstract class Elementwaechter extends Waechter<ItemEvent> implements ItemListener {
        @Override
        public void itemStateChanged(final ItemEvent event) {
            bearbeite(event);
        }
    }

    private class Abstandkonturenwaechter extends Elementwaechter {

        @Override
        protected void extrahiere(final ItemEvent event) {
            final var box = (JRadioButton) event.getSource();
            parameter.abstandkonturen = box.isSelected();
        }

    }

    private abstract class Aktionswaechter implements ActionListener {
        @Override
        public void actionPerformed(final ActionEvent event) {
            extrahiere(event);
            synchronisiere();
            rundschreiben.fireStateChanged();
        }

        protected abstract void extrahiere(ActionEvent event);
    }

    private class Moduswaechter extends Aktionswaechter {

        @Override
        protected void extrahiere(final ActionEvent event) {
            parameter.modus = Modus.valueOf(event.getActionCommand());
        }
    }

    private class Korrekturwaechter extends Aenderungswaechter {
        @Override
        protected void extrahiere(final ChangeEvent event) {
            final var regler = (JSlider) event.getSource();
            parameter.korrektur = (float) Math.exp(1f - regler.getValue() / 50f);
        }
    }

    private class Spektrumwaechter extends Aenderungswaechter {
        @Override
        protected void extrahiere(final ChangeEvent event) {
            final var regler = (JSlider) event.getSource();
            parameter.spektrum = (float) regler.getValue() / (float) regler.getMaximum();
        }
    }

    private class Farbfrequenzwaechter extends Aenderungswaechter {

        private float farbperiode(final float relativstellung) {
            return (float) Math.pow(10, -FREQUENZORDNUNGEN * relativstellung);
        }

        @Override
        protected void extrahiere(final ChangeEvent event) {
            final var regler = (JSlider) event.getSource();
            final var stellung = regler.getValue();
            parameter.farbperiode = farbperiode((float) stellung / (float) regler.getMaximum());
        }
    }

    private class Verschiebungswaechter extends Aenderungswaechter {
        @Override
        protected void extrahiere(final ChangeEvent event) {
            final var regler = (JSlider) event.getSource();
            parameter.verschiebung = (float) regler.getValue() / (float) regler.getMaximum();
        }
    }

    private class Frequenzfaktorwaechter extends Aenderungswaechter {
        @Override
        protected void extrahiere(final ChangeEvent event) {
            final var regler = (JSlider) event.getSource();
            parameter.frequenzfaktor = regler.getValue();
        }
    }

    private class Multiplikatorkonturenwaechter extends Elementwaechter {

        @Override
        protected void extrahiere(final ItemEvent event) {
            final var box = (JRadioButton) event.getSource();
            parameter.multiplikatorkonturen = box.isSelected();
        }

    }

    private class Prinzipienwaechter implements ActionListener {
        @Override
        public void actionPerformed(final ActionEvent ereignis) {
            final var prinzipdarstellungsschalter = (JCheckBoxMenuItem) ereignis.getSource();
            parameter.prinzipdarstellung = prinzipdarstellungsschalter.isSelected();
            synchronisiere();
            // Hier kein Rundschreiben abschicken. Färbung wird durch Neuberechnung angestoßen.
        }
    }

    private class Zentrumhervorhebungswaechter extends Elementwaechter {
        @Override
        protected void extrahiere(final ItemEvent event) {
            final var zentrumhervorhebung = (JCheckBoxMenuItem) event.getSource();
            parameter.zentrumhervorhebung = zentrumhervorhebung.isSelected();
        }
    }

    public static final int FREQUENZORDNUNGEN = 3;
    private static final float ZWEIFARBENSPEKTRUM = 0.25f;

    private final Abstandkonturenwaechter abstandkonturenwaechter = new Abstandkonturenwaechter();
    private final Multiplikatorkonturenwaechter multiplikatorkonturenwaechter = new Multiplikatorkonturenwaechter();
    private final Moduswaechter moduswaechter = new Moduswaechter();
    private final Farbfrequenzwaechter farbfrequenzwaechter = new Farbfrequenzwaechter();
    private final Korrekturwaechter korrekturwaechter = new Korrekturwaechter();
    private final Spektrumwaechter spektrumwaechter = new Spektrumwaechter();
    private final Verschiebungswaechter verschiebungswaechter = new Verschiebungswaechter();
    private final Prinzipienwaechter prinzipienwaechter = new Prinzipienwaechter();
    private final Zentrumhervorhebungswaechter zentrumhervorhebungswaechter = new Zentrumhervorhebungswaechter();
    private final Parameter parameter;
    private final Rundschreiben<Parameter> rundschreiben;
    private Faerber faerber;

    public Modell(final Parameter parameter) {
        this.parameter = parameter;
        rundschreiben = new Rundschreiben<>(parameter);
        synchronisiere();
    }

    @Override
    public void actionPerformed(final ActionEvent event) {
        switch (event.getActionCommand()) {
            case "Ab" -> parameter.versatz--;
            case "Auf" -> parameter.versatz++;
            case "Zentrumhervorhebung" -> parameter.zentrumhervorhebung = !parameter.zentrumhervorhebung;
            case "Retro" -> parameter.modus = Modus.RETRO;
            case "Schnelligkeit", "Kompromiss", "Qualität" -> parameter.modus = Modus.HSB;
        }
        synchronisiere();
        rundschreiben().fireStateChanged();
    }

    public ItemListener abstandkonturenwaechter() {
        return abstandkonturenwaechter;
    }

    public long faerbe(final Charakteristik[] charakteristiken, final int[] farben, final float pixeldistanzquadrat) {
        return faerber.belege(charakteristiken, farben, pixeldistanzquadrat);
    }

    public long faerbe(final Charakteristik[] charakteristiken, final int[] farben, final float pixeldistanzquadrat,
            final int m, final int n) {
        return faerber.belege(charakteristiken, farben, pixeldistanzquadrat, m, n);
    }

    public void faerbe(final Charakteristik[] charakteristiken, final int[] farben, final float pixeldistanzquadrat,
            final int lo, final int m, final int dk, final int dl) {
        faerber.belege(charakteristiken, farben, pixeldistanzquadrat, lo, m, dk, dl);
    }

    public ActionListener moduswaechter() {
        return moduswaechter;
    }

    public ChangeListener korrekturwaechter() {
        return korrekturwaechter;
    }

    public ChangeListener spektrumwaechter() {
        return spektrumwaechter;
    }

    public ChangeListener farbfrequenzwaechter() {
        return farbfrequenzwaechter;
    }

    public ChangeListener verschiebungswaechter() {
        return verschiebungswaechter;
    }

    public ChangeListener frequenzfaktorwaechter() {
        return new Frequenzfaktorwaechter();
    }

    public ItemListener multiplikatorkonturenwaechter() {
        return multiplikatorkonturenwaechter;
    }

    public ActionListener prinzipienwaechter() {
        return prinzipienwaechter;
    }

    public ItemListener zentrumhervorhebungswaechter() {
        return zentrumhervorhebungswaechter;
    }

    public Rundschreiben<Parameter> rundschreiben() {
        return rundschreiben;
    }

    public void synchronisiere() {
        faerber = switch (parameter.modus) {
            case RETRO -> new Retro(parameter.frequenzfaktor, parameter.verschiebung, parameter.versatz,
                    parameter.prinzipdarstellung, parameter.abstandkonturen, parameter.multiplikatorkonturen,
                    parameter.zentrumhervorhebung);
            case GRAU -> new Grau(parameter.farbperiode, parameter.korrektur, parameter.frequenzfaktor,
                    parameter.verschiebung, parameter.versatz, parameter.prinzipdarstellung, parameter.abstandkonturen,
                    parameter.multiplikatorkonturen, parameter.zentrumhervorhebung);
            case HSB -> parameter.spektrum < 1
                    ? new HSB(parameter.farbperiode, parameter.korrektur, parameter.spektrum, parameter.frequenzfaktor,
                            parameter.verschiebung, parameter.versatz, parameter.prinzipdarstellung,
                            parameter.abstandkonturen, parameter.multiplikatorkonturen, parameter.zentrumhervorhebung)
                    : new HSBNeon(parameter.farbperiode, parameter.korrektur, parameter.frequenzfaktor,
                            parameter.verschiebung, parameter.versatz, parameter.prinzipdarstellung,
                            parameter.abstandkonturen, parameter.multiplikatorkonturen, parameter.zentrumhervorhebung);
            case LCH -> new LCH(parameter.farbperiode, parameter.korrektur, parameter.spektrum,
                    parameter.frequenzfaktor, parameter.verschiebung, parameter.versatz, parameter.prinzipdarstellung,
                    parameter.abstandkonturen, parameter.multiplikatorkonturen, parameter.zentrumhervorhebung);
            case ZWEIFARBIG -> new HSB(parameter.farbperiode, parameter.korrektur, ZWEIFARBENSPEKTRUM,
                    parameter.frequenzfaktor, parameter.verschiebung, parameter.versatz, parameter.prinzipdarstellung,
                    parameter.abstandkonturen, parameter.multiplikatorkonturen, parameter.zentrumhervorhebung);
            default -> null;
        };
    }

}
