/******************************************************************************
 ** $Id: LCH.java 3393 2024-07-14 12:39:08Z wmh $
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package ebflmaennle.farbe;

class LCH extends Faerber {

    // private static final float EPSILON = 216f / 24389f;
    // private static final float KAPPA = 243.89f / 27f;
    private static final float D65_REF_WHITE_X = 0.95047f;
    private static final float D65_REF_WHITE_Y = 1f;
    private static final float D65_REF_WHITE_Z = 1.08883f;
    private static final float U0 = 4 * D65_REF_WHITE_X
            / (D65_REF_WHITE_X + 15 * D65_REF_WHITE_Y + 3 * D65_REF_WHITE_Z);
    private static final float V0 = 9 * D65_REF_WHITE_Y
            / (D65_REF_WHITE_X + 15 * D65_REF_WHITE_Y + 3 * D65_REF_WHITE_Z);
    private static float[][] M_INV = {
        { 3.2404542f, -1.5371385f, -0.4985314f },
        { -0.9692660f, 1.8760108f, 0.0415560f },
        { 0.0556434f, -0.2040259f, 1.0572252f } };

    private final boolean gespreizt;
    private final float farbfrequenzfaktor;
    private final float spektrum;
    private final float verschiebung;

    LCH(final float farbperiode, final float korrektur, final float spektrum, final float frequenzfaktor,
            final float verschiebung, final int versatz, final boolean prinzip, final boolean abstandkonturen,
            final boolean multiplikatorkonturen, final boolean zentrumhervorhebung) {
        super(frequenzfaktor, verschiebung, versatz, prinzip, abstandkonturen, multiplikatorkonturen,
                zentrumhervorhebung, ZENTRUMFARBE);
        this.gespreizt = farbperiode < 1;
        this.farbfrequenzfaktor = farbperiode / FARBPERIODE;
        this.spektrum = spektrum;
        this.verschiebung = verschiebung;
        final var potenz1 = (korrektur + 0.5) / 2.4;
        final var potenz2 = korrektur + 0.5;
        nacharbeit = original -> (Math.min(1,
                (float) (original > 0.004f ? Math.pow(original, potenz1) : Math.pow(original * 12f, potenz2))));
    }

    /** LCH_uv -> XYZ -> sRGB nach http://www.brucelindbloom.com */
    private int rgb(final float fluchtfarbe, final float helligkeit) {
        final var chroma = 1.5f;
        final var tonwinkel = ((float) (2 * Math.PI)) * (fluchtfarbe - (float) Math.floor(fluchtfarbe));

        final var u = chroma * (float) Math.cos(tonwinkel);
        final var v = chroma * (float) Math.sin(tonwinkel);

        final var h = (helligkeit + 0.16f) / 1.16f;
        final var y = h * h * h;

        final var a = (52f * helligkeit / (u + 13f * U0 * helligkeit) - 1f) / 3f;
        final var b = -5f * y;
        final var c = -1f / 3f;
        final var d = y * (39f * helligkeit / (v + 13f * V0 * helligkeit) - 5f);

        final var x = (d - b) / (a - c);
        final var z = a * x + b;

        final var rot = M_INV[0][0] * x + M_INV[0][1] * y + M_INV[0][2] * z;
        final var gruen = M_INV[1][0] * x + M_INV[1][1] * y + M_INV[1][2] * z;
        final var blau = M_INV[2][0] * x + M_INV[2][1] * y + M_INV[2][2] * z;

        return allgemeinkodierung(rot, gruen, blau);
    }

    private int rgb(final float fluchtfarbe) {
        return rgb(fluchtfarbe, 1);
    }

    @Override
    protected float fluchtfarbe(final float nu) {
        var fluchtfarbe = nu;
        if (gespreizt) {
            fluchtfarbe = farbfrequenzfaktor * fluchtfarbe;
        } else {
            fluchtfarbe = Math.round(fluchtfarbe) / FARBPERIODE;
        }
        fluchtfarbe -= Math.floor(fluchtfarbe);
        fluchtfarbe = spektrum * fluchtfarbe;
        fluchtfarbe += verschiebung;
        return fluchtfarbe;
    }

    @Override
    protected int potentialfarbe(final float fluchtfarbe, final float rhoq) {
        if (rhoq < RHOQ) {
            return rgb(fluchtfarbe, (1 - ABSTANDSDETAIL) + ABSTANDSDETAIL * r(rhoq));
        }
        return rgb(fluchtfarbe);
    }

}
