/******************************************************************************
 ** $Id: HSB.java 3393 2024-07-14 12:39:08Z wmh $
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package ebflmaennle.farbe;

class HSB extends HSBNeon {

    private final double s;
    private final double sm;
    private final double u;
    private final double v;

    /** Dreiecksfunktion der Periode 1 mit Wert 0 an Stelle 0. */
    private double dreieck0(final float fluchtfarbe) {
        return sm * Math.abs(fluchtfarbe + 0.25 - Math.floor(fluchtfarbe + 0.75)) - s;
    }

    /** Dreiecksfunktion der Periode 1 mit Wert 1 an Stelle 0. */
    private double dreieck1(final float fluchtfarbe) {
        return sm * Math.abs(fluchtfarbe - 0.5 - Math.floor(fluchtfarbe)) - s;
    }

    private double sigmoid(final double m) {
        return 0.25 - 0.25 * m / (1 + Math.abs(m));
    }

    HSB(final float farbperiode, final float korrektur, final float spektrum, final float frequenzfaktor,
            final float verschiebung, final int versatz, final boolean prinzip, final boolean abstandkonturen,
            final boolean multiplikatorkonturen, final boolean zentrumhervorhebung) {
        super(farbperiode, korrektur, frequenzfaktor, verschiebung, versatz, prinzip, abstandkonturen,
                multiplikatorkonturen, zentrumhervorhebung, ZENTRUMFARBE);
        s = spektrum;
        sm = 4 * spektrum;
        final var phi = 2 * Math.PI * verschiebung;
        u = (1 - spektrum) * Math.cos(phi);
        v = (1 - spektrum) * Math.sin(phi);
    }

    @Override
    protected float fluchtfarbe(final float nu) {
        var fluchtfarbe = nu;
        if (gespreizt) {
            fluchtfarbe = relativfarbperiode * fluchtfarbe;
        } else {
            fluchtfarbe = Math.round(fluchtfarbe) / FARBPERIODE;
        }
        fluchtfarbe += verschiebung;
        return fluchtfarbe;
    }

    @Override
    protected int potentialfarbe(final float fluchtfarbe, final float rhoq) {
        final var x = u + dreieck1(fluchtfarbe);
        final var y = v + dreieck0(fluchtfarbe);
        final var r = Math.min(1, Math.abs(x) + Math.abs(y));
        final double chi;
        if (y < 0) {
            chi = sigmoid(x / y) + 0.5;
        } else {
            chi = sigmoid(x / y);
        }
        if (rhoq < RHOQ) {
            return rgb((float) chi, (1 - ABSTANDSDETAIL) + ABSTANDSDETAIL * r(rhoq), ABSTANDSSAETTIGUNG);
        }
        return rgb((float) chi, 1, (float) r);
    }

}
