/******************************************************************************
 ** $Id: Faerber.java 3631 2024-12-24 16:34:32Z wmh $
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package ebflmaennle.farbe;

import java.awt.Color;
import java.util.Arrays;

import ebflmaennle.berechnung.Charakteristik;

abstract class Faerber {

    private static final int PRINZIPPALETTENUMFANG = 37;
    private static final int AUSSENBLOCKFARBE = Color.LIGHT_GRAY.getRGB();
    private static final int INNENBLOCKFARBE = Color.GRAY.getRGB();
    private static final int MANDELBROTMENGENFARBE = Color.BLACK.getRGB();
    private static final Color[] FARBEN = {
        Color.RED,
        Color.ORANGE,
        Color.YELLOW,
        Color.GREEN,
        new Color(0, 240, 0),
        new Color(64, 255, 160),
        Color.CYAN,
        new Color(0, 160, 224),
        Color.BLUE,
        new Color(128, 0, 255),
        Color.MAGENTA,
        new Color(255, 0, 192) };

    protected interface Nacharbeit {
        float wert(float original);
    }

    protected static final int ZENTRUMFARBE = Color.WHITE.getRGB();
    protected static final int ZENTRUMFARBESW = Color.CYAN.getRGB();

    protected static final float ABSTANDKONTRAST = 0.125f;
    protected static final float ABSTANDSSAETTIGUNG = 0.75f;
    protected static final float ABSTANDSSAETTIGUNGGRAUSTUFEN = 0.75f;
    protected static final float ABSTANDSDETAIL = 0.875f;
    protected static final float MULTIPLIKATORRINGSAETTIGUNG = 0.2f;
    protected static final float ABSTANDRINGHUE = 40f / 256f;
    protected static final float ABSTANDRINGSAETTIGUNG = 160f / 256f;
    protected static final float FARBPERIODE = 12;
    protected static final int[] LICHTKONTRASTFARBEN = new int[FARBEN.length];
    {
        for (var i = 0; i < FARBEN.length; i++) {
            LICHTKONTRASTFARBEN[i] = FARBEN[i].getRGB();
        }
    }
    protected static final int[] HELLKONTRASTFARBEN = new int[FARBEN.length];
    {
        for (var i = 0; i < FARBEN.length; i++) {
            HELLKONTRASTFARBEN[i] = FARBEN[i].getRGB();
        }
    }
    protected static final int[] DUNKELKONTRASTFARBEN = new int[FARBEN.length];
    {
        for (var i = 0; i < FARBEN.length; i++) {
            DUNKELKONTRASTFARBEN[i] = FARBEN[i].darker().darker().getRGB();
        }
    }
    protected static final int SONDERHELLKONTRASTFARBE = Color.GREEN.brighter().getRGB();
    protected static final int SONDERDUNKELKONTRASTFARBE = Color.GREEN.darker().getRGB();

    protected static final int RANDFARBE = spezialkodierung(1f / 5f, 1f / 5f, 1f / 5f);
    protected static final float RHOQ = 0.95f;

    private static int hash(final int periode, final int versatz) {
        return ((663608943 * (periode * (versatz % PRINZIPPALETTENUMFANG + 1)))) >>> 29;
    }

    protected static final int spezialkodierung(final float r, final float g, final float b) {
        return (Math.round(255 * r) << 16) | (Math.round(255 * g) << 8) | (Math.round(255 * b) << 0);
    }

    protected final int allgemeinkodierung(final float r, final float g, final float b) {
        return spezialkodierung(nacharbeit.wert(r), nacharbeit.wert(g), nacharbeit.wert(b));
    }

    protected static final float r(final float rhoq) {
        return (float) Math.pow(rhoq, ABSTANDKONTRAST);
    }

    private final float frequenzfaktor;
    private final float verschiebung;
    private final boolean prinzip;
    private final int versatz;
    private final boolean abstandkonturen;
    private final boolean multiplikatorkonturen;
    private final boolean zentrumhervorhebung;
    private final int zentrumfarbe;
    protected Nacharbeit nacharbeit;

    Faerber(final float frequenzfaktor, final float verschiebung, final int versatz, final boolean prinzip,
            final boolean abstandkonturen, final boolean multiplikatorkonturen, final boolean zentrumhervorhebung,
            final int zentrumfarbe) {
        this.frequenzfaktor = frequenzfaktor;
        this.verschiebung = verschiebung;
        this.versatz = versatz;
        this.prinzip = prinzip;
        this.abstandkonturen = abstandkonturen;
        this.multiplikatorkonturen = multiplikatorkonturen;
        this.zentrumhervorhebung = zentrumhervorhebung;
        this.zentrumfarbe = zentrumfarbe;
        nacharbeit = original -> original;
    }

    private int abstandfarbe(final float pixeldistanzquadrat, final float lq) {
        return Color.HSBtoRGB(ABSTANDRINGHUE, ABSTANDRINGSAETTIGUNG,
                (float) Math.abs(Math.sin(Math.PI / 500 * Math.sqrt(-lq / pixeldistanzquadrat) * frequenzfaktor)));
    }

    private int dunkelkontrastfarbe(final int periode) {
        return DUNKELKONTRASTFARBEN[hash(periode, versatz)];
    }

    private int farbe(final Charakteristik[] charakteristiken, final int p, final float pixeldistanzquadrat) {
        final var lq = charakteristiken[p].lq;
        final var nu = charakteristiken[p].nu;
        final var mulq = charakteristiken[p].mulq;
        if (Charakteristik.offen(nu)) {
            assert prinzip;
            if (lq >= 0) {
                return AUSSENBLOCKFARBE;
            }
            return INNENBLOCKFARBE;
        }
        if (nu <= 0) {
            if (Charakteristik.zentral(mulq)) {
                if (zentrumhervorhebung) {
                    return zentrumfarbe;
                }
                return RANDFARBE;
            }
            if (prinzip) {
                final var periode = (int) -nu;
                if (Charakteristik.unbestimmt(nu)) {
                    return SONDERHELLKONTRASTFARBE;
                }
                if (lq >= -pixeldistanzquadrat) {
                    return hellkontrastfarbe(periode);
                }
                if (Charakteristik.unbestimmt(lq)) {
                    return AUSSENBLOCKFARBE;
                }
                return dunkelkontrastfarbe(periode);
            }
            if (multiplikatorkonturen) {
                return multiplikatorfarbe(mulq);
            }
            if (abstandkonturen) {
                return abstandfarbe(pixeldistanzquadrat, lq);
            }
            return MANDELBROTMENGENFARBE;
        }
        final float rhoq;
        if (Charakteristik.unbestimmt(lq)) {
            if (prinzip) {
                return INNENBLOCKFARBE;
            }
            rhoq = Float.POSITIVE_INFINITY;
        } else if (mulq == Charakteristik.MULQ_RANDNAH || prinzip) {
            rhoq = lq / pixeldistanzquadrat;
        } else {
            rhoq = Float.POSITIVE_INFINITY;
        }
        return potentialfarbe(fluchtfarbe(nu), rhoq);
    }

    private int hellkontrastfarbe(final int periode) {
        return HELLKONTRASTFARBEN[hash(periode, versatz)];
    }

    private int multiplikatorfarbe(final float mulq) {
        return Color.HSBtoRGB(verschiebung + mulq, MULTIPLIKATORRINGSAETTIGUNG,
                (float) Math.abs(Math.sin(Math.PI * Math.sqrt(mulq) * frequenzfaktor)));
    }

    protected abstract float fluchtfarbe(final float nu);

    protected abstract int potentialfarbe(final float fluchtfarbe, final float rhoq);

    final long belege(final Charakteristik[] charakteristiken, final int[] farben, final float pixeldistanzquadrat) {
        final var start = System.nanoTime();
        Arrays.parallelSetAll(farben, p -> farbe(charakteristiken, p, pixeldistanzquadrat));
        return System.nanoTime() - start;
    }

    final long belege(final Charakteristik[] charakteristiken, final int[] farben, final float pixeldistanzquadrat,
            final int m, final int n) {
        final var start = System.nanoTime();
        int p;
        for (p = m - 1; p < m * n; p += m) {
            farben[p] = farbe(charakteristiken, p, pixeldistanzquadrat);
        }
        p -= m;
        for (var i = 1; i < m; i++) {
            p--;
            farben[p] = farbe(charakteristiken, p, pixeldistanzquadrat);
        }
        return System.nanoTime() - start;
    }

    final void belege(final Charakteristik[] charakteristiken, final int[] farben, final float pixeldistanzquadrat,
            final int lo, final int m, final int dk, final int dl) {
        for (var p = lo; p < lo + dl * m; p += m) {
            for (var q = p; q < p + dk; q++) {
                farben[q] = farbe(charakteristiken, q, pixeldistanzquadrat);
            }
        }
    }
}
