/******************************************************************************
 ** $Id: Z.java 3594 2024-09-07 16:54:16Z wmh $
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package ebflmaennle.berechnung;

final class Z {
    private final Z original;
    private final double delta;

    private double u;
    private double v;
    int periode;

    double a;
    double b;

    double x;
    double y;
    double yq;
    double xq;
    double absq;

    Z(final double delta, final Z original) {
        this.original = original;
        this.delta = delta;
    }

    Z(final double delta) {
        this(delta, null);
    }

    Z() {
        this(0);
    }

    /** Setzt Startwert z = x+iy und |z|^2. */
    private void starte0(final double x, final double y) {
        this.x = x;
        this.y = y;
        xq = x * x;
        yq = y * y;
        absq = xq + yq;
    }

    void parametrisiere(final double a, final double b) {
        this.a = a;
        this.b = b;
    }

    void parametrisiere() {
        parametrisiere(original.a, original.b);
    }

    void starte0() {
        x = 0;
        y = 0;
        xq = 0;
        yq = 0;
        absq = 0;
        periode = 0;
    }

    /** Setzt Startwert z = (x+iy)²+(x+iy) und |z|^2. */
    void starte1(final double x, final double y) {
        starte0(x * x - y * y + a, 2 * x * y + b);
    }

    /** Berechnet den ersten iterierten Wert z[1] = c = a+b*i und |z[1]|². */
    void starte1() {
        starte0(a, b);
        periode = 1;
    }

    /** Berechnet den zweiten iterierten Wert z[2] = z[1]^2 + c = c^2+c, mit z[1] = c = a+b*i. */
    void starte2() {
        starte0(a * a - b * b + a, 2 * a * b + b);
        periode = 2;
    }

    /** Bereite für Periodenprüfung vor. */
    void merke() {
        periode = 0;
        u = x;
        v = y;
    }

    void iteriere() {
        starte0(xq - yq + a, 2 * x * y + b);
        periode++;
    }

    double abstand(final double x, final double y) {
        return Math.abs(this.x - x) + Math.abs(this.y - y);
    }

    boolean trivial() {
        final var u = a - 0.25;
        final var q = u * u + yq;
        return ((absq + 2 * a <= -0.93751) || (q * (q + u) <= yq / 4));
    }

    boolean periodisch() {
        return periode * abstand(u, v) < delta;
    }

    boolean innenkandidat() {
        return absq <= Charakterisierung.FLUCHTRADIUSQUADRATMINIMUM; // Gleichheit wegen Sonderfall c = a+ib = -2
    }

    boolean gefluechtet() {
        return absq >= Charakterisierung.FLUCHTRADIUSQUADRAT;
    }

    double laenge() {
        return Math.abs(x) + Math.abs(y);
    }

    double logabsq() {
        return Math.log(absq);
    }

}