/******************************************************************************
 ** $Id: Wukleusfinder.java 3594 2024-09-07 16:54:16Z wmh $
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package ebflmaennle.berechnung;

abstract class Wukleusfinder {

    private static final class Leer extends Wukleusfinder {
        Leer(final W w, final Abstandschaetzer abstandschaetzer) {
            super(w, abstandschaetzer);
        }

        @Override
        protected void initialisiere() {
        }

        @Override
        protected void finde(final int domper) {
        }
    }

    private static final class Voll extends Wukleusfinder {

        private static final int ITERMAX = 24;
        private final Z t;
        private final Dz dz;

        Voll(final W w, final Abstandschaetzer abstandschaetzer) {
            super(w, abstandschaetzer);
            t = w.t;
            dz = w.dz;
        }

        @Override
        protected void initialisiere() {
            w.initialisiere();
        }

        @Override
        /**
         * @formatter:off
         *
         * Funktion F°(p)(z,c), Ableitung d(F°(p))/dz(z,c)
         *
         * F°(m+1)(z,c) = F(F°(m)(z,c),c), F°(0)(z,c) = z; F(z,c) = F°(1)(z,c) = z²+c.
         *
         * Gleichung F°(p)(w,c) - F°(0)(w,c) = F°(p)(w,c) - w = 0 für w wird mit Newtonverfahren gelöst.
         * Die Lösung hängt von c ab: w = w(c).
         *
         * w[n+1] = w[n] - ( F°(p)(w[n],c) - w[n] ) / ( d(F°(p)/dz)(w[n],c) - 1 )
         *
         * d(F°(m+1))/dz = 2F°(m)dF(°(m))/dz, dF(°(0))/dz = 1;
         * d²(F°(m+1))/dz² = 2F°(m)d²(F°(m))/dz² + 2(dF(°(m))/dz)², dF(°(0))/dz² = 0
         *
         * t'[m+1] = 2t[m]t'[m] = 2(tx+i*ty)(t'x+i*t'y) =
         * 2(tx*t'x-ty*t'y)+2i(ty*t'x+tx*t'y) t''[m+1] = 2t[m]t''[m]+2(t'[m])² = 2(tx+i*ty)(t''x+i*t''y)+2(t'x+i*t'y)² =
         * 2(tx*t''x-ty*t''y)+2i(tx*t''y+ty*t''x)+2(t'x²-t'y²)+2i*t'x*t'y mul2 = dtx^2 + dty^2, dtx1 = dtx - 1 m1q =
         * dtx1^2 + dty^2 = dtx^2 + dty^2 - 2dtx + 1 = mul2 - 2(dtx - 1) - 1 = mul2 - 2dtx1 - 1 mul2 <= 1 <=> m1q +
         * 2dtx1 + 1 <= 1 <=> m1q + 2dtx1 <= 0
         *
         * @formatter:on
         */
        protected void finde(final int domper) {
            assert w.z.innenkandidat();
            w.beginne();
            var i = 0;
            do {
                w.starte1(); // dz[1], t[1]
                for (var j = 2; j <= domper; j++) {
                    dz.iteriere(); // dz[j]
                    t.iteriere(); // t[j]
                }
                w.iteriere();
                i++;
            } while (i < ITERMAX && !w.konvergent());
        }

    }

    static Wukleusfinder erzeuge(final W w, final Abstandschaetzer abstandschaetzer, final boolean multiplikatoren) {
        if (multiplikatoren) {
            return new Voll(w, abstandschaetzer);
        }
        return new Leer(w, abstandschaetzer);
    }

    protected final W w;
    private final Abstandschaetzer abstandschaetzer;

    Wukleusfinder(final W w, final Abstandschaetzer abstandschaetzer) {
        this.w = w;
        this.abstandschaetzer = abstandschaetzer;
    }

    abstract void initialisiere();

    abstract void finde(final int domper);

    /** Gibt an, ob Punkt Kandidat für die Mandelbrotmenge ist. */
    boolean enthalten(final Charakteristik charakteristik, final int domper) {
        finde(domper);
        final var enthalten = w.hyperbolisch();
        if (enthalten) {
            charakteristik.setze(abstandschaetzer.kern(domper), w.mulq());
        }
        return enthalten;
    }

}
