/******************************************************************************
 ** $Id: W.java 3594 2024-09-07 16:54:16Z wmh $
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package ebflmaennle.berechnung;

/** Wukleus. */
final class W {

    private final double epsilon;

    private double dx1;
    private double m1q;
    private boolean konvergent;

    private double u; // Näherung für den Realteil des Wukleus, im Newtonverfahren berechnet.
    private double v; // Näherung für den Imaginärteil des Wukleus, im Newtonverfahren berechnet.

    final Z z;
    final Z t;
    final Dz dz;

    W(final Z z, final Z t, final Dz dz, final double epsilon) {
        this.z = z;
        this.t = t;
        this.dz = dz;
        this.epsilon = epsilon;
    }

    W(final Z t, final Dz dz, final double epsilon) {
        this(t, new Z(0), dz, epsilon);
    }

    void initialisiere() {
        konvergent = false;
    }

    void beginne() {
        t.parametrisiere();
        u = z.x;
        v = z.y;
    }

    /** Überträgt z auf t und lässt t[1] sowie dz[1] berechnen. */
    void uebertrage() {
        t.parametrisiere();
        dz.starte1(); // dz[1]
        t.starte1(u, v); // t[1]
    }

    /** Lässt dz[1] und dz[1] berechnen. */
    void starte1() {
        t.parametrisiere();
        dz.starte1(u, v); // dz[1]
        t.starte1(u, v); // t[1]
    }

    /** Berechne nächsten Näherungswert mit dem Newton-Verfahren. */
    void iteriere() {
        final var duh = t.x - u;
        final var dvh = t.y - v;
        dx1 = dz.x - 1;
        m1q = dx1 * dx1 + dz.y * dz.y;
        final var du = (dx1 * duh + dz.y * dvh) / m1q;
        final var dv = (dx1 * dvh - dz.y * duh) / m1q;
        u -= du;
        v -= dv;
        konvergent = Math.abs(du) + Math.abs(dv) < epsilon;
    }

    boolean divergent() {
        return t.abstand(u, v) >= epsilon;
    }

    boolean konvergent() {
        return konvergent;
    }

    // Diskriminante lambda^2 - 1 = m1q + 2*dx1, mit Multiplikator lambda
    boolean hyperbolisch() {
        return konvergent && m1q <= -2 * dx1;
    }

    float mulq() {
        return (float) dz.absq;
    }

}
