/******************************************************************************
 ** $Id: Nukleusfinder.java 3594 2024-09-07 16:54:16Z wmh $
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package ebflmaennle.berechnung;

import ebflmaennle.berechnung.Abstandschaetzer.Groessenschaetzer;

// http://www.scideveloper.com/Fractals/Named.txt
public final class Nukleusfinder {

    private static final int ITERMAX = 24;

    private final Z t;
    private final Dc dc;
    private final N n;
    private final Groessenschaetzer groessenschaetzer;

    static Nukleusfinder erzeuge(final N n, final Groessenschaetzer groessenschaetzer) {
        return new Nukleusfinder(n, groessenschaetzer);
    }

    private Nukleusfinder(final N n, final Groessenschaetzer groessenschaetzer) {
        this.t = n.t;
        this.dc = n.dc;
        this.n = n;
        this.groessenschaetzer = groessenschaetzer;
    }

    /**
     * @formatter:off
     *
     * Funktion F°(p)(z,c), Ableitung d(F°(p))/dc(z,c)
     *
     * F°(m+1)(z,c) = F(F°(m)(z,c),c), F°(0)(z,c) = z; F(z,c) = F°(1)(z,c) = z²+c.
     *
     * Gleichung F°(p)(0,c) - F°(0)(0,c) = F°(p)(0,c) - 0 = 0 für z wird mit Newtonverfahren gelöst.
     *
     * z[n+1] = z[n] - F°(p)(0,z[n])/d(F°(p)/dc)(0,z[n])
     *
     * z[0] = 0, z[1] = c, z[2] = c² + c
     *
     * dc := (dF°(p)/dc)(0,c);
     * dc[n+1] = 2z[n]dc[n] + 1, dc[0] = 0, dc[1] = 1, dc[2] = 2c + 1
     *
     * @formatter:on
     */
    public N finde(final double a, final double b, final int periode) {
        n.beginne(a, b);
        var i = 0;
        do {
            n.starte();
            for (var j = 3; j <= periode; j++) {
                dc.iteriere();
                t.iteriere();
            }
            n.iteriere();
            i++;
        } while (i < ITERMAX && n.divergent());
        n.lq = groessenschaetzer.kern(periode).lq();
        return n;
    }

}
