/******************************************************************************
 ** $Id: Linienrechner.java 3621 2024-12-24 12:10:35Z wmh $
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package ebflmaennle.berechnung;

import ebflmaennle.berechnung.Algorithmus.Schritte;

final class Linienrechner {

    private interface Komplettierung {
        int erstelle(final int lo, final int ro, final int lu);
    }

    private interface Rechner extends Linie {
        int m();
    }

    public interface Linie {
        int bestimme(final int p1, final int p2);
    }

    private abstract static class Linienbasis implements Rechner {
        private final int m;
        protected final Charakterisierung charakterisierung;
        protected final Charakteristik[] charakteristiken;

        protected boolean entfernt(final Charakteristik charakteristik, final float skalenquadrat) {
            return Math.abs(charakteristik.lq) > skalenquadrat;
        }

        public Linienbasis(final Charakterisierung charakterisierung, final Charakteristik[] charakteristiken,
                final int m) {
            this.charakterisierung = charakterisierung;
            this.charakteristiken = charakteristiken;
            this.m = m;
        }

        @Override
        public int m() {
            return m;
        }

        protected int schrittweite(final Charakteristik charakteristik, final float skala) {
            if (Float.isInfinite(charakteristik.lq)) {
                return 1;
            }
            return (int) (Math.sqrt(Math.abs(charakteristik.lq)) / skala);
        }
    }

    private static class Spaltenabtastung extends Linienbasis {

        Spaltenabtastung(final Charakterisierung charakterisierung, final Charakteristik[] charakteristiken,
                final int m) {
            super(charakterisierung, charakteristiken, m);
        }

        @Override
        public int bestimme(final int o, final int u) {
            var iterationsmaximum = 0;
            final var m = m();
            for (var p = o + m; p < u; p += m) {
                iterationsmaximum = charakterisierung.bestimme(p, iterationsmaximum);
            }
            return iterationsmaximum;
        }
    }

    private static class Spaltenabschnitt extends Spaltenabtastung {
        private final int sy;

        Spaltenabschnitt(final Charakterisierung charakterisierung, final Charakteristik[] charakteristiken,
                final int m, final int sy) {
            super(charakterisierung, charakteristiken, m);
            this.sy = sy;
        }

        @Override
        public int bestimme(final int o, final int u) {
            var p = o;
            int q;
            final var m = m();
            var grenze = spaltenabschnittgrenze(p, u);
            for (q = p + m; q < grenze; q += m) {
                charakteristiken[p].kopiere(charakteristiken[q]);
            }
            var iterationsmaximum = 0;
            do {
                iterationsmaximum = charakterisierung.bestimme(q, iterationsmaximum);
                grenze = spaltenabschnittgrenze(q, u);
                for (p = q + m; p < grenze; p += m) {
                    charakteristiken[q].kopiere(charakteristiken[p]);
                }
                q = p;
            } while (p < u);
            return iterationsmaximum;
        }

        private int spaltenschrittgrenze(final int p, final int unten, final int schritt) {
            return Integer.min(p + schritt, unten);
        }

        private int spaltenabschnittgrenze(final int p, final int unten) {
            return spaltenschrittgrenze(p, unten, sy);
        }
    }

    private static class Spaltenschritt extends Spaltenabtastung {
        private final float spalteneffektivskala;
        private final float skalenquadrat;

        Spaltenschritt(final Charakterisierung charakterisierung, final Charakteristik[] charakteristiken, final int m,
                final float zeilenskala) {
            super(charakterisierung, charakteristiken, m);
            spalteneffektivskala = zeilenskala / m;
            skalenquadrat = zeilenskala * zeilenskala;
        }

        @Override
        public int bestimme(final int o, final int u) {
            var p = o;
            int q;
            final var m = m();
            var charakteristik = charakteristiken[p];
            var nu = charakteristik.nu;
            if (entfernt(charakteristik, skalenquadrat)) {
                final var grenze = spaltenschrittgrenze(p, u, spaltenschrittweite(charakteristik));
                for (q = o + m; q < grenze; q += m) {
                    charakteristiken[q].setze(nu);
                }
            } else {
                q = o + m;
            }
            var iterationsmaximum = 0;
            while (q < u) {
                iterationsmaximum = charakterisierung.bestimme(q, iterationsmaximum);
                charakteristik = charakteristiken[q];
                nu = charakteristik.nu;
                if (entfernt(charakteristik, skalenquadrat)) {
                    final var grenze = spaltenschrittgrenze(q, u, spaltenschrittweite(charakteristik));
                    for (p = q + m; p < grenze; p += m) {
                        charakteristiken[p].setze(nu);
                    }
                    q = p;
                } else {
                    q += m;
                }
            }
            return iterationsmaximum;
        }

        private int spaltenschrittweite(final Charakteristik charakteristik) {
            return schrittweite(charakteristik, spalteneffektivskala);
        }

        private int spaltenschrittgrenze(final int p, final int unten, final int schritt) {
            return Integer.min(p + schritt, unten);
        }

    }

    private static class Zeilenabtastung extends Linienbasis {

        Zeilenabtastung(final Charakterisierung charakterisierung, final Charakteristik[] charakteristiken,
                final int m) {
            super(charakterisierung, charakteristiken, m);
        }

        @Override
        public int bestimme(final int l, final int r) {
            var iterationsmaximum = 0;
            for (var p = l + 1; p < r; p++) {
                iterationsmaximum = charakterisierung.bestimme(p, iterationsmaximum);
            }
            return iterationsmaximum;
        }
    }

    private static class Zeilenabschnitt extends Zeilenabtastung {
        private final int sx;

        Zeilenabschnitt(final Charakterisierung charakterisierung, final Charakteristik[] charakteristiken, final int m,
                final int sx) {
            super(charakterisierung, charakteristiken, m);
            this.sx = sx;
        }

        @Override
        public int bestimme(final int l, final int r) {
            var p = l;
            int q;
            var grenze = zeilenabschnittgrenze(p, r);
            for (q = l + 1; q < grenze; q++) {
                charakteristiken[p].kopiere(charakteristiken[q]);
            }
            var iterationsmaximum = 0;
            do {
                iterationsmaximum = charakterisierung.bestimme(q, iterationsmaximum);
                grenze = zeilenabschnittgrenze(q, r);
                for (p = q + 1; p < grenze; p++) {
                    charakteristiken[q].kopiere(charakteristiken[p]);
                }
                q = p;
            } while (p < r);
            return iterationsmaximum;
        }

        private int zeilenschrittgrenze(final int p, final int rechts, final int schritt) {
            return Integer.min(p + schritt, rechts);
        }

        private int zeilenabschnittgrenze(final int p, final int rechts) {
            return zeilenschrittgrenze(p, rechts, sx);
        }
    }

    private static class Zeilenschritt extends Zeilenabtastung {
        private final float zeilenskala;
        private final float skalenquadrat;

        Zeilenschritt(final Charakterisierung charakterisierung, final Charakteristik[] charakteristiken, final int m,
                final float zeilenskala) {
            super(charakterisierung, charakteristiken, m);
            this.zeilenskala = zeilenskala;
            skalenquadrat = zeilenskala * zeilenskala;
        }

        @Override
        public int bestimme(final int l, final int r) {
            var p = l;
            int q;
            var charakteristik = charakteristiken[p];
            var nu = charakteristik.nu;
            if (entfernt(charakteristik, skalenquadrat)) {
                final var grenze = zeilenschrittgrenze(p, r, zeilenschrittweite(charakteristik));
                for (q = l + 1; q < grenze; q++) {
                    charakteristiken[q].setze(nu);
                }
            } else {
                q = l + 1;
            }
            var iterationsmaximum = 0;
            while (q < r) {
                iterationsmaximum = charakterisierung.bestimme(q, iterationsmaximum);
                charakteristik = charakteristiken[q];
                nu = charakteristik.nu;
                if (entfernt(charakteristik, skalenquadrat)) {
                    final var grenze = zeilenschrittgrenze(q, r, zeilenschrittweite(charakteristik));
                    for (p = q + 1; p < grenze; p++) {
                        charakteristiken[p].setze(nu);
                    }
                    q = p;
                } else {
                    q++;
                }
            }
            return iterationsmaximum;
        }

        private int zeilenschrittweite(final Charakteristik charakteristik) {
            return schrittweite(charakteristik, zeilenskala);
        }

        protected int zeilenschrittgrenze(final int p, final int rechts, final int schritt) {
            return Integer.min(p + schritt, rechts);
        }
    }

    private final Spaltenabtastung spaltenabtastung;
    private final Zeilenabtastung zeilenabtastung;
    private final Spaltenabschnitt spaltenabschnitt;
    private final Zeilenabschnitt zeilenabschnitt;
    private final Spaltenschritt spaltenschritt;
    private final Zeilenschritt zeilenschritt;
    private final boolean schrittschaetzung;

    private Rechner spaltenrechner;
    private volatile Rechner zeilenrechner;

    private Komplettierung komplettierung;

    private final Komplettierung horizontalkomplettierung = (lo, ro, lu) -> {
        var iterationsmaximum = 0;
        for (int p1 = lo + 1, p2 = lu + 1; p1 < ro; p1++, p2++) {
            iterationsmaximum = Math.max(spaltenrechner.bestimme(p1, p2), iterationsmaximum);
        }
        return iterationsmaximum;
    };
    private final Komplettierung vertikalkomplettierung = (lo, ro, lu) -> {
        var iterationsmaximum = 0;
        final var m = zeilenrechner.m();
        for (int p1 = lo + m, p2 = ro + m; p1 <= lu - m + 1; p1 += m, p2 += m) {
            iterationsmaximum = Math.max(zeilenrechner.bestimme(p1, p2), iterationsmaximum);
        }
        return iterationsmaximum;
    };

    Linienrechner(final Charakteristik[] charakteristiken, final Charakterisierung charakterisierung,
            final Konfiguration konfiguration, final int m, final Schritte schritte) {
        schrittschaetzung = konfiguration.schrittschaetzung;
        spaltenabtastung = new Spaltenabtastung(charakterisierung, charakteristiken, m);
        zeilenabtastung = new Zeilenabtastung(charakterisierung, charakteristiken, m);
        spaltenabschnitt = new Spaltenabschnitt(charakterisierung, charakteristiken, m, schritte.sy());
        zeilenabschnitt = new Zeilenabschnitt(charakterisierung, charakteristiken, m, schritte.sx());
        final var zeilenskala = konfiguration.pxs * 4;
        spaltenschritt = new Spaltenschritt(charakterisierung, charakteristiken, m, zeilenskala);
        zeilenschritt = new Zeilenschritt(charakterisierung, charakteristiken, m, zeilenskala);
        waehle(konfiguration.beschleunigt && konfiguration.bildfrequenz);

    }

    void kippe(final boolean vertikal) {
        if (vertikal) {
            komplettierung = vertikalkomplettierung;
        } else {
            komplettierung = horizontalkomplettierung;
        }
    }

    int komplettiere(final int lo, final int ro, final int lu) {
        return komplettierung.erstelle(lo, ro, lu);
    }

    /**
     * @param abschnittsweise
     *            Gibt an, ob der Linien abschnittsweise berechnet werden sollen.
     */
    void waehle(final boolean abschnittsweise) {
        if (schrittschaetzung) {
            spaltenrechner = spaltenschritt;
            zeilenrechner = zeilenschritt;
        } else if (abschnittsweise) {
            spaltenrechner = spaltenabschnitt;
            zeilenrechner = zeilenabschnitt;
        } else {
            spaltenrechner = spaltenabtastung;
            zeilenrechner = zeilenabtastung;
        }
    }

    Linie spalte() {
        return spaltenrechner;
    }

    Linie zeile() {
        return zeilenrechner;
    }

}